/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "resources.h"
#include <QCoreApplication>
#include <QStringList>
#include <iostream>

void printUsage() {
    std::cout << "Usage:\n\tqyoutube-videos-list OPTION... " \
    "List YouTube videos matching optional criteria.\n\n" \
    "Options:\n\t-apiKey\t\t\tThe api key.\n\t" \
    "-accessToken\t\tThe access token.\n\t" \
    "-refreshToken\t\tThe refresh token.\n\t" \
    "-part\t\t\tA comma-separated list of properties to be included in the response.\n\t\t\t\t" \
    "The default is 'snippet'.\n\t" \
    "-chart\t\t\tThe chart to be retrieved.\n\t" \
    "-id\t\t\tA comma-separated list of video ids.\n\t" \
    "-myRating\t\tRating used to filter results.\n\t" \
    "-h1\t\t\tLanguage used to filter results.\n\t" \
    "-maxResults\t\tThe maximum number of results to be retrieved (default = 5, max = 50).\n\t" \
    "-onBehalfOfContentOwner\tIndicates that the request is made on behalf of the specified content owner.\n\t" \
    "-pageToken\t\tSpecifies a particular page in a result set.\n\t" \
    "-regionCode\t\tSelects the chart for the specified region.\n\t" \
    "-videoCategoryid\tSpecifies the video category for the chart.\n\t" \
    "-output\t\t\tThe output format (html, json, tsv (default), xml).";
}

QString getParam(QStringList &args, const QString &name, const QString &defaultValue = QString()) {
    int i = args.indexOf(name);

    if ((i >= 0) && (i < args.size() - 1)) {
        args.removeAt(i);
        return args.takeAt(i);
    }

    return defaultValue;
}

int main(int argc, char *argv[]) {
    QCoreApplication app(argc, argv);
    app.setOrganizationName("QYouTube");
    app.setApplicationName("cli");

    QStringList args = app.arguments();

    if (args.size() < 2) {
	printUsage();
	return 1;
    }

    args.removeFirst();

    const QString key = getParam(args, "-apiKey");
    
    if (key.isEmpty()) {
        printUsage();
        return 1;
    }

    const QString token = getParam(args, "-accessToken");
    const QString refresh = getParam(args, "-refreshToken");

    QString filter = getParam(args, "-chart");

    if (filter.isEmpty()) {
        filter = getParam(args, "-id");

        if (filter.isEmpty()) {
            filter = getParam(args, "-myRating");

            if ((filter.isEmpty()) || (token.isEmpty()) || (refresh.isEmpty())) {
                printUsage();
                return 1;
            }
        }
    }
    
    const QString contentOwner = getParam(args, "-onBehalfOfContentOwner");

    if ((!contentOwner.isEmpty()) && ((token.isEmpty()) || (refresh.isEmpty()))) {
        printUsage();
        return 1;
    }

    const QString h1 = getParam(args, "-h1");
    const QString maxResults = getParam(args, "-maxResults");
    const QString pageToken = getParam(args, "-pageToken");
    const QString regionCode = getParam(args, "-regionCode");
    const QString category = getParam(args, "-videoCategoryId");
    const QString output = getParam(args, "-output", "tsv");
    const QStringList part = getParam(args, "-part", "snippet").split(",");

    QVariantMap filters;
    filters["filter"] = filter;

    QVariantMap params;

    if (!contentOwner.isEmpty()) {
        params["onBehalfOfContentOwner"] = contentOwner;
    }

    if (!h1.isEmpty()) {
        params["h1"] = h1;
    }

    if (!maxResults.isEmpty()) {
        params["maxResults"] = maxResults;
    }

    if (!pageToken.isEmpty()) {
        params["pageToken"] = pageToken;
    }

    if (!regionCode.isEmpty()) {
        params["regionCode"] = regionCode;
    }

    if (!category.isEmpty()) {
        params["videoCategoryId"] = category;
    }
    
    QYouTube::Resources resources;
    resources.setApiKey(key);
    resources.setAccessToken(token);
    resources.setRefreshToken(refresh);
    resources.setOutputFormat(output);
    resources.list("/videos", part, filters, params);
    QObject::connect(&resources, SIGNAL(finished()), &app, SLOT(quit()));
    
    return app.exec();
}
