/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "SMS.h"

#include "Attachment.h"
#include "DBBackends/RtcomEventLogger.h"
#include "EventParsers/VMGEntities/VMessage.h"
#include "NumberToNameLookup.h"
#include "Settings.h"

#include <QRegExp>
#include <QString>
#include <QTextStream>

#include <rtcom-eventlogger/event.h>
#include <rtcom-eventlogger/eventlogger-attach-iter.h>

using namespace EventTypes;

const DBBackends::iDBBackend &SMS::DB() const
{
	return DBBackends::RtcomEventLogger(CurrentSettings(), *this);
}

SMS::SMS(const Settings &settings) :
	m_Settings(settings)
{
}

SMS::~SMS()
{
	foreach(QSharedPointer<Attachment> attachment, m_Attachments)
	{
		attachment.clear();
	}
}

SMS::SMS(const Settings &settings, const RTComElEvent &event, const QList<RTComElAttachment*> attachments) :
	m_Settings(settings)
{
	Version(2.1);
	IsRead(event.fld_is_read);
	Destination(event.fld_outgoing ? SENT : INBOX);
	Timestamp(QDateTime::fromTime_t(event.fld_start_time).toUTC());
	Tel(QString::fromUtf8(event.fld_remote_uid));
	if(Tel().indexOf("0") == 0)
		Tel(QString(Tel()).replace(QRegExp("^0"), "+44"));
	//Contents(QString(event.fld_free_text).replace("\n", QChar(0x2029)));
	Contents(QString::fromUtf8(event.fld_free_text));
	if(event.fld_flags != 0)
	{
		qDebug() << "fld_flags: = " << event.fld_flags;
	}
	Pending(event.fld_flags & GetFlagValue("RTCOM_EL_FLAG_SMS_PENDING"));

	// We directly access the m_Attachments member variable here rather than the
	// accessor as the accessor is const
	if(attachments.count() > 0)
		foreach(RTComElAttachment *attachment, attachments)
			m_Attachments.append(new Attachment(*attachment));
}
#include <QDebug>
const uint SMS::HashCode() const
{
//	qDebug() << Timestamp().toUTC().toTime_t() << ", " << qHash(Tel()) << ", " << qHash(Destination()) << ", " << qHash(Contents()) << ", " << Attachments().HashCode();

//	foreach(QChar c, Contents().toUtf8())
//	{
//		qDebug() << c.unicode();
//	}

		iHashable::Hash hash = Timestamp().toUTC().toTime_t() ^ qHash(Tel()) ^ qHash(Destination()) ^ qHash(Contents()) ^ Attachments().HashCode();
		qDebug() << hash << Timestamp().toUTC().toTime_t() << qHash(Tel()) << qHash(Destination()) << qHash(Contents()) << Attachments().HashCode();

	return
		Timestamp().toUTC().toTime_t() ^
		qHash(Tel()) ^
		qHash(Destination()) ^
		qHash(Contents()) ^
		Attachments().HashCode();
}

RTComElEvent * SMS::toRTComEvent(const NumberToNameLookup &numberToNameLookup) const
{
	QString groupId((Tel().length() < 7 || Tel().indexOf(QRegExp("[:alpha:]+")) > -1)
		? Tel()
		: Tel().right(7));

	RTComElEvent *event(rtcom_el_event_new());
	memset(event, 0, sizeof(RTComElEvent));

	RTCOM_EL_EVENT_SET_FIELD (event, service, g_strdup("RTCOM_EL_SERVICE_SMS"));
	RTCOM_EL_EVENT_SET_FIELD (event, event_type, g_strdup("RTCOM_EL_EVENTTYPE_SMS_MESSAGE"));
	RTCOM_EL_EVENT_SET_FIELD (event, storage_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, start_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, end_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, is_read, IsRead() ? 1 : 0);
	RTCOM_EL_EVENT_SET_FIELD (event, outgoing, Destination() == SMS::SENT ? 1 : 0);
	RTCOM_EL_EVENT_SET_FIELD (event, local_uid, g_strdup("ring/tel/ring"));
	//RTCOM_EL_EVENT_SET_FIELD (&event, local_name, g_strdup("<SelfHandle>"));
	RTCOM_EL_EVENT_SET_FIELD (event, remote_uid, g_strdup(Tel().toUtf8()));
	//RTCOM_EL_EVENT_SET_FIELD (&event, remote_name, g_strdup(QString::number(numberToNameLookup.ContactDetails().value(Tel()).second).toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, remote_ebook_uid, g_strdup(QString::number(numberToNameLookup.ContactDetails().value(Tel()).first).toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, group_uid, g_strdup(groupId.toUtf8()));
	//RTCOM_EL_EVENT_SET_FIELD (event, free_text, g_strdup(Contents().replace(0x2029, "\n").toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, free_text, g_strdup(Contents().toUtf8()));

	return event;
}

void SMS::WriteCSVSymbian(QTextStream &stream, const ColumnIndicesByIndexHash &headerIndices, const QChar delimiter, const NumberToNameLookup &numberToNameLookup, SymbianEventLogStrings &strings) const
{
	// 0|05/09/2007 11:25:12 am|1||||||<name>|3|8|0|0|<number>|||Unrecognized||
	for(uint columnIndex(0); columnIndex < (uint)headerIndices.count(); ++columnIndex)
	{
		const QString &heading(headerIndices.value(columnIndex));
		if("etype" == heading.toLower())
		{
			stream << "3"; // SMSes are type '3'
		}
		else if("etime" == heading.toLower())
		{
			stream << Timestamp().toUTC().toString("dd/MM/yyyy h:mm:ss ap");
		}
		else if("remote" == heading.toLower())
		{
			stream << numberToNameLookup.ContactDetails().value(Tel()).second;
		}
		else if("direction" == heading.toLower())
		{
			if(SMS::SENT == Destination())
			{
				if(!strings.contains("Outgoing"))
					strings.insert("Outgoing", strings.count());
				stream << strings.value("Outgoing");
			}
			else if (SMS::INBOX == Destination())
			{
				if(!strings.contains("Incoming"))
					strings.insert("Incoming", strings.count());
				stream << strings.value("Incoming");
			}
		}
		else if("duration" == heading.toLower())
		{
			stream << "0"; // SMSes are always 0
		}
		else if("dtype" == heading.toLower())
		{
			stream << "-1"; // -1 seems to match KLogDurationNone
		}
		else if("status" == heading.toLower())
		{
			stream << "0"; // Always '0' for phone calls.
		}
		else if("subject" == heading.toLower())
		{
			// Subject seems to be ignored - this makes sense, but I
			// suspect that non-zero values are a bug that I can't duplicate...
			stream << "0";
		}
		else if("number" == heading.toLower())
		{
			stream << Tel();
		}
		else if("data" == heading.toLower())
		{
			// Event-specfic data - but not supported by DBU-SCV tool
			// Hex editing the DBU suggests there is SIP account info in here...
			// ...along the lines of:
			// "VOIP.URL=sip:<remote#>@<remote domain>.MA=sip:<my#>@<my domain>"
			stream << "Unrecognised";
		}
		else
		{
			// Don't print anything. Makes it obvious which fields we've
			// generated.
		}

		if(columnIndex < (uint)headerIndices.count() - 1)
			stream << delimiter;
	}

	stream << endl;
}

const QString SMS::PathForVMG() const
{
	QString eventPath("/sms");
	eventPath += Destination() == EventTypes::SMS::SENT ? "/Sent/" : "/Inbox/";
	eventPath += QString::number(Timestamp().toUTC().date().year()) + "/";

	return eventPath;
}

void SMS::WriteVMG(QTextStream &stream, const NumberToNameLookup &numberToNameLookup) const
{
	EventParsers::VMGEntities::VMessage writer(CurrentSettings(), NULL, 1.1);
	writer.Write(stream, *this, numberToNameLookup);
	stream.flush();
}

QDebug operator<<(QDebug dbg, SMS& event)
{
	dbg.nospace() << "\tFolder:\t\t" << (event.Destination() == SMS::SENT ? "Sent" : "Inbox") << "\n";
	dbg.nospace() << "\tTimestamp:\t" << event.Timestamp().toUTC() << "\n";
	dbg.nospace() << "\tRemote-Tel:\t" << event.Tel() << "\n";
	//dbg.nospace() << "\tremote-name:\t" << event.fld_remote_name << "\n";
	dbg.nospace() << "\tIs-read:\t\t" << (event.IsRead() ? "true" : "false") << "\n";
	dbg.nospace() << "\tContents:\t" << event.Contents() << "\n";

	return dbg;
}
