/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "PhoneCall.h"

#include "Attachment.h"
#include "DBBackends/RtcomEventLogger.h"
#include "EventParsers/CSVSymbianEventLogParser.h"
#include "NumberToNameLookup.h"
#include "Settings.h"

#include <QDateTime>
#include <QDir>
#include <QFile>
#include <QList>
#include <QRegExp>
#include <QString>
#include <QTextStream>

#include <utime.h>

#include <rtcom-eventlogger/event.h>
#include <rtcom-eventlogger/eventlogger-attach-iter.h>

using namespace EventTypes;

const DBBackends::iDBBackend &PhoneCall::DB() const
{
	return DBBackends::RtcomEventLogger(CurrentSettings(), *this);
}

PhoneCall::PhoneCall(const Settings &settings) :
	m_Settings(settings)
{
	qDebug() << "Created new default Phone Call: " << *this;
}

PhoneCall::~PhoneCall()
{
	foreach(QSharedPointer<Attachment> attachment, m_Attachments)
	{
		attachment.clear();
	}
}

PhoneCall::PhoneCall(const Settings &settings, const RTComElEvent &event, const QList<RTComElAttachment*> attachments) :
	m_Settings(settings)
{
	Destination(event.fld_outgoing ? Settings::OUTGOING : Settings::INCOMING);
	Timestamp(QDateTime::fromTime_t(event.fld_start_time).toUTC());
	DurationInSeconds(QDateTime::fromTime_t(event.fld_start_time).toUTC().secsTo(QDateTime::fromTime_t(event.fld_end_time).toUTC()));
	Tel(event.fld_remote_uid);
	if(Tel().indexOf("0") == 0)
		Tel(QString(Tel()).replace(QRegExp("^0"), "+44"));

	if(attachments.count() > 0)
		foreach(RTComElAttachment *attachment, attachments)
			Attachments().append(new Attachment(*attachment));

	qDebug() << "Created new Phone Call from RtCom:\n" << *this;
}

PhoneCall::PhoneCall(const Settings &settings, const Settings::eDirection destination, const QDateTime &timestamp, const QString &tel, const int durationInSeconds, const bool isMissedCall, const AttachmentCollection &attachments) :
	m_Settings(settings), m_Destination(destination), m_Timestamp(timestamp), m_Tel(tel),
	m_DurationInSeconds(durationInSeconds), m_IsMissedCall(isMissedCall), m_Attachments(attachments)
{
	if(Tel().indexOf("0") == 0)
		Tel(QString(Tel()).replace(QRegExp("^0"), "+44"));

	// TODO: Copy attachments.
//	if(attachments.count() > 0)
//		foreach(const QSharedPointer<Attachment *> &attachment, attachments)
//			Attachments().append(attachment);

	qDebug() << "Created new Phone Call: " << *this;
}

#include <QDebug>
const uint PhoneCall::HashCode() const
{
//	qDebug() << Timestamp().toUTC().toTime_t() << ", " << qHash(Tel()) << ", " << qHash(Destination()) << ", " << qHash(Contents()) << ", " << Attachments().HashCode();

//	foreach(QChar c, Contents().toUtf8())
//	{
//		qDebug() << c.unicode();
//	}

	return
		Timestamp().toUTC().toTime_t() ^
		DurationInSeconds() ^
		qHash(Tel()) ^
		qHash(Destination()) ^
		Attachments().HashCode();
}

RTComElEvent * PhoneCall::toRTComEvent(const NumberToNameLookup &numberToNameLookup) const
{
	QList<QString> voiceMailList;
	voiceMailList << "+447945353070"; // T-Mobile UK Voicemail

	QString groupId((Tel().length() < 7 || Tel().indexOf(QRegExp("[:alpha:]+")) > -1)
		? Tel()
		: Tel().right(7));

	RTComElEvent *event(rtcom_el_event_new());
	memset(event, 0, sizeof(RTComElEvent));

	RTCOM_EL_EVENT_SET_FIELD (event, service, g_strdup("RTCOM_EL_SERVICE_CALL"));
	if(voiceMailList.contains(Tel()))
		RTCOM_EL_EVENT_SET_FIELD (event, event_type, g_strdup("RTCOM_EL_EVENTTYPE_CALL_VOICEMAIL"));
	else if(IsMissedCall())
		RTCOM_EL_EVENT_SET_FIELD (event, event_type, g_strdup("RTCOM_EL_EVENTTYPE_CALL_MISSED"));
	else
		RTCOM_EL_EVENT_SET_FIELD (event, event_type, g_strdup("RTCOM_EL_EVENTTYPE_CALL"));
	RTCOM_EL_EVENT_SET_FIELD (event, start_time, Timestamp().toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, end_time, Timestamp().addSecs(DurationInSeconds()).toUTC().toTime_t());
	RTCOM_EL_EVENT_SET_FIELD (event, storage_time, Timestamp().addSecs(DurationInSeconds()).toUTC().toTime_t());
	//RTCOM_EL_EVENT_SET_FIELD (event, is_read, 0);
	RTCOM_EL_EVENT_SET_FIELD (event, outgoing, Destination() == Settings::OUTGOING ? 1 : 0);
	//if(local_uid) // Voicemail local_uid transform needed here
	RTCOM_EL_EVENT_SET_FIELD (event, local_uid, g_strdup("ring/tel/ring"));
	//RTCOM_EL_EVENT_SET_FIELD (&event, local_name, g_strdup("<SelfHandle>"));
	RTCOM_EL_EVENT_SET_FIELD (event, remote_uid, g_strdup(Tel().toUtf8()));
	//RTCOM_EL_EVENT_SET_FIELD (&event, remote_name, g_strdup(QString::number(numberToNameLookup.ContactDetails().value(Tel()).second).toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, remote_ebook_uid, g_strdup(QString::number(numberToNameLookup.ContactDetails().value(Tel()).first).toUtf8()));
	RTCOM_EL_EVENT_SET_FIELD (event, group_uid, g_strdup(groupId.toUtf8()));

	return event;
}

void PhoneCall::WriteCSVSymbian(QTextStream &stream, const ColumnIndicesByIndexHash &headerIndices, const QChar delimiter, const NumberToNameLookup &numberToNameLookup, SymbianEventLogStrings &strings) const
{
	// 0|05/09/2007 11:25:12 am|1||||||<name>|3|8|0|0|<number>|||Unrecognized||
	for(uint columnIndex(0); columnIndex < (uint)headerIndices.count(); ++columnIndex)
	{
		const QString &heading(headerIndices.value(columnIndex));
		if("etype" == heading.toLower())
		{
			stream << "0"; // Phone calls are type '0'
		}
		else if("etime" == heading.toLower())
		{
			stream << Timestamp().toUTC().toString("dd/MM/yyyy h:mm:ss ap");
		}
		else if("remote" == heading.toLower())
		{
			stream << numberToNameLookup.ContactDetails().value(Tel()).second;
		}
		else if("direction" == heading.toLower())
		{
			if(IsMissedCall())
			{
				if(!strings.contains("Missed Call"))
					strings.insert("Missed Call", strings.count());
				stream << strings.value("Missed Call");
			}
			else if(Settings::OUTGOING == Destination())
			{
				if(!strings.contains("Outgoing"))
					strings.insert("Outgoing", strings.count());
				stream << strings.value("Outgoing");
			}
			else if (Settings::INCOMING == Destination())
			{
				if(!strings.contains("Incoming"))
					strings.insert("Incoming", strings.count());
				stream << strings.value("Incoming");
			}
		}
		else if("duration" == heading.toLower())
		{
			stream << DurationInSeconds();
		}
		else if("dtype" == heading.toLower())
		{
			stream << "1"; // 1 seems to be KLogDurationValid
		}
		else if("status" == heading.toLower())
		{
			stream << "0"; // Always '0' for phone calls.
		}
		else if("subject" == heading.toLower())
		{
			// Subject seems to be ignored - this makes sense, but I
			// suspect that non-zero values are a bug that I can't duplicate...
			stream << "0";
		}
		else if("number" == heading.toLower())
		{
			stream << Tel();
		}
		else if("data" == heading.toLower())
		{
			// Event-specfic data - but not supported by DBU-SCV tool
			// Hex editing the DBU suggests there is SIP account info in here...
			// ...along the lines of:
			// "VOIP.URL=sip:<remote#>@<remote domain>.MA=sip:<my#>@<my domain>"
			stream << "Unrecognised";
		}
		else
		{
			// Don't print anything. Makes it obvious which fields we've
			// generated.
		}

		if(columnIndex < (uint)headerIndices.count() - 1)
			stream << delimiter;
	}

	stream << endl;
}

QDebug operator<<(QDebug dbg, PhoneCall& event)
{
	dbg.nospace() << "\tHash:\t\t" << event.HashCode() << "\n";
	dbg.nospace() << "\tDirection:\t\t" << (event.Destination() == Settings::OUTGOING ? "Made" : "Recieved") << "\n";
	dbg.nospace() << "\tTimestamp:\t" << event.Timestamp().toUTC() << "\n";
	dbg.nospace() << "\tDuration:\t\t" << event.DurationInSeconds() << " seconds\n";
	dbg.nospace() << "\tRemote-Tel:\t" << event.Tel() << "\n";
	//dbg.nospace() << "\tremote-name:\t" << event.fld_remote_name << "\n";

	return dbg;
}
