/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "EventPreventer.h"

#include "CellularRadio.h"
#include "Settings.h"

#include <QDebug>

#include <QtCore/QCoreApplication>
#include <QtDBus/QtDBus>

const char * TP_ACCOUNT_PHONE("/org/freedesktop/Telepathy/Account/ring/tel/ring");
const char * TP_IFACE_ACCOUNT("org.freedesktop.Telepathy.Account");
const char * TP_IFACE_ACCOUNT_MANAGER("org.freedesktop.Telepathy.AccountManager");
const char * TP_PATH_ACCOUNT_MANAGER("/org/freedesktop/Telepathy/AccountManager");
const char * TP_IFACE_DBUS_PROPERTIES("org.freedesktop.DBus.Properties");

EventPreventer::EventPreventer(const Settings & settings) : m_Settings(settings)
{
	StoreAccountStates();
	DisableAccounts();
}

EventPreventer::~EventPreventer()
{
	RestoreAccounts();
}

void EventPreventer::StoreAccountStates()
{
	QDBusInterface tpAccountManager(TP_IFACE_ACCOUNT_MANAGER, TP_PATH_ACCOUNT_MANAGER, TP_IFACE_DBUS_PROPERTIES);
	if (tpAccountManager.isValid())
	{
		// Grab state
		QDBusMessage accountList = tpAccountManager.call("Get", TP_IFACE_ACCOUNT_MANAGER, "ValidAccounts");
		foreach (QVariant arg, accountList.arguments())
		{
			const QDBusArgument argument(qvariant_cast<QDBusVariant>(arg).variant().value<QDBusArgument>());
			argument.beginArray();
			while (!argument.atEnd())
			{
				// Grab the account path
				QDBusObjectPath accountPath;
				argument >> accountPath;

				if(accountPath.path() == TP_ACCOUNT_PHONE)
				{
					// ...as we're not allowed to disable the telephone account,
					// track the state of the cellular radio. Heavy-handed, but only known way.
					OriginalAccountStates().insert(accountPath.path(), CellularRadio::CurrentState());

				}
				else
				{
					// Grab the enabled property of each account
					QDBusInterface tpAccount(TP_IFACE_ACCOUNT_MANAGER, accountPath.path(), TP_IFACE_DBUS_PROPERTIES);
					if (tpAccount.isValid())
					{
						QDBusReply<QVariant> isEnabledProp = tpAccount.call("Get", TP_IFACE_ACCOUNT, "Enabled");

						bool isEnabled(isEnabledProp.value().toBool());
						qDebug() << (isEnabled ? "Enabled:\t\t" : "Not enabled:\t") << accountPath.path();
						OriginalAccountStates().insert(accountPath.path(), isEnabled);
					}
				}
			}
			argument.endArray();
		}
	}
}

void EventPreventer::DisableAccounts()
{
	SetAccountsToSameState(false);
}

void EventPreventer::EnableAccounts()
{
	SetAccountsToSameState(true);
}

void EventPreventer::SetAccountsToSameState(const bool state)
{
	// Set all stored accounts to requested state
	foreach (QString accountPath, OriginalAccountStates().keys())
		SetAccountToState(accountPath, state);
}

void EventPreventer::RestoreAccounts()
{
	// Set all stored accounts to stored state
	foreach (QString accountPath, OriginalAccountStates().keys())
		SetAccountToState(accountPath, OriginalAccountStates().value(accountPath));
}

void EventPreventer::SetAccountToState(const QString &accountPath, const bool desiredState)
{
	if(accountPath == TP_ACCOUNT_PHONE)
	{
		if(CurrentSettings().DisableCellular())
		{
			// ...as we're not allowed to disable the telephone account,
			// disable the whole cellular radio. Heavy-handed, but only known way.
			CellularRadio::SetState(desiredState);
		}
	}
	else
	{
		// Set each account to offline
		QDBusInterface tpAccountManager(TP_IFACE_ACCOUNT_MANAGER, accountPath, TP_IFACE_DBUS_PROPERTIES);
		if (tpAccountManager.isValid())
		{
			QDBusReply<void> isEnabledProp = tpAccountManager.call("Set", TP_IFACE_ACCOUNT, "Enabled",
				QVariant::fromValue(QDBusVariant(QVariant::fromValue(desiredState))));

			if(!isEnabledProp.isValid())
				qDebug() << isEnabledProp.error().message();
		}
		else
			qDebug() << tpAccountManager.lastError().message();
	}
}



