/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "CSV.h"
#include "CSVSymbianEventLogParser.h"
#include "EventTypes/PhoneCall.h"
#include "Settings.h"

#include <QDebug>

#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QString>
#include <QStringList>

#include <stdexcept>

using namespace EventParsers;
using EventTypes::PhoneCall;

class SortByValueDesc
{
public:
	inline bool operator()(const QPair<QChar, uint> &a, const QPair<QChar, uint> &b) const
	{
		return b.second < a.second;
	}
};

const QString ExtractString(const QString &originalString)
{
	QRegExp content("^[\"\']?(.*)?[\"\']?$");
	content.indexIn(originalString.trimmed());
	return content.cap(1);
}

iEventParser *CSVSymbianEventLogParser::IsValid(const Settings &currentSettings, QFile &eventFile)
{
	qDebug() << "Checking if a CSV call log file...";

	CSV eventsReader;
	eventsReader.Open(eventFile);

	// If we found all of the required headings, continue
	if(eventsReader.HasRequiredHeadings(QStringList() << "EType" << "ETime"<< "Remote" << "Direction" << "Status" << "Duration" << "Number" << "Data").count() == 0)
		return new EventParsers::CSVSymbianEventLogParser(currentSettings, eventFile.fileName()/*, delim, numColumnsPerRecord, headingPositions*/);
	else
		return NULL;
}

CSVSymbianEventLogParser::CSVSymbianEventLogParser(const Settings &settings, const QString &/*filename, const QChar delimiter, const int numColumnsPerRecord, const ColumnIndicesHash &headingIndices*/)
	: m_Settings(settings)/*, m_Delimiter(delimiter), m_NumColumnsPerRecord(numColumnsPerRecord), m_HeadingIndices(headingIndices)*/
{
}

EventTypes::EventFromFileList CSVSymbianEventLogParser::ParseFile(QFile &eventFile, const QList<uint> &recordsToReturn)
{
	EventTypes::EventFromFileList fileEvents;

	// Parse secondary table for string values
	{
		// Determine filename
		QFileInfo eventFileInfo(eventFile);
		QString filePath(eventFileInfo.absolutePath() + "/string.csv");

		QFile stringsFile(filePath);
		if(stringsFile.open(QFile::ReadOnly))
		{
			CSV stringsReader;
			stringsReader.Open(stringsFile);
			if(stringsReader.HasRequiredHeadings(QStringList() << "TEXT" << "Id").count() == 0)
			{
				while(!stringsReader.AtEnd())
				{
					QHash<QString, QString> values(stringsReader.ReadRecord());
					Strings().insert(values.value("text").toLower(), values.value("id").toUInt());
				}
			}
		}
		else
			throw std::runtime_error(QString("Unable to open: '%1'. The error was: %2").arg(filePath).arg(stringsFile.error()).toStdString());
	}

	QSet<uint> recordsToReturnSet(QSet<uint>::fromList(recordsToReturn));

	CSV eventsReader;
	eventsReader.Open(eventFile);
	if(eventsReader.HasRequiredHeadings(QStringList() << "EType" << "ETime"
										<< "Remote" << "Direction" << "Status"
										<< "Duration" << "Number" << "Data").count() == 0)
	{
		while(!eventsReader.AtEnd())
		{
			QHash<QString, QString> recordValues(eventsReader.ReadRecord());

			if(recordsToReturnSet.count() == 0 || recordsToReturnSet.contains(eventsReader.CurrentRecordNumber()))
			{
				bool bOK(false);
				int eType(recordValues.value("etype").toUInt(&bOK));
				// We're only interested in phone calls
				if(bOK && eType == 0)
				{
					qDebug() << "Parsing event from record #" << eventsReader.CurrentRecordNumber() << ". Values: " << recordValues;

					QDateTime eTime(QDateTime::fromString(recordValues.value("etime"), "dd/MM/yyyy h:mm:ss ap"));
					Settings::eDirection direction(ReadDirection(recordValues.value("direction").toUInt()));

					// We only care about the requested directions...
					if(CurrentSettings().ShouldProcess(direction, EventTypes::EVENT_TYPE_CALL))
					{
						int duration(recordValues.value("duration").toInt(&bOK));
						if(!bOK)
						{
							qDebug() << QString("Unable to parse '%1' as a duration. Skipping record.")
											.arg(recordValues.value("duration"));
							continue;
						}

						QString number(ExtractString(recordValues.value("number")));

						bool isMissedCall(ExtractString(recordValues.value("direction")).toUInt() == Strings().value("missed call"));

						uint test = ExtractString(recordValues.value("direction")).toUInt();
						if(test != Strings().value("incoming") && test != Strings().value("outgoing"))
							qDebug() << eventFile.fileName() << ", line #" << eventsReader.CurrentRecordNumber() << "Direction? : " << test;

						// TODO: Not currently used...but almost certainly contains SIP call data
						QString data(ExtractString(recordValues.value("data")));

						if(number.trimmed().length() == 0)
							qDebug() << "Empty tel!";

						QSharedPointer<EventTypes::iEvent> newEvent(new PhoneCall(
							CurrentSettings(),
							direction,
							eTime,
							number,
							duration,
							isMissedCall));
						fileEvents.append(EventTypes::EventFromFile(newEvent, eventsReader.CurrentRecordNumber()));
					}
				}
			}
		}
	}

	qDebug() << QString("File pos: %1, bAvail: %2, canReadLine: %3").arg(eventFile.pos()).arg(eventFile.bytesAvailable()).arg(eventFile.canReadLine());
	qDebug() << fileEvents.count() << " events loaded from file";
	return fileEvents;
}

Settings::eDirection CSVSymbianEventLogParser::ReadDirection(const uint value)
{
	// Missed calls are
	if(value == Strings().value("incoming") || value == Strings().value("missed call"))
		return Settings::INCOMING;
	else if(value == Strings().value("outgoing"))
		return Settings::OUTGOING;
	else
	{
		// Determine which string this value corresponds to...
		QString unexpectedString("");
		foreach(QString string, Strings().keys())
			if(Strings().value(string) == value)
				unexpectedString = string;

		throw std::runtime_error(
				QString("Unexpected value found: \"%1\" (maps to \"%2\")")
				.arg(value)
					.arg(unexpectedString != "" ? unexpectedString : "<no known string>")
				.toStdString());
	}
}
