/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#ifndef CSV_H
#define CSV_H

#include <QHash>
class QChar;
class QFile;
class QString;
#include <QSharedPointer>
#include <QStringList>
class QTextStream;

class CSV
{
public:
	typedef QHash<QString, uint> ColumnIndicesHash;
	typedef QHash<uint, QString> ColumnNamesHash;

	CSV();
	CSV(QChar delimiter, int numColumnsPerRecord, const ColumnIndicesHash &headingIndices);
	~CSV();

	void Open(QFile &file);
	void Open(QFile &file, QChar delimiter, int numColumnsPerRecord, const ColumnIndicesHash &headingIndices);
	bool AtEnd() const;
	void Close();
	QHash<QString, QString> ReadRecord();
	const QStringList HasRequiredHeadings(const QStringList &requiredHeadings);

protected:
	void GetHeadings(const QString &firstLine);
	void DetermineDelimiter(const QString &firstLine);

private:
	const QString ExtractString(const QString &originalString);
	void UpdateHeadings(const ColumnIndicesHash &headingIndices);

public:
	const unsigned int CurrentRecordNumber() const { return m_RecordNumber; }
	unsigned int CurrentRecordNumber() { return m_RecordNumber; }

protected:
	const bool IsValid() { return m_IsValid; }
	const QChar Delimiter() const { return m_Delimiter; }
	const int NumColumnsPerRecord() const { return m_NumColumnsPerRecord; }
	const ColumnIndicesHash & HeadingIndices() const { return m_HeadingIndices; }
	const ColumnNamesHash & HeadingNames() const { return m_HeadingNames; }
	QFile * const File() { return m_File; }
	const QSharedPointer<QTextStream> Stream() const { return m_Stream; }
	const unsigned int &LineNumber() const { return m_LineNumber; }
	unsigned int &LineNumber() { return m_LineNumber; }
	const QStringList &LineValues() const { return m_LineValues; }
	const unsigned int & RecordNumber() const { return m_RecordNumber; }
	unsigned int & RecordNumber() { return m_RecordNumber; }

protected:
	ColumnIndicesHash & HeadingIndices() { return m_HeadingIndices; }
	ColumnNamesHash & HeadingNames() { return m_HeadingNames; }
	QStringList & LineValues() { return m_LineValues; }

protected:
	void IsValid(const bool isValid) { m_IsValid = isValid; }
	void Delimiter(const QChar delimiter) { m_Delimiter = delimiter; }
	void NumColumnsPerRecord(const int numColumnsPerRecord) { m_NumColumnsPerRecord = numColumnsPerRecord; }
	void HeadingIndices(const ColumnIndicesHash headingIndices) { m_HeadingIndices = headingIndices; }
	void HeadingNames(const ColumnNamesHash headingNames) { m_HeadingNames = headingNames; }
	void File(QFile * const file) { m_File = file; }
	void Stream(QTextStream * const stream) { m_Stream = QSharedPointer<QTextStream>(stream); }
	void LineNumber(unsigned int lineNumber) { m_LineNumber = lineNumber; }
	void RecordNumber(unsigned int recordNumber) { m_RecordNumber = recordNumber; }
	void LineValues(QStringList &lineValues) { m_LineValues = lineValues; }

private:
	bool m_IsValid;
	QChar m_Delimiter;
	int m_NumColumnsPerRecord;
	ColumnIndicesHash m_HeadingIndices;
	ColumnNamesHash m_HeadingNames;
	QFile *m_File;
	QSharedPointer<QTextStream> m_Stream;
	unsigned int m_LineNumber;
	unsigned int m_RecordNumber;
	QStringList m_LineValues;
};

#endif // CSV_H
