/*
 * Copyright (C) 2011, Jamie Thompson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "RtcomEventLogger.h"

#include "EventProcessors/iEventProcessor.h"
#include "EventTypes/iEvent.h"
#include "EventTypes/SMS.h"
#include "Settings.h"

#include <QDebug>

#include <uuid/uuid.h>

#include <rtcom-eventlogger/event.h>
#include <rtcom-eventlogger/eventlogger.h>

using namespace DBBackends;

QDebug operator<<(QDebug, RTComElEvent &);
QDebug operator<<(QDebug, RTComElAttachment &);
QDebug operator<<(QDebug, GList &);
QDebug operator<<(QDebug, QList<RTComElAttachment*> &);

RtcomEventLogger::RtcomEventLogger(const Settings &settings) :
	m_Settings(settings)
{
	RTComEl *el(rtcom_el_new());
	if(NULL != el)
	{
		// Grab the service IDs we want to work with
		m_ServiceIDs.insert(SERVICE_ID_CALL, rtcom_el_get_service_id(el, "RTCOM_EL_SERVICE_CALL"));
		m_ServiceIDs.insert(SERVICE_ID_CHAT, rtcom_el_get_service_id(el, "RTCOM_EL_SERVICE_CHAT"));
		m_ServiceIDs.insert(SERVICE_ID_SMS, rtcom_el_get_service_id(el, "RTCOM_EL_SERVICE_SMS"));
		m_ServiceIDs.insert(SERVICE_ID_MMS, rtcom_el_get_service_id(el, "RTCOM_EL_SERVICE_MMS"));

		// Remove any service IDs that weren't found
		foreach(ServiceID serviceID, m_ServiceIDs.keys())
			if(m_ServiceIDs.value(serviceID) == -1)
				m_ServiceIDs.remove(serviceID);

		g_object_unref(el);
	}
	else
		qDebug() << "Failed to create event logger.";
}

RtcomEventLogger::RtcomEventLogger(const Settings &settings, const EventTypes::RtcomEvent &event) :
	m_Settings(settings)
{
}

void RtcomEventLogger::Process(EventProcessors::iEventProcessor &processor)
{
	// Initialise the event logger
	RTComEl *el = rtcom_el_new();
	if(NULL != el)
	{
		// Initialise a query
		RTComElQuery *query = rtcom_el_query_new(el);
		if(query != NULL)
		{
			bool incoming = CurrentSettings().ShouldProcess( Settings::TYPE_RECIEVED, Settings::EVENTTYPE_SMS);
			bool outgoing = CurrentSettings().ShouldProcess( Settings::TYPE_SENT, Settings::EVENTTYPE_SMS);

			// Prepare it...
			bool prepared = false;
			if(incoming && outgoing)
			{
				prepared = rtcom_el_query_prepare(query,
					"service-id",
					m_ServiceIDs.value(SERVICE_ID_SMS),
					RTCOM_EL_OP_EQUAL,

					NULL);
			}
			else
			{
				prepared = rtcom_el_query_prepare(query,
					"service-id",
					m_ServiceIDs.value(SERVICE_ID_SMS),
					RTCOM_EL_OP_EQUAL,

					"outgoing",
					incoming ? 0 : 1,
					RTCOM_EL_OP_EQUAL,

					NULL);
			}

			qDebug() << "SQL:\n" << rtcom_el_query_get_sql(query);

			if(prepared)
			{
				RTComElIter *it = rtcom_el_get_events(el, query);
				if(it != NULL)
				{
					if(rtcom_el_iter_first(it))
					{
						int eventCount = 0;
						qDebug() << "Getting event count...";
						while(rtcom_el_iter_next(it))
							++eventCount;

						// Reset the iterator and grab the actual values
						qDebug() << "Resetting iterator...";
						g_object_unref(it);
						it = rtcom_el_get_events(el, query);
						if(it != NULL)
						{
							if(rtcom_el_iter_first(it))
							{
								int idx = 0;
								qDebug() << "Getting events...";
								do
								{
									++idx;
									qDebug() << "Event #" << idx;

									RTComElEvent revent;
									memset(&revent, 0, sizeof(revent));

									if(rtcom_el_iter_get_values (
										it,
										"id", &revent.fld_id,
										"service-id", &revent.fld_service_id,
										"start-time", &revent.fld_start_time,
										"end-time", &revent.fld_end_time,
										"local-uid", &revent.fld_local_uid,
										"local-name", &revent.fld_local_name,
										"remote-uid", &revent.fld_remote_uid,
										"remote-name", &revent.fld_remote_name,
										"is-read", &revent.fld_is_read,
										"outgoing", &revent.fld_outgoing,
										"free-text", &revent.fld_free_text,
										NULL))
									{
										qDebug() << revent;

										QList<RTComElAttachment *> rattachments;
										RTComElAttachIter *at_it = rtcom_el_iter_get_attachments(it);
										if(it != NULL)
										{
											qDebug() << "Attachments OK";
											if(rtcom_el_attach_iter_first(at_it))
											{
												qDebug() << "Getting events...";

												do
												{
													rattachments.append(rtcom_el_attach_iter_get(at_it));
													qDebug() << "Attachment ID #" << rattachments.last()->id << endl;
													qDebug() << "desc: " << rattachments.last()->desc << endl;
													qDebug() << "path: " << rattachments.last()->path << endl;
												}while(rtcom_el_attach_iter_next(at_it));
											}
										}

										EventTypes::iEvent *const newEvent(CreateEvent(revent, rattachments));
										processor.Process(*newEvent);
										delete newEvent;

										processor.EmitEventProcessed(idx, eventCount);
									}

									rtcom_el_event_free_contents(&revent);
								}
								while(rtcom_el_iter_next(it));
								qDebug() << "...all events retrieved.";
							}
						}
						else
							qDebug() << "Failed to reset iterator";
					}
					else
						qDebug() << "Failed to start iterator";
				}
				else
					qDebug() << "Failed to get iterator. Do you have any events?";
			}
			else
				qDebug() << "Failed to prepare the query.";

			g_object_unref(query);
		}
		else
			qDebug() << "Failed to create query.";

		g_object_unref(el);
	}
	else
		qDebug() << "Failed to create event logger.";
}

EventTypes::iEvent *const RtcomEventLogger::CreateEvent(RTComElEvent &revent, QList<RTComElAttachment*> &rattachments)
{
	//if(m_ServiceIDs.contains(SERVICE_ID_CALL) && revent.fld_service_id == m_ServiceIDs.value(SERVICE_ID_CALL))
	//	return new EventTypes::Call(CurrentSettings(), revent, rattachments);

	//if(m_ServiceIDs.contains(SERVICE_ID_CHAT) && revent.fld_service_id == m_ServiceIDs.value(SERVICE_ID_CHAT))
	//	return new EventTypes::Chat(CurrentSettings(), revent, rattachments);

	if(m_ServiceIDs.contains(SERVICE_ID_SMS) && revent.fld_service_id == m_ServiceIDs.value(SERVICE_ID_SMS))
		return new EventTypes::SMS(CurrentSettings(), revent, rattachments);

	//if(m_ServiceIDs.contains(SERVICE_ID_MMS) && revent.fld_service_id == m_ServiceIDs.value(SERVICE_ID_MMS))
	//	return new EventTypes::MMS(CurrentSettings(), revent, rattachments);

	return NULL;
}

void RtcomEventLogger::Insert(EventTypes::iEvent &event, const NumberToNameLookup &numberToNameLookup)
{
	if(EventTypes::RtcomEvent *rtcomEvent = dynamic_cast<EventTypes::RtcomEvent *>(&event))
	{
		const uint UUID_STR_LEN(36);

		int newEventID(0);
		_RTComEl *el(rtcom_el_new());
		if(NULL != el)
		{
			// Convert our objects into RTCom structs
			RTComElEvent *revent(rtcomEvent->toRTComEvent(numberToNameLookup));
			GList *rattachments(event.Attachments().toRTComAttachments());

			GError *error(NULL);

			// Generate the headers for the event
			GHashTable *rheaders(g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free));
			uuid_t uuid;
			char key[UUID_STR_LEN + 1];
			uuid_generate_random(uuid);
			uuid_unparse(uuid, key);
			g_hash_table_insert(rheaders, g_strdup ("message-token"), key);
			qDebug() << "headers: " << rheaders;

			qDebug() << "Inserting event:";
			qDebug() << *revent;
			qDebug() << *rattachments;

			// Add the event
			newEventID = rtcom_el_add_event_full(el, revent, rheaders, rattachments, &error);
			qDebug() << "new id: " << newEventID;
			if(error != NULL)
			{
				qDebug() << "err: " << error->message;
				g_error_free(error);
			}

			// Release the attachments
			g_list_foreach (rattachments, (GFunc) rtcom_el_free_attachment, NULL);
			g_list_free (rattachments);

			rtcom_el_event_free_contents(revent);
			rtcom_el_event_free(revent);
		}
		else
			qDebug() << "Unable to initalise eventlogger for insertion.";

		g_object_unref(el);
	}

	return;
}

QDebug operator<<(QDebug dbg, RTComElEvent &event)
{
	dbg.nospace() << "\tid:\t\t" << event.fld_id << "\n";
	dbg.nospace() << "\tFolder:\t\t" << (event.fld_outgoing ? "Sent" : "Inbox") << "\n";
	dbg.nospace() << "\tstart-time:\t" << QDateTime::fromTime_t(event.fld_start_time) << "\n";
	dbg.nospace() << "\tend-time:\t\t" << QDateTime::fromTime_t(event.fld_end_time) << "\n";
	//dbg.nospace() << "\tlocal-uid:\t" << event.fld_local_uid << "\n";
	//dbg.nospace() << "\tlocal-name:\t" << event.fld_local_name << "\n";
	dbg.nospace() << "\tremote-uid:\t" << event.fld_remote_uid << "\n";
	dbg.nospace() << "\tremote-name:\t" << event.fld_remote_name << "\n";
	dbg.nospace() << "\tis-read:\t\t" << (event.fld_is_read ? "true" : "false") << "\n";
	dbg.nospace() << "\tfree-text:\t" << event.fld_free_text << "\n";
	dbg.nospace() << "\tgroup-uid:\t" << event.fld_group_uid << "\n";

	return dbg;
}

QDebug operator<<(QDebug dbg, RTComElAttachment &attachment)
{
	dbg.nospace() << "Event-id:\t" << attachment.event_id << "\n";
	dbg.nospace() << "Path:\t" << attachment.path << "\n";
	dbg.nospace() << "Desc:\t" << attachment.desc << "\n";

	return dbg;
}

QDebug operator<<(QDebug dbg, GList &attachments)
{
	dbg.nospace() << "Attachments" << "\n";

	for (GList *attachment(&attachments); NULL != attachment; attachment = attachment->next)
	{
		qDebug() << *(RTComElAttachment*)attachment->data;
	}

	dbg.nospace() << "\n";

	return dbg;
}

QDebug operator<<(QDebug dbg, QList<RTComElAttachment *> &attachments)
{
	dbg.nospace() << "Attachments" << "\n";

	foreach(RTComElAttachment *attachment, attachments)
		dbg.nospace() << *attachment << "\n";

	dbg.nospace() << "\n";

	return dbg;
}
