/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "contentlist.h"
#include <QPointer>

#include <ImtkScrollArea>

// ContentListItem

ContentListItem::ContentListItem(QGraphicsItem *parent)
    : QGraphicsObject(parent),
      m_state(NormalState),
      m_geometry(QRectF(0.0, 0.0, 0.0, 0.0)),
      m_geometryReady(false)
{
    setFlag(QGraphicsItem::ItemHasNoContents, true);
    setFlag(QGraphicsItem::ItemSendsGeometryChanges, true);
}

QRectF ContentListItem::boundingRect() const
{
    if (!m_geometryReady)
        return QRectF(0.0, 0.0, 0.0, contentHeight());
    return m_geometry;
}

void ContentListItem::updateGeometry()
{
    m_geometryReady = parentItem() != 0;
    qreal width = parentItem() ? parentItem()->boundingRect().width() - pos().x() : 0.0;
    QRectF geometry(0.0, 0.0, width, contentHeight());
    if (m_geometry != geometry) {
        prepareGeometryChange();
        m_geometry = geometry;
        update();
    }
}

ContentListItem::State ContentListItem::state() const
{
    return m_state;
}

void ContentListItem::setState(State state)
{
    if (m_state != state) {
        m_state = state;
        stateChanged(state);
    }
}

QVariant ContentListItem::itemChange(GraphicsItemChange change, const QVariant &value)
{
    Q_UNUSED(value);
    switch (change) {
    case QGraphicsItem::ItemParentHasChanged:
    case QGraphicsItem::ItemPositionHasChanged:
        updateGeometry();
        break;
    default:
        break;
    }
    return value;
}

void ContentListItem::paint(QPainter *painter,
                            const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(painter);
    Q_UNUSED(option);
    Q_UNUSED(widget);
}


// ContentList

ContentList::ContentList(QGraphicsItem *parent)
    : QGraphicsWidget(parent),
      m_frozen(false),
      m_lastIndex(-1),
      m_itemPressed(false),
      m_currentItem(-1)
{
    setFocusPolicy(Qt::StrongFocus);
    setFlag(QGraphicsItem::ItemIsFocusable);
    setFlag(QGraphicsItem::ItemHasNoContents, true);
}

ContentList::ContentList(QList<ContentListItem*> items, QGraphicsItem *parent)
    : QGraphicsWidget(parent),
      m_frozen(false),
      m_lastIndex(-1),
      m_itemPressed(false),
      m_currentItem(-1)
{
    setFocusPolicy(Qt::StrongFocus);
    setFlag(QGraphicsItem::ItemIsFocusable);
    setFlag(QGraphicsItem::ItemHasNoContents, true);
    appendItems(items);
}

bool ContentList::addItem(ContentListItem *item)
{
    return insertItem(m_items.count(), item);
}

bool ContentList::insertItem(int index, ContentListItem* item)
{
    if (m_frozen || index < 0)
        return false;

    m_lastIndex = index;
    QAbstractAnimation *animation = insertAnimation(index, item->contentHeight());

    m_items.insert(index, item);
    resize(size().width(), size().height() + item->contentHeight());

    if (!animation)
        doInsertItem(true);
    else {
        m_frozen = true;
        connect(animation, SIGNAL(finished()), SLOT(doInsertItem()));
        animation->start(QAbstractAnimation::DeleteWhenStopped);
    }

    return true;
}

void ContentList::doInsertItem(bool reposition)
{
    int index = m_lastIndex;

    if (index < 0 || index >= m_items.count())
        return;

    int finalY;

    if (index > 0)
        finalY = m_items[index - 1]->y() +
            m_items[index - 1]->boundingRect().height();
    else
        finalY = 0;

    ContentListItem *item = m_items[index];
    item->setParentItem(this);
    item->setPos(0, finalY);

    if (reposition) {
        for (int i = index + 1; i < m_items.count(); i++)
            m_items[index]->setY(m_items[index]->y() + item->contentHeight());
    }

    m_frozen = false;
}

bool ContentList::removeItem(int index)
{
    if (m_frozen || (index < 0 || index >= m_items.count()))
        return false;

    m_lastIndex = index;
    QAbstractAnimation *animation = removeAnimation(index);

    if (!animation)
        doRemoveItem(true);
    else {
        m_frozen = true;
        connect(animation, SIGNAL(finished()), SLOT(doRemoveItem()));
        animation->start(QAbstractAnimation::DeleteWhenStopped);
    }

    return true;
}

bool ContentList::removeItem(ContentListItem *item)
{
    return removeItem(indexFromItem(item));
}

void ContentList::doRemoveItem(bool reposition)
{
    int index = m_lastIndex;

    if (index < 0 || index >= m_items.count())
        return;

    ContentListItem *item = m_items[index];
    item->setParentItem(0);
    m_items.removeAt(index);

    resize(size().width(), size().height() - item->contentHeight());

    if (reposition) {
        for (int i = index + 1; i < m_items.count(); i++)
            m_items[index]->setY(m_items[index]->y() - item->contentHeight());
    }

    delete item;
    m_frozen = false;

    if (index == m_currentItem)
        m_currentItem = -1;
}

void ContentList::updateItems()
{
    qreal top = 0;
    foreach(ContentListItem *item, m_items) {
        item->setPos(0, top);
        top += item->contentHeight();
    }
    update();
}

void ContentList::appendItems(QList<ContentListItem*> items)
{
    qreal top = 0;
    foreach(ContentListItem *item, m_items)
        top += item->contentHeight();
    foreach(ContentListItem *item, items) {
        item->setParentItem(this);
        item->setPos(0, top);
        top += item->contentHeight();
        m_items.append(item);
        resize(size().width(), size().height() + item->contentHeight());
    }
}

int ContentList::itemCount() const
{
    return m_items.count();
}

void ContentList::resizeEvent(QGraphicsSceneResizeEvent *e)
{
    QGraphicsWidget::resizeEvent(e);
    foreach(ContentListItem *item, m_items)
        item->updateGeometry();
}

QAbstractAnimation *ContentList::insertAnimation(int, qreal)
{
    return 0;
}

QAbstractAnimation *ContentList::removeAnimation(int)
{
    return 0;
}

void ContentList::keyPressEvent(QKeyEvent *event)
{
    if (m_items.isEmpty()) {
        QGraphicsWidget::keyPressEvent(event);
        return;
    }

    if (m_itemPressed)
        return;

    if (event->key() == Qt::Key_Return
        || event->key() == Qt::Key_Enter
        || event->key() == Qt::Key_Select) {
        if (!event->isAutoRepeat() && m_currentItem >= 0) {
            m_itemPressed = true;
            setItemState(m_currentItem, ContentListItem::PressedState);
            return;
        }
    } else {
        int current = m_currentItem;

        if (current < 0)
            current = 0;
        else if (event->key() == Qt::Key_Up && current > 0)
            current--;
        else if (event->key() == Qt::Key_Down && current < itemCount() - 1)
            current++;

        if (m_currentItem != current) {
            setItemState(m_currentItem, ContentListItem::NormalState);
            setItemState(current, ContentListItem::SelectedState);
            m_currentItem = current;
            return;
        }
    }

    QGraphicsWidget::keyPressEvent(event);
}

void ContentList::keyReleaseEvent(QKeyEvent *event)
{
    if (m_itemPressed && (event->key() == Qt::Key_Return
                          || event->key() == Qt::Key_Enter
                          || event->key() == Qt::Key_Select)) {
        if (!event->isAutoRepeat()) {
            setItemState(m_currentItem, ContentListItem::SelectedState);
            onItemActivated(m_currentItem);
            emit itemActivated(m_currentItem);
            m_itemPressed = false;
            return;
        }
    }

    QGraphicsWidget::keyReleaseEvent(event);
}

bool ContentList::setItemState(int index, ContentListItem::State state)
{
    if (index < 0 || index >= m_items.count())
        return false;

    ContentListItem *item = m_items[index];
    item->setState(state);

    // Try to fit to screen if it's embedded in a scrollarea
    // This is not the best approach
    if (state == ContentListItem::SelectedState) {
        ImtkScrollArea *scroll = qobject_cast<ImtkScrollArea *>(parentWidget());
        if (scroll) {
            if (item->y() < scroll->offset())
                scroll->setOffset(item->y());
            else {
                qreal ix = item->y() + item->boundingRect().height();
                qreal sx = scroll->offset() + scroll->size().height();
                if (ix > sx)
                    scroll->setOffset(scroll->offset() + ix - sx);
            }
        }
    }

    return true;
}

void ContentList::doActivateItem(ContentListItem *item)
{
    emit itemActivated(indexFromItem(item));
}
