#include "optionslist.h"
#include "settings.h"
#include "pixmaploader.h"
#include "forecastmodel.h"

#include <cmath>
#include <QPainter>
#include <QPropertyAnimation>
#include <QParallelAnimationGroup>

#define MAX_VISIBLE_ITEMS           3
#define ITEM_BACKGROUND             (PixmapLoader::pixmap("options_bg"))
#define SELECTED_ITEM_BACKGROUND    (PixmapLoader::pixmap("options_focus"))
#define PRESSED_ITEM_BACKGROUND    (PixmapLoader::pixmap("options_pressed"))


OptionsListItem::OptionsListItem(QAction *action, OptionsContentList *list)
    : ContentListItem(0)
    , m_height(ITEM_BACKGROUND.height())
    , m_list(list)
    , m_background(new QGraphicsPixmapItem(ITEM_BACKGROUND, this))
    , m_action(action)
{
    QFont f("Nokia Sans");
    f.setPixelSize(Settings::intValue("option-item-font-size"));

    m_text = new ImtkLabel(this);
    m_text->setFont(f);
    m_text->setText(action->text());
    m_text->setPos(Settings::scaleHeight(15), Settings::scaleHeight(19));

    m_text->setMinimumWidth(m_background->boundingRect().width());
    m_text->setMaximumWidth(m_background->boundingRect().width());

    connect(action, SIGNAL(changed()), SLOT(onUpdate()));
}

void OptionsListItem::onUpdate()
{
    m_text->setText(m_action->text());
}

void OptionsListItem::select(bool selected)
{
    m_background->setPixmap(selected ? SELECTED_ITEM_BACKGROUND : ITEM_BACKGROUND);
}

qreal OptionsListItem::contentHeight() const
{
    return m_height;
}

void OptionsListItem::doActivate()
{
    if (m_action)
        m_action->activate(QAction::Trigger);
}

void OptionsListItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

void OptionsListItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
    doActivate();
}

void OptionsListItem::stateChanged(ContentListItem::State state)
{
    switch(state) {
    case ContentListItem::NormalState:
        m_background->setPixmap(ITEM_BACKGROUND);
        break;
    case ContentListItem::PressedState:
        m_background->setPixmap(PRESSED_ITEM_BACKGROUND);
        break;
    case ContentListItem::SelectedState:
        m_background->setPixmap(SELECTED_ITEM_BACKGROUND);
        break;
    }
}

// OptionsContentList

OptionsContentList::OptionsContentList()
    : ContentList(0)
{

}

void OptionsContentList::setOptions(const QList<QAction *> &options)
{
    while (itemCount() > 0)
        removeItem((int)0);

    QList<ContentListItem *> items;

    foreach (QAction *value, options)
        items.append(new OptionsListItem(value, this));

    appendItems(items);
}

void OptionsContentList::select(const QString &selected)
{
    for (int i = 0; i < itemCount(); ++i) {
        OptionsListItem *item = qobject_cast<OptionsListItem*>(itemAt(i));
        if (item)
            item->select(item->text() == selected);
    }
}

void OptionsContentList::onItemActivated(int index)
{
    OptionsListItem *item = qobject_cast<OptionsListItem *>(itemAt(index));
    if (item)
        item->doActivate();
}


// OptionsList

OptionsList::OptionsList(QGraphicsItem *parent)
    : QGraphicsWidget(parent),
      m_topPixmap(PixmapLoader::pixmap("options_shadow")),
      m_itemHeight(ITEM_BACKGROUND.height()),
      m_list(new OptionsContentList())
{
    int w = Settings::windowSize().width();

    resize(w, MAX_VISIBLE_ITEMS * m_itemHeight);

    m_scrollBox = new ImtkScrollArea(this);
    m_scrollBox->setWidget(m_list);
    m_scrollBox->setGeometry(0, m_topPixmap.height(), w, MAX_VISIBLE_ITEMS * m_itemHeight);
}

void OptionsList::setOptions(const QList<QAction *> &options)
{
    m_list->setOptions(options);
}

OptionsContentList *OptionsList::contents() const
{
    return m_list;
}

qreal OptionsList::desiredHeight() const
{
    return qMin(MAX_VISIBLE_ITEMS, m_list->itemCount()) * m_itemHeight
        + m_topPixmap.height();
}

int OptionsList::loadImages()
{
    PixmapLoader::load("scroll");
    PixmapLoader::load("scroll_knob");
    PixmapLoader::load("list_top");
    PixmapLoader::load("list_item_bg");
    PixmapLoader::load("list_item_selected_bg");
    PixmapLoader::load("button_list_delete");
    PixmapLoader::load("list_check");
    return 5;
}

void OptionsList::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

void OptionsList::paint(QPainter *painter, const QStyleOptionGraphicsItem *, QWidget *)
{
    painter->drawPixmap(0, 0, m_topPixmap);
}
