/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef __CARROUSSEL_H__
#define __CARROUSSEL_H__

#include <QList>
#include <QDebug>


template<class T> class Carroussel
{
public:
    Carroussel() : m_pos(0) {}
    ~Carroussel() {}
    int count() const { return m_list.count(); }
    int pos() const { return m_pos; }
    void move(int offset) { m_pos = getIndex(m_pos + offset); }
    int add(const T &item);
    void remove(int idx);

    void clear() { m_list.clear(); }

    T &at(int idx) { return m_list[getIndex(idx + m_pos)]; }

    T &operator[](int aIdx) { return m_list[getIndex(aIdx + m_pos)]; }
    const T &operator[](int aIdx) const { return m_list[getIndex(aIdx + m_pos)]; }
    void reset(const QList<T> &list, int newPos);
private:
    int m_pos;
    QList<T> m_list;
    int getIndex(int idx) const;
    int getCarrousselIndex(int idx);
};

class CarrousselGroup
{
public:
    inline ~CarrousselGroup();
    template<class T> void add(T *carroussel);
    inline void remove(void *carroussel);
    inline void move(int offset);

private:
    class BasicCarrosselHandler
    {
    public:
        virtual ~BasicCarrosselHandler() {}
        virtual void move(int offset) = 0;
        virtual bool isEqual(void *value) = 0;
    };
    template<class T> class CarrousselHandler : public BasicCarrosselHandler
    {
    public:
        CarrousselHandler(T *carroussel) : m_carroussel(carroussel) {}
        void move(int offset) { m_carroussel->move(offset); }
        bool isEqual(void *value) { return m_carroussel == value; }
    private:
        T *m_carroussel;
    };

    QList<BasicCarrosselHandler*> m_items;
    inline int find(void *value);
};

// Carroussel

template<class T> int Carroussel<T>::add(const T &item)
{
    int idx = m_list.count() - m_pos;
    m_list.append(item);
    return idx <= m_list.count() / 2 ? idx : idx - m_list.count();
}

template<class T> void Carroussel<T>::remove(int idx)
{
    idx = getIndex(m_pos + idx);
    m_list.removeAt(idx);
    m_pos = getIndex(idx < m_pos ? m_pos - 1 : m_pos);
}

template<class T> void Carroussel<T>::reset(const QList<T> &list, int newPos)
{
    m_list = list;
    m_pos = getIndex(newPos);
}

template<class T> int Carroussel<T>::getIndex(int idx) const
{
    if (m_list.count() == 0)
        return 0;
    idx %= m_list.count();
    return idx >= 0 ? idx : m_list.count() + idx;
}

// CarrousselGroup

CarrousselGroup::~CarrousselGroup()
{
    qDeleteAll(m_items);
}

template<class T> void CarrousselGroup::add(T *carroussel)
{
    if (find(carroussel) == -1)
        m_items.append(new CarrousselHandler<T>(carroussel));
}

void CarrousselGroup::remove(void *carroussel)
{
    int idx = find(carroussel);
    if (idx >= 0)
        delete m_items.takeAt(idx);
}

int CarrousselGroup::find(void *value)
{
    for (int i = 0; i < m_items.count(); ++i)
        if (m_items[i]->isEqual(value))
            return i;
    return -1;
}

void CarrousselGroup::move(int offset)
{
    foreach (BasicCarrosselHandler *handler, m_items)
        handler->move(offset);
}




#endif /* __CARROUSSEL_H__ */
