/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QGraphicsScene>
#include <QCoreApplication>
#include <QGraphicsSceneMouseEvent>

#include "widget_p.h"
#include "flickablearea.h"
#include "flickablearea_p.h"


ImtkFlickableAreaPrivate::ImtkFlickableAreaPrivate(ImtkFlickableArea *qptr)
    : ImtkWidgetPrivate(qptr),
      hthreshold(50),
      vthreshold(50),
      isPressed(false),
      isSliding(false),
      isFlickable(true)
{

}

void ImtkFlickableAreaPrivate::sendClick(const QPointF &pos)
{
    Q_Q(ImtkFlickableArea);

    if (!q->scene())
        return;

    QGraphicsSceneMouseEvent *event;

    event = new QGraphicsSceneMouseEvent(QEvent::GraphicsSceneMousePress);
    event->setButton(Qt::LeftButton);
    event->setScenePos(pos);
    ignoreList << event;
    QCoreApplication::postEvent(q->scene(), event);

    event = new QGraphicsSceneMouseEvent(QEvent::GraphicsSceneMouseRelease);
    event->setButton(Qt::LeftButton);
    event->setScenePos(pos);
    ignoreList << event;
    QCoreApplication::postEvent(q->scene(), event);
}


ImtkFlickableArea::ImtkFlickableArea(QGraphicsItem *parent)
    : ImtkWidget(*new ImtkFlickableAreaPrivate(this), parent)
{

}

ImtkFlickableArea::ImtkFlickableArea(ImtkFlickableAreaPrivate &dptr,
                                     QGraphicsItem *parent)
    : ImtkWidget(dptr, parent)
{

}

ImtkFlickableArea::~ImtkFlickableArea()
{

}

int ImtkFlickableArea::threshold(Qt::Orientation orientation) const
{
    Q_D(const ImtkFlickableArea);
    return (orientation == Qt::Vertical) ? d->vthreshold : d->hthreshold;
}

void ImtkFlickableArea::setThreshold(Qt::Orientation orientation, int value)
{
    Q_D(ImtkFlickableArea);

    if (orientation == Qt::Vertical)
        d->vthreshold = value;
    else
        d->hthreshold = value;
}

bool ImtkFlickableArea::isFlickable() const
{
    Q_D(const ImtkFlickableArea);
    return d->isFlickable;
}

void ImtkFlickableArea::setFlickable(bool enabled)
{
    Q_D(ImtkFlickableArea);

    if (d->isFlickable != enabled) {
        d->isFlickable = enabled;

        if (!enabled) {
            d->isPressed = false;
            d->isSliding = false;
        }
    }
}

QRectF ImtkFlickableArea::flickableRect() const
{
    Q_D(const ImtkFlickableArea);
    return d->flickableRect;
}

void ImtkFlickableArea::setFlickableRect(const QRectF &rect)
{
    Q_D(ImtkFlickableArea);
    d->flickableRect = rect;
}

QVariant ImtkFlickableArea::itemChange(GraphicsItemChange change, const QVariant &value)
{
    if (change == QGraphicsItem::ItemSceneChange) {
        QGraphicsScene *oldScene = scene();

        if (oldScene)
            oldScene->removeEventFilter(this);

        if (value.canConvert<QGraphicsScene *>()) {
            QGraphicsScene *newScene = value.value<QGraphicsScene *>();

            if (newScene)
                newScene->installEventFilter(this);
        }
    }

    return QGraphicsWidget::itemChange(change, value);
}

bool ImtkFlickableArea::eventFilter(QObject *object, QEvent *event)
{
    Q_D(ImtkFlickableArea);

    Q_UNUSED(object);

    if (!d->isFlickable)
        return false;

    const QEvent::Type type = event->type();

    if (type != QEvent::GraphicsSceneMouseMove &&
        type != QEvent::GraphicsSceneMousePress &&
        type != QEvent::GraphicsSceneMouseRelease)
        return false;

    if (d->ignoreList.contains(event)) {
        d->ignoreList.removeOne(event);
        return false;
    }

    QRectF realRect;
    if (d->flickableRect.isEmpty())
        realRect = mapToScene(boundingRect()).boundingRect();
    else
        realRect = mapToScene(d->flickableRect).boundingRect();

    QGraphicsSceneMouseEvent *e = static_cast<QGraphicsSceneMouseEvent *>(event);

    const QPointF &scenePos = e->scenePos();

    bool consumed = false;

    switch(type) {
    case QEvent::GraphicsSceneMousePress:
        if (realRect.contains(scenePos)) {
            consumed = true;
            d->isPressed = true;
            d->clickPos = scenePos;
        }
        break;
    case QEvent::GraphicsSceneMouseMove:
        if (d->isPressed) {
            if (d->isSliding) {
                consumed = true;
                mouseSlideMoved(scenePos);
            } else if (qAbs(scenePos.x() - d->clickPos.x()) >= d->hthreshold ||
                       qAbs(scenePos.y() - d->clickPos.y()) >= d->vthreshold) {
                consumed = true;
                d->isSliding = true;
                mouseSlideStarted(scenePos);
            }
        }
        break;
    case QEvent::GraphicsSceneMouseRelease:
        if (d->isSliding) {
            consumed = true;
            mouseSlideFinished(scenePos);
            d->isSliding = false;
        } else if (d->isPressed) {
            consumed = true;
            d->sendClick(scenePos);
        }
        d->isPressed = false;
        break;
    default:
        break;
    }

    return consumed;
}
