/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "utils.h"
#include "weatherresponse.h"
#include "forecastresolver.h"

#include <QUrl>
#include <QDomDocument>
#include <QNetworkRequest>
#include <QNetworkReply>


ForecastResolver::ForecastResolver(QObject *parent)
    : QObject(parent),
      m_locationResolver(0)
{
    connect(&m_network, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(receiveResponse(QNetworkReply*)));
    connect(&m_network2, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(forecastReceiveResponse(QNetworkReply*)));
}

ForecastResolver::~ForecastResolver()
{
    if (m_locationResolver)
        delete m_locationResolver;
}

ForecastResolver::ForecastResolver(LocationResolver *resolver, QObject *parent)
    : QObject(parent),
      m_locationResolver(resolver)
{
    connect(&m_network, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(receiveResponse(QNetworkReply*)));


    connect(m_locationResolver, SIGNAL(locationResolved(int, QString)),
            this, SLOT(resolve(int, QString)));

    /// xxx
    connect(m_locationResolver, SIGNAL(locationQueryError(int, QString)),
            this, SLOT(resolve(int, QString)));

    connect(&m_network2, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(forecastReceiveResponse(QNetworkReply*)));
}

int ForecastResolver::resolve(const QString &locId)
{
    if (m_requests.contains(locId))
        return m_requests[locId][0];

    int result = generateSequentialId();

    doAddRequest(result, locId);
    return result;
}

int ForecastResolver::resolveByQuery(const QString &query)
{
    if (m_locationResolver)
        return m_locationResolver->resolve(query);

    return 0;
}

void ForecastResolver::resolve(int reqId, const QString &locId)
{
    if (m_requests.contains(locId)) {
        m_requests[locId].append(reqId);
        return;
    }
    doAddRequest(reqId, locId);
}

void ForecastResolver::doAddRequest(int reqId, const QString &locId)
{
    const QUrl &url = formatQuery(locId);
    m_network.get(QNetworkRequest(url));

    QList<int> list;
    list.append(reqId);
    m_requests[locId] = list;
}

void ForecastResolver::receiveResponse(QNetworkReply *reply)
{
    const QString locId = reply->request().url().queryItemValue(locationParam());

    if (locId.isNull() || !m_requests.contains(locId)) {
        qWarning() << "Unexpected forecast query response. (locId = " << locId << ")";
        return;
    }

    const QList<int> list = m_requests[locId];

    if (reply->error() == QNetworkReply::NoError) {
        WeatherResponse *result = parseResponse(locId, reply);

        if (result) {
            if (!isDual()) {
                m_requests.remove(locId);
                foreach (int reqId, list)
                    emit forecastResolved(reqId, result);
                return;
            } else {
                m_responses[locId] = result;
                resolveForecast(locId);
                return;
            }
        }
    }

    m_requests.remove(locId);
    foreach (int reqId, list)
        emit forecastResponseError(reqId);
}

void ForecastResolver::resolveForecast(const QString &locId)
{
    const QUrl &url = formatQuery(locId, true);
    m_network2.get(QNetworkRequest(url));
}

void ForecastResolver::forecastReceiveResponse(QNetworkReply *reply)
{
    const QString &locId = reply->request().url().queryItemValue(locationParam());

    if (locId.isNull() || !m_requests.contains(locId) || !m_responses.contains(locId)) {
        qWarning() << "Unexpected forecast query response. (locId = " << locId << ")";
        return;
    }

    WeatherResponse *result = m_responses[locId];

    if (reply->error() == QNetworkReply::NoError) {
        if (result)
            parseForecastResponse(result, reply);
    }

    const QList<int> list = m_requests[locId];
    m_requests.remove(locId);

    foreach (int reqId, list)
        emit forecastResolved(reqId, result);
}
