/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QTimer>
#include <QPixmap>
#include <QPropertyAnimation>

#include "scrollbar.h"
#include "pixmapwidget_p.h"


class ImtkScrollBarPrivate : public ImtkPixmapWidgetPrivate
{
public:
    ImtkScrollBarPrivate(ImtkScrollBar *qptr);
    void updateKnob();

    int value;
    int pageSize;
    int maximum;
    QTimer fadeTimer;
    bool hasFadeEffect;
    QPropertyAnimation *fadeAnimation;

    ImtkPixmapWidget *knob;

    void fadeIn();
    void _q_fadeTimeout();

private:
    IMTK_DECLARE_PUBLIC(ImtkScrollBar);
};


ImtkScrollBarPrivate::ImtkScrollBarPrivate(ImtkScrollBar *qptr)
    : ImtkPixmapWidgetPrivate(qptr),
      value(0), pageSize(10), maximum(100),
      hasFadeEffect(false), fadeAnimation(0)
{
    fadeTimer.setInterval(1000);
    fadeTimer.setSingleShot(true);
}

void ImtkScrollBarPrivate::fadeIn()
{
    Q_Q(ImtkScrollBar);
    fadeAnimation->stop();
    q->setOpacity(1.0);

    if (hasFadeEffect)
        fadeTimer.start();
    else
        fadeTimer.stop();
}

void ImtkScrollBarPrivate::_q_fadeTimeout()
{
    fadeAnimation->setDuration(400);
    fadeAnimation->setEndValue(0.0);
    fadeAnimation->start();
}

void ImtkScrollBarPrivate::updateKnob()
{
    Q_Q(ImtkScrollBar);

    qreal th = q->size().height();
    qreal fh = ((qreal)pageSize / qMax(pageSize + maximum, 1)) * th;

    int tb, bb;
    knob->getBorders(0, &tb, 0, &bb);
    knob->resize(knob->preferredWidth(), qBound<qreal>(tb + bb, fh, th));

    qreal max = qMax<qreal>(0, th - knob->size().height());
    knob->setY((value * max) / qMax(1, maximum));
}


ImtkScrollBar::ImtkScrollBar(const QPixmap &background,
                             const QPixmap &foreground,
                             QGraphicsItem *parent)
    : ImtkPixmapWidget(*new ImtkScrollBarPrivate(this), background, parent)
{
    setCacheMode(QGraphicsItem::ItemCoordinateCache);
    setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);

    Q_D(ImtkScrollBar);
    d->fadeAnimation = new QPropertyAnimation(this, "opacity", this);

    d->knob = new ImtkPixmapWidget(foreground, this);
    d->knob->setCacheMode(QGraphicsItem::ItemCoordinateCache);
    d->knob->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);

    d->knob->setPos(0, 0);
    d->updateKnob();

    connect(&d->fadeTimer, SIGNAL(timeout()), this, SLOT(_q_fadeTimeout()));
}

int ImtkScrollBar::value() const
{
    Q_D(const ImtkScrollBar);
    return d->value;
}

void ImtkScrollBar::setValue(int value)
{
    Q_D(ImtkScrollBar);
    if (d->value != value) {
        d->value = value;
        d->updateKnob();

        if (d->hasFadeEffect)
            d->fadeIn();
    }
}

int ImtkScrollBar::maximum() const
{
    Q_D(const ImtkScrollBar);
    return d->maximum;
}

void ImtkScrollBar::setMaximum(int maximum)
{
    Q_D(ImtkScrollBar);
    if (d->maximum != maximum) {
        d->maximum = maximum;
        d->updateKnob();

        if (d->hasFadeEffect)
            d->fadeIn();
    }
}

int ImtkScrollBar::pageSize() const
{
    Q_D(const ImtkScrollBar);
    return d->pageSize;
}

void ImtkScrollBar::setPageSize(int pageSize)
{
    Q_D(ImtkScrollBar);
    if (d->pageSize != pageSize) {
        d->pageSize = pageSize;
        d->updateKnob();
    }
}

void ImtkScrollBar::setKnobBorders(int left, int top, int right, int bottom)
{
    Q_D(ImtkScrollBar);
    d->knob->setBorders(left, top, right, bottom);
}

void ImtkScrollBar::setFadeEffect(bool enabled)
{
    Q_D(ImtkScrollBar);
    if (d->hasFadeEffect != enabled) {
        d->hasFadeEffect = enabled;
        d->fadeIn();
    }
}

void ImtkScrollBar::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    ImtkPixmapWidget::resizeEvent(event);
    Q_D(ImtkScrollBar);
    d->updateKnob();
}

#include "moc_scrollbar.cpp"
