//    QtLockscreen
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "lsnotification.h"
#include <QUuid>
#include "lockscreen.h"

LsNotification::LsNotification(QString name) :
        LsWidget(name.isEmpty() ? "Notification": name)
{
    applySettings();

#ifdef Q_WS_MAEMO_5
    QString dbName = QString("/home/user/.config/hildon-desktop/notifications.db");
    QString connName = QUuid::createUuid();
    m_Db = QSqlDatabase::addDatabase("QSQLITE", connName);
    m_Db.setConnectOptions("QSQLITE_OPEN_READONLY");
    m_Db.setDatabaseName(dbName);
    if (!m_Db.open()){
        qWarning() << QString("WARNING: error opening notifications database");
    }
#endif
}

void LsNotification::paint(QPainter *painter, const QStyleOptionGraphicsItem*,
                           QWidget*)
{
    QRectF rect(boundingRect());
    if (!m_Icon.isNull()){
        painter->drawPixmap(size().width() / 2 - m_Icon.width() / 2, 0, m_Icon);
        rect.setTop(rect.top() + m_Icon.height() + MINIMUM_TEXT_MARGIN);
    }

    painter->setFont(m_Font);
    painter->setPen(m_Color);
    painter->drawText(rect, Qt::AlignTop | Qt::AlignCenter | Qt::TextSingleLine, m_Text);
}

void LsNotification::applySettings()
{
    setSizeAndPosition(QSize(100, 100), QPoint(0, 0));

    QString defName;
    QString type = getSetting("Type", "call").toString();
    if (type == "call"){
        m_Type = NotificationMissedCall;
        defName = ":/images/call.png";
    }else if (type == "sms"){
        m_Type = NotificationNewSMS;
        defName = ":/images/message.png";
    }else if (type == "email"){
        m_Type = NotificationNewEmail;
        defName = ":/images/email.png";
    }

    m_Icon = QPixmap(getSetting("Icon", defName).toString());

    m_Color = getColorSetting("Color", QColor(255,255,255));
    m_Font = getFontSetting("Font", QFont("", 14, QFont::Normal, false));
}

void LsNotification::stopUpdate()
{

}

void LsNotification::startUpdate()
{
    updateNotification();
}

void LsNotification::updateNotification()
{
    int sms = 0;
    int email = 0;
    int call = 0;

#ifdef Q_WS_MAEMO_5
    if (m_Db.isOpen()){
        QSqlQuery qry(m_Db);
        QString sql = "SELECT H.value, H2.value, COUNT(*)   FROM notifications N, hints H, hints \
                       H2 WHERE N.id=H.nid AND H.id='category' and H2.id = 'time' and H2.nid = H.nid \
                       GROUP BY  H.value ORDER BY H2.value;";
        if (!qry.prepare(sql)){
            qWarning() << "Error preparing query:" << qry.lastError().text();
        }else{
            if (!qry.exec()){
                qWarning() << "Error executing query:" << qry.lastError().text();
            }else{
                QSqlRecord rec;
                while(qry.next()){
                    rec = qry.record();
                    QString name = rec.value(0).toString();
                    qDebug() << name << rec.value(2).toInt();

                    if (name == "email-message")
                        email = rec.value(2).toInt();
                    else if (name == "sms-message")
                        sms = rec.value(2).toInt();
                    else if (name == "missed-call")
                        call = rec.value(2).toInt();
                }
            }
        }
    }

    if (LockScreen::Debug){
        sms = 1;
        email = 1;
        call = 1;
    }
#else
    sms = 1;
    email = 1;
    call = 1;
#endif

    qDebug() << "Emails:" << email;
    qDebug() << "Calls:" << call;
    qDebug() << "SMS:" << sms;

    int value = 0;
    switch (m_Type){
    case NotificationMissedCall:
        value = call;
        m_Text = QObject::tr("%1 missed calls").arg(value);
        break;
    case NotificationNewSMS:
        value = sms;
        m_Text = QObject::tr("%1 unread messages").arg(value);
        break;
    case NotificationNewEmail:
        value = email;
        m_Text = QObject::tr("%1 new emails").arg(value);
        break;
    default:
        break;
    }

    if (value == 0){
        hide();
    }else{
        show();
    }
    update();
}
