#include "qtirreco.h"
#include "ui_qtirreco.h"

QtIrreco::QtIrreco(QWidget *parent)
    : QMainWindow(parent), m_ui(new Ui::QtIrrecoClass)
{
    m_ui->setupUi(this);
    this->initMenu();
     //Creates empty remotes.conf file
    QFile file(remotesConf);
    if (!file.open(QIODevice::ReadWrite | QIODevice::Text)) {
        QMessageBox::warning(this,"Error with remotes file",
                             "The program was unable to make remotes.conf file",Qt::NoButton,Qt::NoButton);
        return;
    }
    file.close();

    remoteNames = new QStringList();
    lw = new LayoutWidget(this);

    lw->show();
    lw->setFocus();

    connect(this,SIGNAL(remoteNameToLw(QString)),lw,SLOT(setActiveRemoteName(QString)));
    connect(this,SIGNAL(bgImageToLw(QString)),lw,SLOT(setBackgroundImagePath(QString)));
    connect(this,SIGNAL(rgbToLw(int,int,int)),lw,SLOT(setRgbInteger(int,int,int)));
    connect(this,SIGNAL(remoteDeleted()),lw,SLOT(clearScreen()));

    useModeEnabled = true;
    
    QUrl url;
    url.setHost("mercury.wipsl.com");
    url.setPath("/qtirreco/webdb/");
    url.setPort(80);

    client = new QxtXmlRpcClient(this, url);
    connect(client,SIGNAL(finish(int,QVariant)),this,SLOT(processReturnValue(int,QVariant)));
    connect(client,SIGNAL(networkError(int,QString)),this,SLOT(error(int,QString)));
    connect(client,SIGNAL(fault(int,int,QString)),this,SLOT(fault(int,int,QString)));

    this->checkForRemotes();
    if(remoteNames->count() != 0) {
       this->setWindowTitle("QtIrreco:"+activeRemote);
       this->setActiveRemote();
    }

    if(remoteNames->count() <= 0)
       this->noRemotes();
}

QtIrreco::~QtIrreco()
{
    // Stop lirc server
    system("sudo /etc/init.d/lirc stop");
    delete m_ui;

}
void QtIrreco::changeOldRemoteButtonPaths(QString fileName) // CAN BE DELETED LATER WHEN PROLLY ALL USERS HAVE
                                                               //   TRIED OLD REMOTES SO THEY ARE FIXED
{
    QFile file(remotesPath + fileName);
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream in(&file);
    QString data = in.readAll();

    if(data.contains(("/home/user/.qtirreco/themes/default"),Qt::CaseInsensitive))
        data.replace("/home/user/.qtirreco/themes/default","/home/opt/qtirreco/defaulttheme");

    if(data.contains(("/home/user/MyDocs/qtirreco/.themes"),Qt::CaseInsensitive))
        data.replace("/home/user/MyDocs/qtirreco/.themes","/home/opt/qtirreco/themes");

    file.remove();

    QFile newFile(remotesPath + fileName);
    newFile.open(QIODevice::WriteOnly | QIODevice::Text);
    newFile.write(data.toLatin1(),qstrlen(data.toLatin1()));

    newFile.close();
}

void QtIrreco::setActiveRemote()
{
    // If remote has old paths change them to new.
    changeOldRemoteButtonPaths(activeRemote+".conf");
    // Can be deleted later.


     lw->setActiveRemoteName(activeRemote);
     backgroundName.clear();
     QFile file(remotesPath + activeRemote+".conf");
        if (!file.open(QIODevice::ReadWrite | QIODevice::Text))
            return;
        QTextStream in(&file);
        while (!in.atEnd()) {
            QString line = in.readLine();
            this->readRemoteFile(line, in.atEnd());
        }
    file.close();

    // THESE LINES CAN BE DELETED LATER! ! !!! ! ! !
    this->editRemote();
    this->saveRemote();
}

void QtIrreco::checkForRemotes()
{
    QDir dir;

    QFile file(remotesConf);

    if (!file.open(QIODevice::ReadWrite | QIODevice::Text))
        return;

    remoteNames->clear();
    QTextStream in(&file);
    while (!in.atEnd()) {
        QString line = in.readLine();
        process_line(line);
    }
    file.close();
}

void QtIrreco::noRemotes()
{
    activeRemote.clear();
    this->setWindowTitle("Please make a new remote!");
    // Remove all files in Remotes
    QDir dir(remotesPath);
    QStringList files = dir.entryList(QDir::Files);
    for(int i=0;i<files.count(); i++) {
        QFile file(remotesPath + files.at(i));
        file.remove();
    }

    // Creates empty remotes.conf file

     QFile file(remotesConf);

     if (!file.open(QIODevice::ReadWrite | QIODevice::Text))
         return;
     file.close();

     m_ui->actionEditRemote->setDisabled(true);
     m_ui->actionSelectRemote->setDisabled(true);
     m_ui->actionDeleteRemote->setDisabled(true);
     m_ui->actionUploadDb->setDisabled(true);

     QColor defaultBgColor(50,50,50);
     palette.setColor(QPalette::Window,defaultBgColor);
     this->setPalette(palette);
     this->setAutoFillBackground(true);

     palette.setBrush(QPalette::Window, QPixmap(qtirrecoImage).scaled(800,424,Qt::IgnoreAspectRatio,Qt::FastTransformation));
     lw->setPalette(palette);
     lw->setAutoFillBackground(true);
}

void QtIrreco::newRemote() // Using RenameRemoteDlg class
{
    RenameRemoteDlg *rrDlg = new RenameRemoteDlg(this,activeRemote);
    rrDlg->setWindowTitle("Create a new remote");
    rrDlg->setNameLabelText("Please enter remote name: ");
    rrDlg->setPushButtonText("Create remote");
    rrDlg->setLabelGeometry(260,10,360,31);
    connect(rrDlg,SIGNAL(newRemote(QString)),this,SLOT(createRemote(QString)));
    rrDlg->exec();
    delete rrDlg;
}

void QtIrreco::createRemote(QString newRemoteName)
{
    bool alreadyExists = false;
    QString newRemoteFileName = (newRemoteName+".conf");
    for(int i=0;i<remoteNames->count();i++) {
        if(remoteNames->at(i) == newRemoteFileName) {
            QMessageBox::warning(this,"Remote name already exists!","Please choose a different remote name.",
                                 QMessageBox::NoButton,QMessageBox::NoButton);
            alreadyExists = true;
        }
    }
    if(!alreadyExists) {
        QDir dir;
        QFile newRemoteFile(remotesPath + newRemoteName+".conf");
        if (!newRemoteFile.open(QIODevice::ReadWrite | QIODevice::Text))
            return;
        lw->clearScreen();
        backgroundMode = defaultBackground;
        lw->setBackgroundMode(backgroundMode);

        lw->setBackgroundImagePath(qtirrecoImage);

        QColor defaultBgColor(50,50,50);
        palette.setColor(QPalette::Window,defaultBgColor);
        this->setPalette(palette);
        this->setAutoFillBackground(true);

        palette.setBrush(QPalette::Window, QPixmap(qtirrecoImage).scaled(800,424,Qt::IgnoreAspectRatio,Qt::FastTransformation));

        lw->setPalette(palette);
        lw->setAutoFillBackground(true);

        remoteNames->append(newRemoteName+".conf");

        m_ui->actionEditRemote->setEnabled(true);
        m_ui->actionDeleteRemote->setEnabled(true);
        m_ui->actionSelectRemote->setEnabled(true);
        m_ui->actionUploadDb->setEnabled(true);

        activeRemote = newRemoteName;
        lw->setActiveRemoteName(activeRemote);
        writeRemoteChanges();
        lw->setUseModeEnabled(false);
        useModeEnabled = false;
        this->setWindowTitle("QtIrreco:"+activeRemote);
        newRemoteFile.close();
        lw->saveRemoteToFile();
        }
}

void QtIrreco::editRemote()
{
    this->enableEditMenu();
    useModeEnabled = false;
    lw->setUseModeEnabled(useModeEnabled);
    lw->setFocus();

    // emits background properties to layout in case no background changes are made so saving file works.
    lw->setRgbInteger(r,g,b);
    lw->setBackgroundImagePath(backgroundName);
    lw->setBackgroundMode(backgroundMode);
}

void QtIrreco::renameRemote()
{
    RenameRemoteDlg *rrDlg = new RenameRemoteDlg(this,activeRemote);
    connect(rrDlg,SIGNAL(changeName(QString)),this,SLOT(changeNewRemoteName(QString)));
    rrDlg->setLabelGeometry(230,10,360,31);
    rrDlg->exec();
    delete rrDlg;
}

void QtIrreco::changeNewRemoteName(QString newName)
{
    bool alreadyExists = false;
    for(int i=0;i<remoteNames->count();i++) {
        if(remoteNames->at(i).contains(newName))
            alreadyExists = true;

    }
    if(alreadyExists) {
            QMessageBox::warning(this,"Remote name already exists!","Please choose different remote name.",
                         QMessageBox::NoButton,QMessageBox::NoButton);
            RenameRemoteDlg *rrDlg = new RenameRemoteDlg(this,activeRemote);
            connect(rrDlg,SIGNAL(changeName(QString)),this,SLOT(changeNewRemoteName(QString)));
            rrDlg->setLabelGeometry(230,10,360,31);
            rrDlg->exec();
            delete rrDlg;
    }
    else {
        oldRemoteName = activeRemote;
        QFile remoteFile(remotesPath + oldRemoteName + ".conf");
        remoteFile.rename(newName+".conf");
        activeRemote = newName;
        this->setWindowTitle("QtIrreco:"+activeRemote);
        lw->setActiveRemoteName(activeRemote);
        for(int i=0;i<remoteNames->count();i++) {
            if(remoteNames->at(i).contains(oldRemoteName))
                remoteNames->replace(i,activeRemote+".conf");

        }
        this->saveRemote();
        useModeEnabled = false;
        lw->setUseModeEnabled(useModeEnabled);
        this->editRemote();
        // Make changes to remotes.conf
        writeRemoteChanges();

    }
}

void QtIrreco::writeRemoteChanges() // when renaming remote or changing active remote or deleting remote
{
    QFile exConf(remotesConf);
    exConf.remove();
    QFile newConf(remotesConf);
          if (!newConf.open(QIODevice::ReadWrite | QIODevice::Text))
          return;
          QTextStream in(&newConf);
          if(remoteNames->count() > 0)
              in << activeRemote+(".conf")+ (" | current remote\n");
          for (int i=0;i<remoteNames->count();i++)
              in << remoteNames->at(i) + (" | filename of remote\n");

          newConf.close();
}

void QtIrreco::downloadRemote()
{
    DownloadRemoteDlg *dloadRmDlg = new DownloadRemoteDlg(this,true,remoteNames);
    connect(dloadRmDlg,SIGNAL(lwClearScreen()),lw,SLOT(clearScreen()));
    connect(dloadRmDlg,SIGNAL(addButtonToLayoutSignalFromRSD(QString,QString, QString, QString, QStringList,int)),this,SLOT(addButtonToLayout(QString,QString, QString, QString, QStringList,int)));
    connect(dloadRmDlg,SIGNAL(remoteNameToLayoutWidget(QString)),lw,SLOT(setActiveRemoteName(QString)));
    connect(dloadRmDlg,SIGNAL(remoteNameToLayoutWidget(QString)),this,SLOT(addRemoteName(QString)));
    dloadRmDlg->exec();
    delete dloadRmDlg;
}

void QtIrreco::addRemoteName(QString newRemoteName)
{
    remoteNames->append(newRemoteName+".conf");
    activeRemote = newRemoteName;

    m_ui->actionEditRemote->setEnabled(true);
    m_ui->actionSelectRemote->setEnabled(true);
    m_ui->actionDeleteRemote->setEnabled(true);
    m_ui->actionUploadDb->setEnabled(true);

    lw->setFocus();
    lw->setBackgroundImagePath(qtirrecoImage);
    backgroundMode = defaultBackground;
    lw->setBackgroundMode(backgroundMode);

    QColor defaultBgColor(50,50,50);
    palette.setColor(QPalette::Window,defaultBgColor);
    this->setPalette(palette);
    this->setAutoFillBackground(true);

    palette.setBrush(QPalette::Window, QPixmap(qtirrecoImage).scaled(800,424,Qt::IgnoreAspectRatio,Qt::FastTransformation));
    lw->setPalette(palette);
    lw->setAutoFillBackground(true);

    this->setActiveRemote();
    this->setWindowTitle("QtIrreco:"+activeRemote);

    writeRemoteChanges();
    this->saveRemote();
}

void QtIrreco::addButtonToLayout(QString fileNameUnPressed, QString fileNamePressed, QString confFileName, QString buttonText, QStringList commands, int number)
{
    lw->addButton(fileNameUnPressed, fileNamePressed, confFileName, buttonText, commands, number);
}
void QtIrreco::addButtonToLayout(QString fileNameUnPressed, QString fileNamePressed, QString confFileName, QString buttonText, QStringList commands, int number,int timeOut)
{
    Q_UNUSED(number);
    lw->addButton(fileNameUnPressed, fileNamePressed, confFileName, buttonText, commands, QPoint(0,0), timeOut);
}

void QtIrreco::selectRemote()
{
    ChangeRemoteDlg *crDlg = new ChangeRemoteDlg(this, remoteNames);
    connect(crDlg,SIGNAL(setActiveRemoteText(QString)),this,SLOT(setActiveRemoteText(QString)));
    crDlg->exec();
    crDlg->deleteLater();
}

void QtIrreco::setActiveRemoteText(QString name)
{
    activeRemote.clear();
    activeRemote = name;

    lw->clearScreen();
    this->setActiveRemote();
    this->writeRemoteChanges();

    this->setWindowTitle("QtIrreco:"+activeRemote);
}

void QtIrreco::deleteRemote() // Using RenameRemoteDlg class
{
    RenameRemoteDlg *rrDlg = new RenameRemoteDlg(this,activeRemote);
    rrDlg->setGeometry(0,0,800,175);
    rrDlg->setWindowTitle("Delete remote");
    rrDlg->setNameLabelText("Are you sure you wish to delete remote: "+activeRemote+" ?");
    rrDlg->setLabelGeometry(150,25,600,50);
    rrDlg->setPushButtonText("Delete");
    rrDlg->setPushButtonGeometry(300,85,200,60);
    rrDlg->enableButton(true);
    rrDlg->hideLineEdit();
    connect(rrDlg,SIGNAL(deleteRemote()),this,SLOT(deleteActiveRemote()));
    rrDlg->exec();
    delete rrDlg;
}

void QtIrreco::deleteActiveRemote()
{
    emit remoteDeleted();

    QFile file(remotesPath +activeRemote+".conf");
    file.remove();
    for (int i=0;i<remoteNames->count();i++) {
        if(remoteNames->at(i).contains(activeRemote+".conf"))
            remoteNames->removeAt(i);

    }

    if(remoteNames->count() > 0) {
        activeRemote = remoteNames->at(0);
        activeRemote.chop(5);
        this->setActiveRemote();
        lw->setActiveRemoteName(activeRemote);
        this->setWindowTitle("QtIrreco:"+activeRemote);
    }
    this->writeRemoteChanges();
    this->checkForRemotes();
    if(remoteNames->count() <= 0) {
       activeRemote.clear();
       QFile confFile(remotesConf);
       confFile.remove();
       this->noRemotes();
    }
}

void QtIrreco::about()
{
    AboutDlg *about = new AboutDlg(this);
    about->exec();
    delete about;
}

void QtIrreco::theme()
{
    ThemeManager *themeDlg = new ThemeManager(this);
    themeDlg->exec();
    delete themeDlg;
}

void QtIrreco::newButton()
{
    NewButtonDlg *newButton = new NewButtonDlg(this, lw->vector);
    connect(newButton,SIGNAL(addButtonToLayoutSignal(QString,QString, QString, QString, QStringList,int,int)),this,SLOT(addButtonToLayout(QString,QString, QString, QString, QStringList,int,int)));
    newButton->exec();
    delete newButton;
}

void QtIrreco::saveRemote()
{
    this->enableUseMenu();
    useModeEnabled = true;
    lw->setUseModeEnabled(useModeEnabled);
    lw->setFocus();
    lw->saveRemoteToFile();
    lircFileNames.clear();

    QFile file(remotesPath + activeRemote+".conf");
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    return;
    QTextStream stream(&file);
    while (!stream.atEnd()) {
        QString line = stream.readLine();

        if(line.contains(("command: "),Qt::CaseInsensitive)) {
            line = line.trimmed();
            line.remove(0,9);
            QString string;
            for(int i=0;i<line.length();i++) {
                if(line.at(i).isSpace() == false)
                    string.append(line.at(i));

                if(line.at(i).isSpace() == true) {
                    if(!lircFileNames.contains(string))
                        lircFileNames.append(string);

                    i = line.length();
                }
            }
        }
    }


    file.close();
    QStringList lircFilePaths;
    QFile lircPathsFile(lircpathsConf);
    lircPathsFile.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream in(&lircPathsFile);
    while (!in.atEnd()) {
        QString line = in.readLine();
        for(int i=0;i<lircFileNames.length();i++) {
            if(line.contains(lircFileNames.at(i),Qt::CaseInsensitive)) {
                for(int j=0;j<line.length();j++) {
                    if(line.at(j).isSpace())
                        line.remove(0,j+1);
                }
                if(!lircFilePaths.contains((line),Qt::CaseInsensitive))
                    lircFilePaths.append(line);
            }
        }
    }
    lircPathsFile.close();

    QFile remFile(remotesPath + activeRemote +".conf");
    remFile.open(QIODevice::WriteOnly | QIODevice::Append);


    QTextStream out(&remFile);
    out<< "\nLircConfs: ";
    for(int i= 0;i<lircFileNames.length();i++)
        out<< lircFileNames.at(i) +" ";

    out<< "\nLircPaths: " ;

    for(int i= 0;i<lircFilePaths.length();i++)
        out<< lircFilePaths.at(i) +" ";

    remFile.close();
}

void QtIrreco::downloadDB()
{
    DownloadUserRemoteDlg *downloadDlg = new DownloadUserRemoteDlg(this);
    connect(downloadDlg,SIGNAL(remoteDownloaded(QString)),this,SLOT(addDownloadedRemote(QString)));
    downloadDlg->exec();
    delete downloadDlg;
}

void QtIrreco::uploadDB()
{
    UploadDBDlg *uploadDlg = new UploadDBDlg(this, activeRemote);
    uploadDlg->exec();
    delete uploadDlg;
}

void QtIrreco::showRemote()
{
    if(!remoteNames->count() > 0) {

            activeRemote.clear();
            remoteNames->clear();

            QFile file(remotesConf);
            if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
            return;
            QTextStream in(&file);
            while (!in.atEnd()) {
                QString line = in.readLine();
                process_line(line);
            }
    }
}

void QtIrreco::readRemoteFile(QString line, bool lastLine)
{
    if(line.contains(("Background: "),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,12);
        if(line.contains(("default"),Qt::CaseInsensitive))
            backgroundMode = defaultBackground;
        else if(line.contains(("theme"),Qt::CaseInsensitive))
            backgroundMode = backgroundFromTheme;
        else if(line.contains(("image"),Qt::CaseInsensitive))
            backgroundMode = backgroundFromImage;
        else if(line.contains(("color"),Qt::CaseInsensitive))
            backgroundMode = backgroundFromColor;

        lw->setBackgroundMode(backgroundMode);
    }
    if(line.contains(("Image: "),Qt::CaseInsensitive)) {
        line = line.trimmed();
        backgroundName = line.remove(0,7);
    }
    if(line.contains(("Color:"),Qt::CaseInsensitive)) {
        QString str[3];

        line = line.trimmed();
        line.remove(0,7);
        int lineLength = line.length();
        int index = 0;
        for(int i=0;i<lineLength;i++) {
            if(line.at(i).isSpace() == false)
                str[index].append(line.at(i));

            if(line.at(i).isSpace() == true && index <=2)
                index++;

        }
        red = str[0];
        green = str[1];
        blue = str[2];
    }
     // Button reading
    if(line.contains(("Button:"),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,8);
        buttonText = line;
    }
    if(line.contains(("fileNameUnPressed:"),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,19);
        fileNameUnPressed = line;
    }
    if(line.contains(("fileNamePressed:"),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,17);
        fileNamePressed = line;
    }
    if(line.contains(("confFileName:"),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,14);
        confFileName = line;
    }
    if(line.contains(("position:"),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,10);
        int index;
        int lineLength;
        lineLength = line.length();
        index = line.indexOf(",");
        xPosition = line.left(index);
        yPosition = line.right(lineLength - (index+1));
    }
     if(line.contains(("timeout: "),Qt::CaseInsensitive)) {
        timeOut = 1000;
        line = line.trimmed();
        line.remove(0,9);
        bool ok;
        timeOut = line.toInt(&ok);
    }
    if(line.contains(("command: "),Qt::CaseInsensitive)) {
        line = line.trimmed();
        line.remove(0,9);
        commandChain.append(line);

    }
    if(line.contains(("end commandChain"),Qt::CaseInsensitive)) {
        int x,y;
        bool ok;
        x = xPosition.toInt(&ok,10);
        y = yPosition.toInt(&ok,10);
        position.setX(x);
        position.setY(y);
        lw->addButton(fileNameUnPressed,fileNamePressed,confFileName,buttonText,commandChain,position,timeOut);
        commandChain.clear();
    }

    if(lastLine) {
        // button reading ends
        bool ok;
        r = red.toInt(&ok,10);
        g = green.toInt(&ok,10);
        b = blue.toInt(&ok,10);

        emit rgbToLw(r,g,b);
        QColor rgb(r,g,b);
        this->changeBackground(backgroundName,rgb);
    }
}

void QtIrreco::process_line(QString line)
{
    line = line.trimmed();

    if(line.contains(("current remote"),Qt::CaseInsensitive)) {
          line.chop(17);
          activeRemote = line;
          activeRemote.chop(5);
    }
    if(line.contains(("filename of remote"),Qt::CaseInsensitive)) {
        line.chop(21);
        remoteNames->append(line);
    }
}

void QtIrreco::background()
{
    bg = new BackgroundDlg(this,activeRemote);
    connect(bg,SIGNAL(backgroundChanged(int)),lw,SLOT(setBackgroundMode(int)));
    connect(bg,SIGNAL(backgroundChanged(int)),this,SLOT(changeBackground()));
    connect(bg,SIGNAL(rgbToLw(QString,QString,QString)),lw,SLOT(setRgbValues(QString,QString,QString)));
    bg->exec();
}

void QtIrreco::changeBackground()
{
    lw->setBackgroundImagePath(bg->returnBackgroundImagePath());

    if(bg->isDefaultImageSelected()) {
        QColor defaultBgColor(50,50,50);
        palette.setColor(QPalette::Window,defaultBgColor);
        this->setPalette(palette);
        this->setAutoFillBackground(true);

        palette.setBrush(QPalette::Window, QPixmap(qtirrecoImage).scaled(800,424,Qt::IgnoreAspectRatio,Qt::FastTransformation));
        lw->setPalette(palette);
        lw->setAutoFillBackground(true);
    }
    else {

        QPixmap backImage = bg->returnBackgroundPixmap();
        QPixmap backColor = bg->returnBackgroundPixmapColor();

        if(!backColor.isNull()) {
            palette.setBrush(QPalette::Window,backColor);
            this->setPalette(palette);
            this->setAutoFillBackground(true);
        }

        if(!backImage.isNull()) {
            palette.setBrush(QPalette::Window, backImage);
            lw->setPalette(palette);
            lw->setAutoFillBackground(true);
        }
    }
}

void QtIrreco::changeBackground(QString backgroundName, QColor backgroundColor)
{
    lw->setBackgroundImagePath(backgroundName);

    if( backgroundName == "default") {
        QColor defaultBgColor(50,50,50);
        palette.setColor(QPalette::Window,defaultBgColor);
        this->setPalette(palette);
        this->setAutoFillBackground(true);

        palette.setBrush(QPalette::Window, QPixmap(qtirrecoImage).scaled(800,424,Qt::IgnoreAspectRatio,Qt::FastTransformation));
        lw->setPalette(palette);
        lw->setAutoFillBackground(true);
    }

    else {

        if(backgroundColor.isValid()) {
            palette.setColor(QPalette::Window,backgroundColor);
            this->setPalette(palette);
            this->setAutoFillBackground(true);
        }

        QPixmap backImage(backgroundName);
        if(!backImage.isNull()) {

            palette.setBrush(QPalette::Window, backImage.scaled(800,424,Qt::IgnoreAspectRatio,Qt::FastTransformation));
            lw->setPalette(palette);
            lw->setAutoFillBackground(true);
        }
    }
}
void QtIrreco::initMenu()
{
    m_ui->menu->setTitle("");
    m_ui->actionEditRemote->setText("Edit Remote");
    m_ui->actionNewRemote->setText("New Remote");
    m_ui->actionDownloadRemote->setText("Download Remote");
    m_ui->actionSelectRemote->setText("Select Remote");
    m_ui->actionThemeManager->setText("Theme Manager");
    m_ui->actionDeleteRemote->setText("Delete Remote");
    m_ui->actionAbout->setText("About");
    m_ui->actionDownloadDb->setText("Download from DB");
    m_ui->actionUploadDb->setText("Upload to DB");

    connect(m_ui->actionEditRemote,SIGNAL(triggered()),this,SLOT(editRemote()));
    connect(m_ui->actionNewRemote,SIGNAL(triggered()),this,SLOT(newRemote()));
    connect(m_ui->actionDownloadRemote,SIGNAL(triggered()),this,SLOT(downloadRemote()));
    connect(m_ui->actionSelectRemote,SIGNAL(triggered()),this,SLOT(selectRemote()));
    connect(m_ui->actionThemeManager,SIGNAL(triggered()),this,SLOT(theme()));
    connect(m_ui->actionDeleteRemote,SIGNAL(triggered()),this,SLOT(deleteRemote()));
    connect(m_ui->actionAbout,SIGNAL(triggered()),this,SLOT(about()));
    connect(m_ui->actionDownloadDb,SIGNAL(triggered()),this,SLOT(downloadDB()));
    connect(m_ui->actionUploadDb,SIGNAL(triggered()),this,SLOT(uploadDB()));

    m_ui->actionSaveRemote->setText("Save Remote");
    m_ui->actionNewButton->setText("New Button");
    m_ui->actionSnapToGrid->setText("Snap to grid");
    m_ui->actionRenameRemote->setText("Rename Remote");
    m_ui->actionBackground->setText("Background");

    connect(m_ui->actionSaveRemote,SIGNAL(triggered()),this,SLOT(saveRemote()));
    connect(m_ui->actionNewButton,SIGNAL(triggered()),this,SLOT(newButton()));
    connect(m_ui->actionSnapToGrid,SIGNAL(triggered()),this,SLOT(snapToGrid()));
    connect(m_ui->actionRenameRemote,SIGNAL(triggered()),this,SLOT(renameRemote()));
    connect(m_ui->actionBackground,SIGNAL(triggered()),this,SLOT(background()));

    m_ui->actionSaveRemote->setEnabled(false);
    m_ui->actionNewButton->setEnabled(false);
    m_ui->actionSnapToGrid->setEnabled(false);
    m_ui->actionRenameRemote->setEnabled(false);
    m_ui->actionBackground->setEnabled(false);

}

void QtIrreco::enableEditMenu()
{
    m_ui->actionSaveRemote->setEnabled(true);
    m_ui->actionNewButton->setEnabled(true);
    m_ui->actionSnapToGrid->setEnabled(true);
    m_ui->actionRenameRemote->setEnabled(true);
    m_ui->actionBackground->setEnabled(true);

    m_ui->actionEditRemote->setEnabled(false);
    m_ui->actionNewRemote->setEnabled(false);
    m_ui->actionDownloadRemote->setEnabled(false);
    m_ui->actionSelectRemote->setEnabled(false);
    m_ui->actionThemeManager->setEnabled(false);
    m_ui->actionDeleteRemote->setEnabled(false);
    m_ui->actionAbout->setEnabled(false);
    m_ui->actionDownloadDb->setEnabled(false);
    m_ui->actionUploadDb->setEnabled(false);
}

void QtIrreco::enableUseMenu()
{
    m_ui->actionEditRemote->setEnabled(true);
    m_ui->actionNewRemote->setEnabled(true);
    m_ui->actionDownloadRemote->setEnabled(true);
    m_ui->actionSelectRemote->setEnabled(true);
    m_ui->actionThemeManager->setEnabled(true);
    m_ui->actionDeleteRemote->setEnabled(true);
    m_ui->actionAbout->setEnabled(true);
    m_ui->actionDownloadDb->setEnabled(true);
    m_ui->actionUploadDb->setEnabled(true);

    m_ui->actionSaveRemote->setEnabled(false);
    m_ui->actionNewButton->setEnabled(false);
    m_ui->actionSnapToGrid->setEnabled(false);
    m_ui->actionRenameRemote->setEnabled(false);
    m_ui->actionBackground->setEnabled(false);
}

void QtIrreco::snapToGrid()
{
    if(lw->snapToGridEnabled)
        m_ui->actionSnapToGrid->setChecked(false);

    else  m_ui->actionSnapToGrid->setChecked(true);


    lw->snapToGrid();
}

void QtIrreco::addDownloadedRemote(QString remName)
{
    lastTheme = false;
    QString remConf = remName;
    remConf.chop(5);
    remoteNames->append(remName);
    activeRemote = remConf;
    //Downloads Lirc configurations if needed
    checkLircFilesForRemote(remName);
    //Downloads themes if needed
    checkThemesForRemote(remName);
}

void QtIrreco::checkThemesForRemote(QString remoteFile)
{
    QString themeTemp;
    QFile file(remotesPath + remoteFile);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        QMessageBox::warning(this,"Error with remote file",
                             "The program was unable to open " + remoteFile,Qt::NoButton,Qt::NoButton);
        return;
    }
        QTextStream in(&file);
        while (!in.atEnd()) {
            QString line = in.readLine();
            if(line.contains(("Themes: "),Qt::CaseInsensitive)) {
                line.remove(0,8);
                for(int i=0;i<line.length();i++) {
                    if(!line.at(i).isSpace())
                        themeTemp.append(line.at(i));

                    else {
                        themesForRemote.append(themeTemp);
                        themeTemp.clear();
                    }
                }

            }
            if(line.contains(("Theme versions: "),Qt::CaseInsensitive)) {
                line.remove(0,16);
                for(int i=0;i<line.length();i++) {
                    if(line.at(i) != '/')
                        themeTemp.append(line.at(i));

                    else {
                        themeVersionsForRemote.append(themeTemp);
                        themeTemp.clear();
                    }
                }

            }
        }
    file.close();

    QDir dir(themesPath);
    for(int i=0;i<themesForRemote.length();i++) {
        if(dir.exists(themesForRemote.at(i))) {
            themesForRemote.removeAt(i);
            themeVersionsForRemote.removeAt(i);
            i--;
        }
    }

    if(themesForRemote.isEmpty()) {
        this->setEnabled(true);
        lw->clearScreen();
        lw->setActiveRemoteName(activeRemote);
        this->writeRemoteChanges();
        this->checkForRemotes();
        this->setActiveRemote();
        this->setWindowTitle("QtIrreco:"+activeRemote);
        m_ui->actionEditRemote->setEnabled(true);
        m_ui->actionSelectRemote->setEnabled(true);
        m_ui->actionDeleteRemote->setEnabled(true);
        m_ui->actionUploadDb->setEnabled(true);


    }
    else {
        themeDownloadCount = 0;
        progress = new QProgressBar(this);
        progress->setGeometry(650,10,128,64);
        progress->setRange(0,100);
        connect(this,SIGNAL(progressState(int)),progress,SLOT(setValue(int)));
        connect(this,SIGNAL(themeLoaded()),progress,SLOT(deleteLater()));
        progress->show();
        this->downloadTheme();
    }
}
void QtIrreco::downloadTheme()
{
    if( themeDownloadCount < themesForRemote.count() ) {
        themeButtonIndex = 0;
        themeBgIndex = 0;
        progressValue = 0;

        QString upperCaseName = themesForRemote.at(themeDownloadCount);
        upperCaseName.replace(QString(upperCaseName.at(0)),QString(upperCaseName.at(0).toUpper()),Qt::CaseSensitive);
        QMessageBox::information(this,"Downloading theme","Downloading " + upperCaseName + " theme version " + themeVersionsForRemote.at(themeDownloadCount),
                                 QMessageBox::NoButton,QMessageBox::NoButton);
        themeNameFromConf = upperCaseName;
        themeVersion = themeVersionsForRemote.at(themeDownloadCount);
        QList<QVariant> params;
        params.append(themeNameFromConf);
        params.append(themeVersion);

        themeIdFromNameAndVersion = client->call("getThemeIdByNameAndDate",params);
    }
    else {
       if(!themesForRemote.isEmpty())
            QMessageBox::information(this,"Theme manager","Themes downloaded succesfully.",
                                 QMessageBox::NoButton,QMessageBox::NoButton);
        this->setEnabled(true);
        lw->clearScreen();
        lw->setActiveRemoteName(activeRemote);
        this->writeRemoteChanges();
        this->checkForRemotes();
        this->setActiveRemote();
        this->setWindowTitle("QtIrreco:"+activeRemote);
        m_ui->actionEditRemote->setEnabled(true);
        m_ui->actionSelectRemote->setEnabled(true);
        m_ui->actionDeleteRemote->setEnabled(true);
        m_ui->actionUploadDb->setEnabled(true);
        themesForRemote.clear();
        progressValue = 0;
        emit themeLoaded();
    }
}
void QtIrreco::checkLircFilesForRemote(QString remoteFile)
{
    QString temp;
    lircPathsForRemote.clear();
    lircConfs.clear();
    QFile file(remotesPath + remoteFile);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
    return;
    QTextStream in(&file);
    while (!in.atEnd()) {
        QString line = in.readLine();
        if(line.contains(("LircPaths: "),Qt::CaseInsensitive)) {
            line.remove(0,11);
            for(int i=0;i<line.length();i++) {
                if(!line.at(i).isSpace()) {
                    temp.append(line.at(i));
                }
                else {
                    lircPathsForRemote.append(temp);
                    temp.clear();
                }
            }
        }
        if(line.contains(("LircConfs: "),Qt::CaseInsensitive)) {
            line.remove(0,11);
            for(int i=0;i<line.length();i++) {
                if(!line.at(i).isSpace())
                    temp.append(line.at(i));

                else {
                    lircConfs.append(temp);
                    temp.clear();
                }
            }

        }
    }
    file.close();
    QDir dir(devicesPath);
    for(int i=lircConfs.length()-1;i>-1;i--) {
        if(dir.exists(lircConfs.at(i))) {
            lircConfs.removeAt(i);
            lircPathsForRemote.removeAt(i);
        }
    }

    QList<QVariant> params;
    for(callCounter=0;callCounter<lircConfs.length();callCounter++) {
        QMessageBox::information(this,"Lirc configuration","Configuration " + lircConfs.at(callCounter) + " downloaded succesfully!",
                                 QMessageBox::NoButton,QMessageBox::NoButton);
        params.append(lircPathsForRemote.at(callCounter));
        fileId = client->call("getLircFile",params);
        params.clear();
    }
}
void QtIrreco::processReturnValue(int Id,QVariant data)
{
    QStringList sList;
    sList = data.toStringList();
    if(Id == themeIdFromNameAndVersion) {
        downloadId = data.toInt();
        QList<QVariant> params;
        params.clear();
        params.append(data.toInt());
        themeDatas = client->call("getThemeById",params);

    }
    else if (Id == themeDatas) {
        QMap<QString,QVariant> array = data.toMap();
        QVariant name = array.value("name");
        themeNameFromConf = name.toString();
        QVariant user = array.value("user");
        themeAuthor = user.toString();
        QVariant comment = array.value("comment");
        themeComments = comment.toString();
        QVariant folder = array.value("folder");
        themeFolder = folder.toString();

        QVariant preview_button = array.value("preview_button");
        themePreviewButton = preview_button.toString();

        this->setWindowTitle("Loading theme buttons...");
        this->setDisabled(true);
        dbButtonIdsList.clear();
        QList<QVariant> params;
        params.clear();
        params.append(downloadId);
        themeButtonIds = client->call("getButtons",params);
    }
    else if (Id == themeButtonIds) {
        dbButtonIdsList = sList;
        this->getThemeButtonsByIds();

    }
    else if (Id == buttonData) { // COPY BUTTONS AND MAKE FOLDERS

        progressValue += 100 / (dbButtonIdsList.count()+2);
        emit progressState(progressValue);
        themeButtonIndex++;
        QMap<QString,QVariant> array = data.toMap();
        QVariant name = array.value("name");
        QVariant allow_text = array.value("allow_text");
        QVariant text_format_up = array.value("text_format_up");
        QVariant text_format_down = array.value("text_format_down");
        QVariant text_padding = array.value("text_padding");
        QVariant h_align = array.value("h_align");
        QVariant v_align = array.value("v_align");
        QVariant image_up_hash = array.value("image_up_hash");
        unpressedHash = image_up_hash.toString();
        QVariant image_up_name = array.value("image_up_name");
        QVariant image_down_hash = array.value("image_down_hash");
        pressedHash = image_down_hash.toString();
        QVariant image_down_name = array.value("image_down_name");
        QVariant folder = array.value("folder");

        QDir dir;
        dir.mkpath(themesPath + themeFolder + "/buttons/" + folder.toString().toLower());
        buttonPath = QString(themesPath + themeFolder + "/buttons/" + folder.toString().toLower());
        if(themeButtonIndex == 1) {
            QFile themeConf(themesPath + themeFolder + "/theme.conf");
            if (!themeConf.open(QIODevice::ReadWrite | QIODevice::Text))
            return;
            QTextStream in(&themeConf);
            in << "[theme]\n";
            in << "name="+ themeNameFromConf +"\n";
            in << "source=database\n";
            in << "author="+ themeAuthor +"\n";
            in << "comments="+ themeComments +"\n";
            in << "version="+ themeVersion +"\n";
            in << "preview_button=" + themePreviewButton;
            themeConf.close();
        }



        QFile buttonConf(buttonPath+"/button.conf");
        if (!buttonConf.open(QIODevice::ReadWrite | QIODevice::Text))
        return;
        QTextStream in2(&buttonConf);
        in2 << "[theme-button]\n";
        in2 << "name="+name.toString()+"\n";
        in2 << "allow-text="+allow_text.toString()+"\n";
        in2 << "up=unpressed.png\n";
        in2 << "down=pressed.png\n";
        if(allow_text.toBool() == true) {
            in2 << "text-format-up="+text_format_up.toString()+"\n";
            in2 << "text-format-down="+text_format_down.toString()+"\n";
        }

        buttonConf.close();
        // Get button imagedatas and make .png files
        this->getUnpressedImage();
    }
    else if (Id == unpressedToFile)  { // Name always unpressed.png
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        previewPix.save(buttonPath+"/unpressed.png","PNG");
        this->getPressedImage();
    }
    else if (Id == pressedToFile) { // Name always pressed.png
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        previewPix.save(buttonPath+"/pressed.png","PNG");
        this->getThemeButtonsByIds();
    }
    else if (Id == themeBgIds) {
        dbBackgroundIdsList = sList;
        if(dbBackgroundIdsList.count() > 0 && !dbBackgroundIdsList.contains("0")) {
            this->setWindowTitle("Downloading theme backgrounds...");
            this->getThemeBackgroundsByIds();
        }
        else {
            themeDownloadCount++;
            this->downloadTheme();
        }
    }
    else if (Id == bgData) {
        themeBgIndex++;
        progressValue += 100 / (dbButtonIdsList.count()+2);
        emit progressState(progressValue);
        QMap<QString,QVariant> array = data.toMap();
        QVariant name = array.value("name");
        QVariant image_hash = array.value("image_hash");
        bgHash = image_hash.toString();
        QVariant image_name = array.value("image_name");
        QVariant folder = array.value("folder");

        QDir dir;
        dir.mkpath(themesPath + themeFolder + "/bg/" + folder.toString());
        bgPath = QString(themesPath + themeFolder + "/bg/" + folder.toString());
        QFile bgConf(bgPath+"/bg.conf");
        if (!bgConf.open(QIODevice::ReadWrite | QIODevice::Text))
        return;
        QTextStream in(&bgConf);
        in << "[theme-bg]\n";
        in << "name="+name.toString()+"\n";
        in << "image=image.png";
        bgConf.close();
        // Get background imagedatas and make .png files
        this->getBgImage();
    }
    else if (Id == bgToFile) {
        QPixmap previewPix(128,128);
        QByteArray pb = data.toByteArray();
        QByteArray ba = QByteArray::fromBase64(pb);
        previewPix.loadFromData(ba);
        previewPix.save(bgPath+"/image.png","PNG");
        this->getThemeBackgroundsByIds();
    }
    else if (Id == fileId) {
        if(!QFile::exists(devicesPath + lircConfs.at(callCounter-1))) {
            QFile deviceFile(devicesPath +lircConfs.at(callCounter-1));
            deviceFile.open(QIODevice::WriteOnly | QIODevice::Text);
            QTextStream out(&deviceFile);
            for(int i= 0; i< sList.count();i++)
                out << sList.at(i);

            deviceFile.close();

            QFile LircConfFile("/etc/lircd.conf");
            QStringList deviceList;
            LircConfFile.open(QIODevice::ReadWrite | QIODevice::Text);
            QTextStream io(&LircConfFile);
            while (!io.atEnd()) {
                QString line = io.readLine();
                deviceList.append(line);
            }
            LircConfFile.resize(0);
            QString lastLine;

            for(int j = 0; j<sList.count();j++) {
                lastLine =  sList.at(j);

                if (lastLine.contains(("end"), Qt::CaseInsensitive)) {
                    if(!deviceList.contains((lircConfs.at(callCounter-1)),Qt::CaseInsensitive))
                       deviceList.append("include \"" + devicesPath + lircConfs.at(callCounter-1) + "\"");

                    for(int i= 0; i< deviceList.count();i++)
                        io << deviceList.at(i)+"\n";

                    j=sList.count();
                }
            }
            LircConfFile.close();

            QFile lircPathsFile(lircpathsConf);
            lircPathsFile.open(QIODevice::WriteOnly | QIODevice::Append);
            QTextStream stream(&lircPathsFile);
            stream << lircConfs.at(callCounter-1) + " "+ lircPathsForRemote.at(callCounter-1) + "\n";
            lircPathsFile.close();

        }
    }
}

void QtIrreco::error(int i ,QString string)
{
    QMessageBox::warning(this,"Error: "+QString::number(i,10),string,
                                 QMessageBox::NoButton,QMessageBox::NoButton);
}

void QtIrreco::fault(int i,int j,QString string)
{
    Q_UNUSED(j);
    QMessageBox::warning(this,"Fault: "+QString::number(i,10), string,
                                 QMessageBox::NoButton,QMessageBox::NoButton);
}
void QtIrreco::getUnpressedImage()
{
    QList<QVariant> params;
    params.clear();
    params.append(unpressedHash);
    QString unpressedName = "unpressed.png";
    params.append(unpressedName);
    unpressedToFile = client->call("getImageData",params);
}
void QtIrreco::getPressedImage()
{
    QList<QVariant> params;
    params.clear();
    params.append(pressedHash);
    QString pressedName = "pressed.png";
    params.append(pressedName);
    pressedToFile = client->call("getImageData",params);
}
void QtIrreco::getBgImage()
{
    QList<QVariant> params;
    params.clear();
    params.append(bgHash);
    QString pressedName = "image.png";
    params.append(pressedName);
    bgToFile = client->call("getImageData",params);
}
void QtIrreco::getThemeButtonsByIds()
{
     if(themeButtonIndex < dbButtonIdsList.count()) {
            QList<QVariant> params;
            params.clear();
            QString idString = dbButtonIdsList.at(themeButtonIndex);
            bool ok;
            int idInt = idString.toInt(&ok,10);
            params.append(idInt);
            buttonData = client->call("getButtonById",params);
        }
    else {
        dbBackgroundIdsList.clear();
        QList<QVariant> params;
        params.clear();
        params.append(downloadId);
        themeBgIds = client->call("getBackgrounds",params);
    }
}
void QtIrreco::getThemeBackgroundsByIds()
{
    if(themeBgIndex < dbBackgroundIdsList.count()) {
         QList<QVariant> params;
         params.clear();
         QString idString = dbBackgroundIdsList.at(themeBgIndex);
         bool ok;
         int idInt = idString.toInt(&ok,10);
         params.append(idInt);
         bgData = client->call("getBgById",params);
    }
    if(themeBgIndex == dbBackgroundIdsList.count()) {
        themeDownloadCount++;
        this->downloadTheme();
    }
}
