#include <QDebug>
#include <list>
#include <cstdio>

#include "mnenc.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "despotifycallback.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    /* Initialize despotify */
    this->dw = new DespotifyWrap();
    qDebug() << "Is spotify initialized? " << this->dw->IsInitialized();

    this->last_sec = 0;

    /* Connect signals */
    this->connect(this->dw, SIGNAL(PlaylistsUpdated()),this, SLOT(PlaylistsUpdated()));
    this->connect(&dc, SIGNAL(NewTrack(int,int,int,track*)),this, SLOT(NewTrack(int,int,int,track*)));
    this->connect(&dc, SIGNAL(EndOfPlaylist()),this, SLOT(EndOfPlaylist()));
    this->connect(&dc, SIGNAL(TellTime(int,int,int)),this, SLOT(TellTime(int,int,int)));

    this->connect(this->dw->player, SIGNAL(Playing()),this, SLOT(Playing()));
    this->connect(this->dw->player, SIGNAL(Paused()),this, SLOT(Paused()));
    this->connect(this->dw->player, SIGNAL(Exiting()),this, SLOT(Exiting()));

    /* Initiate UI */
    this->ui->swWidgets->setCurrentWidget(this->ui->pgLogin);

    this->ui->actAlbum->setEnabled(false);
    this->ui->actPlaylist->setEnabled(false);
    this->ui->actSearch->setEnabled(false);

    this->ui->twSearch->setColumnWidth(0,60);
    this->ui->twSearch->setColumnWidth(1,250);
    this->ui->twSearch->setColumnWidth(2,150);
    this->ui->twSearch->setColumnWidth(3,150);
    this->ui->twSearch->setColumnWidth(4,70);
    this->ui->twSearch->setColumnWidth(5,70);
    this->ui->twSearch->sortByColumn(0,Qt::AscendingOrder);

    this->ui->twPlaylist->setColumnWidth(0,60);
    this->ui->twPlaylist->setColumnWidth(1,250);
    this->ui->twPlaylist->setColumnWidth(2,150);
    this->ui->twPlaylist->setColumnWidth(3,150);
    this->ui->twPlaylist->setColumnWidth(4,70);
    this->ui->twPlaylist->setColumnWidth(5,70);
    this->ui->twPlaylist->sortByColumn(0,Qt::AscendingOrder);

    this->ui->twAlbum->setColumnWidth(0,55);
    this->ui->twAlbum->setColumnWidth(1,170);
    this->ui->twAlbum->setColumnWidth(2,190);
    this->ui->twAlbum->setColumnWidth(2,70);
    this->ui->twAlbum->setColumnWidth(3,60);

    /* Check if we have an stored user and password pair */
    string username = getenv("USER"); //Get username
    string pfile = "/home/" + username + "/.qtify/.pfile";
    string ufile = "/home/" + username + "/.qtify/.ufile";

    mnenc passtore;

    if(passtore.from_file(pfile)) {
        if(passtore.user_from_file(ufile)) {
            /* We have a stored password, go on */
            this->ui->leLogin->setText(QString(passtore.get_user().c_str()));
            this->ui->lePassword->setText(QString(passtore.get_dec().c_str()));
            this->ui->cbSave->setChecked(true);
        }
    }
}

MainWindow::~MainWindow()
{
    delete ui;
    this->dw->Exit();
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::on_pbLogin_clicked()
{

    /* Convert lineedit text to char* */
    QByteArray user = this->ui->leLogin->text().toLatin1();
    char *c_user = user.data();
    QByteArray pass = this->ui->lePassword->text().toLatin1();
    char *c_pass = pass.data();

    /* Try to authenticate */
    this->ui->lblLoginError->setText(tr("Authenticating..."));
    qApp->processEvents();

    if(!this->dw->Authenticate(c_user,c_pass)) {
        qDebug() << "Authentication failure";
        this->ui->lblLoginError->setText(this->dw->GetError());
    } else {
        qDebug() << "Authentication successful";
        /* Update playlists */
        this->ui->lblLoginError->setText(tr("Updating playlists..."));
        qApp->processEvents();
        this->dw->UpdatePlaylists();
    }
}

void MainWindow::PlaylistsUpdated() {

    qDebug() << "Received signal: PlaylistsUpdated";

    /* Show inside widget */
    this->ui->swWidgets->setCurrentWidget(this->ui->pgInside);
    this->ui->actPlaylist->setEnabled(true);
    this->ui->actSearch->setEnabled(true);

    /* Clear list */
    this->ui->cbPlaylists->clear();

    /* Insert all playlists in combo box */
    playlist* p = this->dw->GetPlaylists();
    if(p && p != NULL){
        this->ui->cbPlaylists->addItem(tr("Select playlist"),QString("0"));
        for (; p; p = p->next)
        {
            if(p != NULL) {
                this->ui->cbPlaylists->addItem(QString::fromUtf8(p->name), QString::fromUtf8((char*)p->playlist_id));
                qDebug() << "Playlist_id: " << p->playlist_id;
            } else {
                qDebug() << "This really shouldn't happen... really...";
            }
        }
    } else {
        this->ui->cbPlaylists->addItem(tr("Select playlist"),QString("0"));
    }
}

void MainWindow::PrintTracksSearch() {

    this->ui->twSearch->clear();

    track* head = this->dw->pending_search;

    int count=0;
    for (track* t = head; t; t = t->next) {
        if (t->has_meta_data) {
            //wrapper_wprintf(L"%3d: %-40s %2d:%02d ", count++, t->title,t->length / 60000, t->length % 60000 / 1000);
            QString t_artist;

            for (struct artist* a = t->artist; a; a = a->next)
                t_artist += QString().fromUtf8(a->name);

            QString t_song = QString().fromUtf8(t->title);
            QString t_length = QString("%1:%2").arg((int)(t->length / 60000),2,10,QChar('0')).arg((int)(t->length % 60000 / 1000),2,10,QChar('0'));
            QString t_album = QString().fromUtf8(t->album);
            QString t_popularity = QString().setNum(t->popularity);

            /* Check if song is playable */
            if(t->playable) {

                QTreeWidgetItem *t_item = new QTreeWidgetItem(this->ui->twSearch);
                t_item->setSizeHint(0,QSize(60,30));
                t_item->setText(0, QString("%1").arg(count + 1,3,10,QChar(' ')));
                t_item->setText(1, t_song);
                t_item->setText(2, t_artist);
                t_item->setText(3, t_album);
                t_item->setText(4, t_length);
                int icon=(int)(t->popularity*10/2);
                if(icon>0&&icon<6) {
                    t_item->setIcon(5, QPixmap(QString(":/gfx/res/rating_%1.png").arg(icon)));
                }
            }
        }
        count++;
    }
}

void MainWindow::PrintTracksAlbum() {

    this->ui->twAlbum->clear();

    track* head = this->dw->album->tracks;

    int count=0;
    for (track* t = head; t; t = t->next) {
        if (t->has_meta_data) {
            //wrapper_wprintf(L"%3d: %-40s %2d:%02d ", count++, t->title,t->length / 60000, t->length % 60000 / 1000);
            QString t_artist;

            for (struct artist* a = t->artist; a; a = a->next)
                t_artist += QString().fromUtf8(a->name);

            QString t_song = QString().fromUtf8(t->title);
            QString t_length = QString("%1:%2").arg((int)(t->length / 60000),2,10,QChar('0')).arg((int)(t->length % 60000 / 1000),2,10,QChar('0'));
            QString t_popularity = QString().setNum(t->popularity);

            /* Check if song is playable */
            if(t->playable) {

                QTreeWidgetItem *t_item = new QTreeWidgetItem(this->ui->twAlbum);
                t_item->setSizeHint(0,QSize(60,30));
                t_item->setText(0, QString("%1").arg(count + 1,3,10,QChar(' ')));
                t_item->setText(1, t_song);
                t_item->setText(2, t_artist);
                t_item->setText(3, t_length);
                int icon=(int)(t->popularity*10/2);
                if(icon>0&&icon<6) {
                    t_item->setIcon(4, QPixmap(QString(":/gfx/res/rating_%1.png").arg(icon)));
                }
            }
        }
        count++;
    }
}

void MainWindow::ShowAlbum(track *t,int offset) {
    /* Make sure we have a playlist to fetch from */
    if(t) {
        for (int i=0; i<offset && t; i++) {
            t = t->next;
        }
        if (t) {
            /* Show the album */
            this->dw->GetAlbumFromID(t->album_id);
            if(this->dw->album) {
                this->ui->lblAAlbum->setText(QString(this->dw->album->name)+QString(" (")+QString().setNum(this->dw->album->year)+QString(")"));

                if(!t->artist->next) {
                    this->ui->lblAArtist->setText(QString(t->artist->name));
                    /* Status changed to playing - increasing priority */
                } else {
                    this->ui->lblAArtist->setText(tr("Various artists"));
                }

                this->ui->lblAMoreInfo->setText(tr("Tracks: ")+QString().setNum(this->dw->album->num_tracks));


                /* Get albumart */
                const uchar* albumart;
                int img_len;
                albumart = this->dw->GetImage(this->dw->album->cover_id,&img_len);
                if(albumart != NULL) {
                    QImage i;
                    i.loadFromData(albumart,img_len,0);
                    this->ui->lblAlbumArt->setPixmap(QPixmap::fromImage(i));
                }

                this->PrintTracksAlbum();
                this->dw->pending_album = this->dw->album->tracks;
                /* Make sure the album-view button is enabled */
                this->ui->actAlbum->setEnabled(true);
                this->ui->swInside->setCurrentWidget(this->ui->pgAlbum);
            }
        }
    } else {
        qDebug() << "No talbums to show there mate";
    }
}

void MainWindow::PrintTracksPlaylist() {

    this->ui->twPlaylist->clear();

    track* head = this->dw->pending_playlist;
    int count=0;
    for (track* t = head; t; t = t->next) {
        if (t->has_meta_data) {
            //wrapper_wprintf(L"%3d: %-40s %2d:%02d ", count++, t->title,t->length / 60000, t->length % 60000 / 1000);
            QString t_artist;

            for (struct artist* a = t->artist; a; a = a->next)
                t_artist += QString().fromUtf8(a->name);

            QString t_song = QString().fromUtf8(t->title);
            QString t_length = QString("%1:%2").arg((int)(t->length / 60000),2,10,QChar('0')).arg((int)(t->length % 60000 / 1000),2,10,QChar('0'));
            QString t_album = QString().fromUtf8(t->album);
            QString t_popularity = QString().setNum(t->popularity);

            /* Check if song is playable */
            if(t->playable) {

                QTreeWidgetItem *t_item = new QTreeWidgetItem(this->ui->twPlaylist);
                t_item->setSizeHint(0,QSize(60,30));
                t_item->setText(0, QString("%1").arg(count + 1,3,10,QChar(' ')));
                t_item->setText(1, t_song);
                t_item->setText(2, t_artist);
                t_item->setText(3, t_album);
                t_item->setText(4, t_length);
                int icon=(int)(t->popularity*10/2);
                if(icon>0&&icon<6) {
                    t_item->setIcon(5, QPixmap(QString(":/gfx/res/rating_%1.png").arg(icon)));
                }
            }
        }
        count++;
    }
}

void MainWindow::on_pushButton_4_clicked()
{
    this->dw->PlayPrev();
}

void MainWindow::on_pushButton_3_clicked()
{
    this->dw->PlayNext();
}

void MainWindow::on_pushButton_clicked()
{
    this->dw->player->PlayPause();
}

void MainWindow::NewTrack(int min, int sec, int totsec, track *t) {


    this->song_tot_len=totsec;
    this->ui->hsElapsed->setMaximum(totsec);

}

void MainWindow::EndOfPlaylist() {

    /* Force player to try to pley next song (=first song) */
    this->dw->PlayNext();

}

void MainWindow::TellTime(int min, int sec, int totsec) {

    if(this->last_sec!=totsec) {
        int t_min = (int)this->song_tot_len/60;
        int t_sec = (int)this->song_tot_len%60;

        QString elapse;
        elapse = QString("%1:%2 / %3:%4").arg(min,2,10,QChar('0')).arg(sec,2,10,QChar('0')).arg(t_min,2,10,QChar('0')).arg(t_sec,2,10,QChar('0'));

        this->ui->lblElapsed->setText(elapse);
        this->ui->hsElapsed->setValue(totsec);

        this->last_sec = totsec;
    }

}

void MainWindow::Playing() {
    qDebug() << "Playing";
}

void MainWindow::Paused() {
    qDebug() << "Paused";
}

void MainWindow::Exiting() {
    qDebug() << "Exiting";
}

void MainWindow::on_cbPlaylists_currentIndexChanged(int index)
{

    qDebug() << "Show playlist button clicked";
    /* Iterate through all playlists */
    playlist* p = this->dw->GetPlaylists();
    if(p!=NULL) {
        for(; p; p = p->next)
        {
            if(QString((char*)p->playlist_id) == this->ui->cbPlaylists->itemData(this->ui->cbPlaylists->currentIndex())) {
                this->dw->pending_playlist = p->tracks;
                this->PrintTracksPlaylist();
            }
        }
    }
    this->ui->cbPlaylists->setCurrentIndex(0);
}

void MainWindow::on_leSearch_returnPressed()
{
    QByteArray search = this->ui->leSearch->text().toUtf8();
    char *c_search = search.data();

    if(this->dw->Search(c_search))
        this->PrintTracksSearch();

    this->ui->leSearch->setText(QString(""));
}

void MainWindow::on_twSongs_itemClicked(QTreeWidgetItem* item, int column)
{

}

void MainWindow::on_pbSearch_clicked()
{
    this->on_leSearch_returnPressed();
}

void MainWindow::on_actionPlaylist_triggered()
{

}

void MainWindow::on_actionTest_triggered()
{

}

void MainWindow::on_twSearch_itemClicked(QTreeWidgetItem* item, int column)
{

    if(column==3) {
        this->ShowAlbum(this->dw->pending_search,QString(item->text(0)).toInt()-1);
    } else {
        qDebug() << "Playing offset " << QString(item->text(0));
        if(this->dw->pending_search){
            this->dw->lastlist = this->dw->pending_search;
        }
        this->dw->Play(QString(item->text(0)).toInt()-1);
    }

}

void MainWindow::on_actPlaylist_triggered()
{
    this->ui->swInside->setCurrentWidget(this->ui->pgPlaylists);
}

void MainWindow::on_actSearch_triggered()
{
    this->ui->swInside->setCurrentWidget(this->ui->pgSearch);
}

void MainWindow::on_actAlbum_triggered()
{
    this->ui->swInside->setCurrentWidget(this->ui->pgAlbum);
}

void MainWindow::on_twPlaylist_itemClicked(QTreeWidgetItem* item, int column)
{

    if(column==3) {
        this->ShowAlbum(this->dw->pending_playlist,QString(item->text(0)).toInt()-1);
    } else {
        qDebug() << "Playing offset " << QString(item->text(0));
        if(this->dw->pending_playlist){
            this->dw->lastlist = this->dw->pending_playlist;
        }
        this->dw->Play(QString(item->text(0)).toInt()-1);
    }

}

void MainWindow::on_twAlbum_itemClicked(QTreeWidgetItem* item, int column)
{
    qDebug() << "Playing offset " << QString(item->text(0));
    if(this->dw->pending_album){
        this->dw->lastlist = this->dw->pending_album;
    }
    this->dw->Play(QString(item->text(0)).toInt()-1);
}

void MainWindow::on_cbSave_clicked(bool checked)
{
    /* Prepare file paths */
    string username = getenv("USER"); //Get username
    string pfile = "/home/" + username + "/.qtify/.pfile";
    string ufile = "/home/" + username + "/.qtify/.ufile";

    if(checked) { /* checked - save password */

        /* Convert lineedit text to char* */
        QByteArray user = this->ui->leLogin->text().toLatin1();
        char *c_user = user.data();
        QByteArray pass = this->ui->lePassword->text().toLatin1();
        char *c_pass = pass.data();


        /* Save username and encrypted password */
        mnenc passtore;
        passtore.set_user(c_user);
        passtore.encrypt(c_pass);

        if(!(passtore.to_file(pfile))) qDebug() << "Failed saving username/password";
        if(!(passtore.user_to_file(ufile))) qDebug() << "Failed saving username/password";

    } else {

        /* unchecked - clear password */
        std::remove(pfile.c_str());
        std::remove(ufile.c_str());

    }
}

void MainWindow::on_pbHideAlbumart_clicked()
{
    if(this->ui->pbHideAlbumart->text() == QString("<")){
        this->ui->lblAAlbum->hide();
        this->ui->lblAArtist->hide();
        this->ui->lblAMoreInfo->hide();
        this->ui->lblAlbumArt->hide();

        this->ui->twAlbum->setColumnWidth(0,55);
        this->ui->twAlbum->setColumnWidth(1,270);
        this->ui->twAlbum->setColumnWidth(2,270);
        this->ui->twAlbum->setColumnWidth(3,70);
        this->ui->twAlbum->setColumnWidth(4,60);

        this->ui->pbHideAlbumart->setText(QString(">"));
    } else {
        this->ui->lblAAlbum->show();
        this->ui->lblAArtist->show();
        this->ui->lblAMoreInfo->show();
        this->ui->lblAlbumArt->show();

        this->ui->twAlbum->setColumnWidth(0,55);
        this->ui->twAlbum->setColumnWidth(1,130);
        this->ui->twAlbum->setColumnWidth(2,130);
        this->ui->twAlbum->setColumnWidth(3,70);
        this->ui->twAlbum->setColumnWidth(4,60);

        this->ui->pbHideAlbumart->setText(QString("<"));
    }
}
