/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the examples of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QtGui>
#include <QtMaemo5>

// ![0]
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
// ![0]

/*! \class SimpleProgressBar
  The SimpleProgressBar class will draw a progress bar that looks similar to the one used
  by the Maemo 5 volume status.
  The "normal" progress bar cannot be used here as it has uggly black bars on the top and bottom
  that are caused by the theme pictures used.
*/
// ![3]
class SimpleProgressBar : public QWidget
// ![3]
{
    Q_OBJECT

public:
    SimpleProgressBar(QWidget *parent = 0)
        : QWidget(parent)
          , _value(0)
    { }

    QSize sizeHint() const
    {
        return QSize(300, 25);
    }

    QSize minimumSize() const
    {
        return QSize(20, 5);
    }

    void paintEvent(QPaintEvent* event)
    {
        Q_UNUSED(event);

        QPainter painter(this);

        int barHeight = qMin(25, height());
        int barOffsetY = (height() - barHeight) / 2;

        // -- first paint the bar itself
        int barWidth = width() * _value / 100;

        painter.fillRect(1, barOffsetY, barWidth - 1, barHeight - 2,
                palette().color(QPalette::WindowText));

        // -- and now paint the border
        painter.setPen(palette().color(QPalette::Midlight));
        painter.drawRoundedRect(0, barOffsetY, width() - 1, barHeight - 1, 2, 2);
    }

    int value()
    {
        return _value;
    }

    void setValue(int newValue)
    {
        _value = qBound(0, newValue, 100);
        update();
    }

protected:
    int _value;

};

class ZoomWidget : public QWidget
{
    Q_OBJECT

public:
    ZoomWidget()
        : zoom(50)
    {
        QVBoxLayout *layout = new QVBoxLayout(this);
        label = new QLabel;
        layout->addWidget(label);

        box = new QMaemo5InformationBox(this);
        QWidget *innerBox = new QWidget();
        QHBoxLayout *hBox = new QHBoxLayout(innerBox);

        boxLabel = new QLabel(tr("Zoom"));
        boxLabel->setTextInteractionFlags(Qt::NoTextInteraction);
        boxLabel->setFocusPolicy(Qt::NoFocus);

        boxProgress = new SimpleProgressBar();

        hBox->addStretch(1); // use two stretch to center the label and the progress
        hBox->addWidget(boxLabel);
        hBox->addWidget(boxProgress);
        hBox->addStretch(1);
        box->setWidget(innerBox);

        // we have a special hack to give the right palette to labels inside an information box. So we also need to give this style to the progress bar.
// ![4]
        boxProgress->setPalette(boxLabel->palette());
// ![4]

        updateText();
        grabZoomKeys(true); // on my device there was no need to ungrab them after closing.
    }

    /*! Tell maemo-status-volume to grab/ungrab increase/decrease keys
    */
// ![1]
    void grabZoomKeys(bool grab)
    {
        if (!winId())
            qWarning("can't grab keys unless we have a window id");

        unsigned long val = (grab) ? 1 : 0;
        Atom atom;
        atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", 0);
        XChangeProperty (QX11Info::display(),
                winId(),
                atom,
                XA_INTEGER,
                32,
                PropModeReplace,
                (unsigned char *) &val,
                1);
    }
// ![1]

    void updateText()
    {
        label->setText(tr("<p align=\"center\">Please press the device zoom keys<br>Zoom is at %1</p>").arg(zoom));
    }

    /*! Shows an information box with the current zoom state */
    void showInfo()
    {
        // -- reset the timeout in case the box is already visible we want the full time again.
        box->setTimeout(QMaemo5InformationBox::DefaultTimeout);

        boxProgress->setValue(zoom);
        box->show();
    }

protected:
// ![2]
    void keyPressEvent(QKeyEvent* event)
    {
        switch (event->key()) {
        case Qt::Key_F7:
            if (zoom < 100)
                zoom++;
            updateText();
            showInfo();
            event->accept();
            break;

        case Qt::Key_F8:
            if (zoom > 0)
                zoom--;
            updateText();
            showInfo();
            event->accept();
            break;
        }
    }
// ![2]

    QLabel *label;
    QMaemo5InformationBox *box;
    SimpleProgressBar *boxProgress;
    QLabel *boxLabel;

    int zoom;
};

int main(int argc, char *argv[])
{
    QApplication app(argc, argv);

    ZoomWidget w;
    w.show();

    return app.exec();
}

#include "main.moc"
