#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow); //window is part of window stack
#endif

    QCoreApplication::setApplicationName("QSpot"); //Setup app name (needs to be FMRadio in order to override silent profile)
    QCoreApplication::setApplicationVersion(APP_VERSION); //Setup app version

    m_pAppSettings = new QSettings("qspot","qspot", this); //allocate application settings instance
    if(m_pAppSettings->contains("QSpot/SaveCredentials"))
    { //removed unused settings entries
        m_pAppSettings->remove("QSpot/SaveCredentials");
        m_pAppSettings->remove("QSpot/UserID");
        m_pAppSettings->remove("QSpot/Password");
    }

    setupUILang(); //setup the UI language
    setWindowTitle(tr("QSpot - Playlists"));

    m_playing = false;
    m_update = false;

    m_pSpectrum = new SpectrumAnalyzer(this,55,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    m_pSyncProgress = new QProgressBar(this); //allocate progressbar indicator (offline synchronisation status)
    m_pSyncProgress->setVisible(false);
    m_pSyncProgress->setMaximum(100);
    m_pSyncProgress->setMinimum(0);

    m_pSession = new Spotify::Session(this); //create new Spotify session instance
    m_pOssoWrapper = new OssoWrapper("QSpot", APP_VERSION, this); //create wrapper instance for libosso

#ifdef Q_WS_MAEMO_5
   if(m_pOssoWrapper)
    { //register / declare new playback object with OSSO libplayback (whitelisting support / support for playing in silent profile)
        m_pOssoWrapper->initOssoPlayback("QSpot");
        m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_STOP); //request to enter initial stopped state
    }

#endif

    m_pDBusWrapper = new DBusWrapper(this); //allocate D-bus interface wrapper
    if(m_pDBusWrapper)
    {
        connect(m_pDBusWrapper,SIGNAL(onDisplayStateChange(bool)),SLOT(OnDisplayStateChanged(bool)));
#ifdef Q_WS_MAEMO_5
        m_pDBusWrapper->enableHWZoomKeys(); //enable / grab control of the HW zoom controls (for volume control when phone is locked)
        connect(m_pDBusWrapper,SIGNAL(onZoomKeyPressed(quint32)),SLOT(OnHWZoomKeyPressed(quint32)));
#endif
    }

    m_pLastFMScrobbler = new LastFMScrobbler(this); //allocate Last.fm scrobbler instance
    m_pLastFMScrobbler->setAPIKey("f950a6e7870997ea77cb0ecc9c8fb962"); //set associated Last.fm API key
    m_pLastFMScrobbler->setSecret("48875d761d06f7b0efd37304e0226e9f"); //set associated Last.fm shared secret
    m_pLastFMScrobbler->setUserAgent("QSpot " + QString(APP_VERSION)); //set client user agent
    m_pLastFMScrobbler->setCachePath("/home/user/");
    connect(m_pLastFMScrobbler,SIGNAL(authenticateSessionStatus(bool,const QString&)),SLOT(OnScrobblingStatus(bool, const QString&)));
    connect(m_pLastFMScrobbler,SIGNAL(updateNowPlayingStatus(bool,const QString&)),SLOT(OnScrobblingStatus(bool, const QString&)));
    connect(m_pLastFMScrobbler,SIGNAL(scrobbleTracksStatus(bool,const QString&)),SLOT(OnScrobblingStatus(bool, const QString&)));

    m_pHelpViewer = new HelpViewer(this); //allocate online help instance

    m_pLoginDlg = new LoginDialog(m_pSession, m_pAppSettings, this); //allocate login dialog instance
    connect(m_pLoginDlg,SIGNAL(doSpotifyLogin(const QString&, const QString&, bool)),SLOT(OnSpotifyLogin(const QString&, const QString&, bool)));
    m_pSettingsDlg = new SettingsDialog(m_pAppSettings, this); //allocate settings dialog instance

    //set up the the main window menu

    m_pSettingsAction = menuBar()->addAction(tr("Settings"));
    m_pSettingsAction->setData(MENU_ACTION_SETTINGS);
#ifdef Q_WS_MAEMO_5
    m_pFMTransAction = menuBar()->addAction(tr("FM Transmitter"));
    m_pFMTransAction->setData(MENU_ACTION_FM_TRANSMITTER);
#endif
    m_pHelpAction = menuBar()->addAction(tr("Online Help"));
    m_pHelpAction->setData(MENU_ACTION_ONLINE_HELP);
    m_pAboutAction = menuBar()->addAction(tr("About"));
    m_pAboutAction->setData(MENU_ACTION_ABOUT);
    m_pLogoutAction =  menuBar()->addAction(tr("Log Out"));
    m_pLogoutAction->setData(MENU_ACTION_LOGOUT);
    m_pOnlineStateAction = menuBar()->addAction(tr("Go Offline"));
    m_pOnlineStateAction->setData(MENU_ACTION_OFFLINE_MODE);
    m_pOnlineStateAction->setCheckable(true); //checkable action
    m_pOnlineStateAction->setChecked(false);
    connect(m_pOnlineStateAction,SIGNAL(triggered(bool)),SLOT(OnOfflineModeStateToggled(bool)));

    m_pTrackView = new TrackView(m_pSession,this);
    connect(m_pTrackView,SIGNAL(playQueueSet(const QList<Spotify::Track*>&,qint32)),SLOT(OnSetPlayQueue(const QList<Spotify::Track*>&,qint32)));
    connect(m_pTrackView,SIGNAL(showFMTransmitterDialog()),SLOT(OnDisplayFMTransmitterDialog()));
    connect(m_pTrackView,SIGNAL(showAlbumView(Spotify::Album*)),SLOT(OnShowAlbumView(Spotify::Album*)));
    connect(m_pTrackView,SIGNAL(showArtistView(Spotify::Artist*)),SLOT(OnShowArtistView(Spotify::Artist*)));

    m_pAlbumView = new AlbumView(m_pSession,m_pTrackView);
    connect(m_pAlbumView,SIGNAL(playQueueSet(const QList<Spotify::Track*>&,qint32)),SLOT(OnSetPlayQueue(const QList<Spotify::Track*>&,qint32)));

    m_pQueueView = new QueueView(m_pSession,m_pOssoWrapper,m_pDBusWrapper,this); //initial parent set to current instance
    connect(m_pQueueView,SIGNAL(playQueuePosUpdated(qint32)),SLOT(OnPlayQueuePosUpdated(qint32)));
    connect(m_pQueueView,SIGNAL(playQueueSet(QList<Spotify::Track*>,qint32)),SLOT(OnSetPlayQueue(QList<Spotify::Track*>,qint32)));
    connect(m_pQueueView,SIGNAL(showFMTransmitterDialog()),SLOT(OnDisplayFMTransmitterDialog()));
    connect(m_pQueueView,SIGNAL(showAlbumView(Spotify::Album*)),SLOT(OnShowAlbumView(Spotify::Album*)));
    connect(m_pQueueView,SIGNAL(showArtistView(Spotify::Artist*)),SLOT(OnShowArtistView(Spotify::Artist*)));
    connect(m_pQueueView,SIGNAL(updateNowPlaying(const QString&,const QString&,const QString&,const QString&)),SLOT(OnScrobbleUpdateNowPlaying(const QString&, const QString&, const QString&, const QString&)));
    connect(m_pQueueView,SIGNAL(scrobbleTrack()),SLOT(OnScrobbleTrack()));


    m_pArtistView = new ArtistView(m_pSession,m_pTrackView);
    connect(m_pArtistView,SIGNAL(showAlbumView(Spotify::Album*)),SLOT(OnShowAlbumView(Spotify::Album*)));

    //init the playlist listview instance
    m_pPlaylistView = new QListView(this);
    m_pPlaylistView->setContextMenuPolicy(Qt::CustomContextMenu);
    m_pPlaylistModel = new QStandardItemModel(this);
    m_pPlaylistDelegate = new ListViewDelegate(this);
    m_pPlaylistDelegate->setSubHeaderFontPointSize(14);
    m_pPlaylistDelegate->setItemTextDisabledColor(QColor(190,190,190)); //set disabled item color (gray)

    m_pPlaylistView->setItemDelegate(m_pPlaylistDelegate); //connect the delegate to view
    m_pPlaylistView->setModel(m_pPlaylistModel); //connect the model to view.

     //setup signal / slot connections etc.

    connect(m_pLoginDlg,SIGNAL(rejected()),SLOT(OnLoginDialogRejected()));
    connect(m_pSettingsDlg,SIGNAL(settingsSaved()),SLOT(OnSettingsSaved()));
    connect(m_pSession,SIGNAL(loggedIn(Spotify::Error)),SLOT(OnLoginResponse(Spotify::Error)));
    connect(m_pSession,SIGNAL(loggedOut()),SLOT(OnLoggedOut()));
    connect(m_pSession,SIGNAL(playlistContainersReady(QList<Spotify::Playlist*>)),this,SLOT(OnPlaylistContainersReady(QList<Spotify::Playlist*>)));
    connect(m_pSession,SIGNAL(playlistAdded(Spotify::Playlist*,qint32)),this,SLOT(OnPlaylistAdded(Spotify::Playlist*,qint32)));
    connect(m_pSession,SIGNAL(playlistRemoved(qint32)),this,SLOT(OnPlaylistRemoved(qint32)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(searchComplete(Spotify::Search*)),SLOT(OnSearchComplete(Spotify::Search*)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(playTokenLost()),SLOT(OnPlaybackTokenLost()),Qt::QueuedConnection);
    connect(m_pSession,SIGNAL(offlineSyncStatusUpdated(bool, qint32)),SLOT(OnOfflineSyncStatusUpdated(bool, qint32)));
    connect(m_pSession,SIGNAL(offlineSyncError(const QString&)),SLOT(OnOfflineSyncError(const QString&)));
    connect(m_pSession,SIGNAL(offlineModeStateChanged(bool)),SLOT(OnOfflineModeStateChanged(bool)));
    connect(m_pPlaylistView,SIGNAL(activated(QModelIndex)),SLOT(OnPlaylistActivated(QModelIndex)));
    connect(m_pPlaylistView,SIGNAL(customContextMenuRequested(QPoint)),SLOT(OnContextMenuRequested(QPoint)));
    connect(menuBar(),SIGNAL(triggered(QAction*)),SLOT(OnMainMenuAction(QAction*)));
    connect(this,SIGNAL(showLogin()),SLOT(OnShowLogin()),Qt::QueuedConnection);
    connect(&m_NowPlayingIcon,SIGNAL(clicked()),SLOT(OnNowPlaying()));
    connect(&m_NowPlayingInfo,SIGNAL(clicked()), SLOT(OnNowPlaying()));
    connect(&m_NowPlayingPositionInfo,SIGNAL(clicked()), SLOT(OnNowPlaying()));
    connect(m_pSpectrum,SIGNAL(clicked()),SLOT(OnNowPlaying()));
    connect(&m_SearchBtn,SIGNAL(clicked()),SLOT(OnSearch()));
    connect(&m_SearchInput,SIGNAL(returnPressed()),SLOT(OnSearch()));
    connect(QApplication::desktop(), SIGNAL(resized(int)),SLOT(OnOrientationChanged()));

    m_SearchInput.setInputMethodHints(Qt::ImhNoAutoUppercase); //no auto-uppercase mode
    m_SearchInput.setDisabled(true);

    m_SearchBtn.setStyleSheet("QPushButton {min-width: 78px; min-height: 48px; max-width: 78px; max-height: 48px;border-style: none; background-image: url(/opt/qspot/images/icon-search.png);}"
                              "QPushButton:pressed {background-image: url(/opt/qspot/images/icon-search-on.png);}");
    m_SearchBtn.setDisabled(true);

    m_pPlaylistModel->clear();

    QPixmap iconMap;
    iconMap.load("/opt/qspot/images/icon-nowplaying.png"); //load icon from file

    m_NowPlayingIcon.setPixmap(iconMap); //set now playing icon
    m_NowPlayingIcon.setFixedSize(24,24);
    m_NowPlayingInfo.setAlignment(Qt::AlignLeft);

    m_NowPlayingFont.setFamily(QApplication::font().family());
    m_NowPlayingFont.setPointSize(14);
    m_NowPlayingInfo.setFont(m_NowPlayingFont);
    m_NowPlayingInfo.setFixedWidth(510);
    m_NowPlayingInfo.setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    m_NowPlayingInfo.setText("----"); //default (empty) value for now playing info
    m_NowPlayingPositionInfo.setFont(m_NowPlayingFont);
    m_NowPlayingPositionInfo.setFixedWidth(132);
    m_NowPlayingPositionInfo.setAlignment(Qt::AlignLeft|Qt::AlignVCenter);

    m_GridLayout.addWidget(&m_SearchInput,0,0,1,3);
    m_GridLayout.addWidget(&m_SearchBtn,0,3);
    m_GridLayout.addWidget(m_pSyncProgress,1,0,1,-1);
    m_GridLayout.addWidget(m_pPlaylistView,2,0,1,-1);
    m_GridLayout.addWidget(&m_NowPlayingIcon,3,0);
    m_GridLayout.addWidget(&m_NowPlayingInfo,3,1,1,1,Qt::AlignLeft);
    m_GridLayout.addWidget(&m_NowPlayingPositionInfo,3,2,1,1,Qt::AlignRight);
    m_GridLayout.addWidget(m_pSpectrum,3,3);

    m_MainLayout.setLayout(&m_GridLayout);
    setCentralWidget(&m_MainLayout);

    setDisplayOrientation(m_pAppSettings->value("QSpot/displayOrientation",ORIENTATION_DEFAULT).toInt()); //set initial display orientation for views / dialogs

    m_pSession->setCachePath(m_pAppSettings->value("QSpot/cacheLocation", CACHE_PATH_EMMC + "/" + CACHE_DIR_NAME_DEFAULT).toString()); //set libspotify cache path
    m_pSession->setSettingsPath(m_pAppSettings->value("QSpot/cacheLocation", CACHE_PATH_EMMC + "/" + CACHE_DIR_NAME_DEFAULT).toString()); //set libspotify settings path
    m_pSession->init(); //initialise new libspotify session prior to login

    QTimer::singleShot(1500,this,SLOT(OnHandleLoginType())); //proceed to login process (timer event delays process so that any initialisation has time to finish)
}

void MainWindow::OnSpotifyLogin(const QString& strUserID, const QString& strPassword, bool saveCredentials)
{
    //Sends of actual login request to Spotify service backend
    Spotify::Error err;

    if(strUserID!="")
    {//use credentials supplied via login dialog (manual login). Stores user credentials in case saveCredentials set to true.
        //reset possible credentials in settings file (not used)
        m_pSession->login(strUserID,strPassword,saveCredentials);
    }
    else
    { //try to use remembered credentials (automatic login)
       err =  m_pSession->relogin();
       if(err.isError())
       {
           qDebug() << "Relogin failed (no stored credentials)";
           m_pLoginDlg->resetState();
           emit showLogin(); //display login dialog
       }
    }
}

void MainWindow::OnHandleLoginType()
{
   //Slot for detecting / handling automatic / manual login to Spotify. Performs automatic-login in case user has specified that credentials should be saved;
   //otherwise the login dialog is displayed so the user can provide login credentials.

    if(!m_pSession->getRememberedUser().isNull())
    {
        //perform automatic login in case user has stored credentials
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this,tr("Logging in to Spotify"));
#else
        Banner::information(this,tr("Logging in to Spotify"));
#endif
#ifdef Q_WS_MAEMO_5
         setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
#endif
        OnSpotifyLogin("","",true);
    }
    else
    {
        emit showLogin(); //emit signal requesting to show login dialog (no stored login credentials)
    }
}

void MainWindow::setupUILang()
{
    //setup the UI language (localisation)
    m_appTranslator.load("/opt/qspot/translations/qspot_" + m_pAppSettings->value("QSpot/language",LANG_DEFAULT).toString());
    qApp->installTranslator(&m_appTranslator);
}

void MainWindow::showAboutDialog()
{
    //display the application about dialog

    QString strOfflineTimeLeftFormatted = "----";
    QTextStream out(&strOfflineTimeLeftFormatted);
    qint32 daysLeft = 0;
    qint32 hoursLeft = 0;
    qint32 minsLeft  = 0;

    if(m_pSession)
    {    //retrieve time left until user is required to go online (for display)
        if(m_pSession->getOfflineTimeLeft(daysLeft,hoursLeft,minsLeft))
        {
            strOfflineTimeLeftFormatted = "";
            out << daysLeft << " " << tr("days") << " "  << hoursLeft << " " << tr("hours") << " " << minsLeft << " " << tr("minutes");
        }
    }

    m_AboutDlg.setWindowTitle(tr("About QSpot") + " " + QString(APP_VERSION));
    m_AboutDlg.setTopIcon("/usr/share/icons/hicolor/64x64/apps/qspot.png");
    m_AboutDlg.setBottomIcon("/opt/qspot/images/spotify-core-logo.png"); //powered by Spotify(R) Core
    m_AboutDlg.setText(tr("An open-source Qt-based Spotify client for N900/Maemo5.\n"
                        "(C) 2011 Andreas Enbacka and The QSpot Team.\n\n"
                        "Special thanks to Markus for assistance with the graphical design and layout, "
                        "and to Tomi, Florian, Benno and Nicolas for the translations.\n\n") +
                        tr("Go online within") + " " + strOfflineTimeLeftFormatted);


    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry
    if (screenGeometry.width() > screenGeometry.height())
    {//landscape mode
        m_AboutDlg.setFixedSize(APP_WIDTH,APP_HEIGHT); //resize dialog to fixed size
    }
    else
    {
        m_AboutDlg.setFixedSize(APP_HEIGHT,APP_WIDTH); //resize dialog to fixed size
    }

    m_AboutDlg.exec();
}

MainWindow::~MainWindow()
{
#ifdef Q_WS_MAEMO_5
    if(m_pDBusWrapper)
    {
        m_pDBusWrapper->disableHWZoomKeys(); //release HW keypad (zoomkeys)
    }
#endif
}

void MainWindow::OnDisplayFMTransmitterDialog()
{
    //request to display the standard FMTransmitter configuration dialog (control panel plugin);

#ifdef Q_WS_MAEMO_5
    if(m_pOssoWrapper)
    {
        m_pOssoWrapper->executePlugin("libcpfmtx.so");
    }
#endif
}

void MainWindow::OnSettingsSaved()
{
    //called when app settings have been saved (modified)  
    updateSettings();
}

void MainWindow::updateSettings()
{
   //update / set application-related settings

   sp_bitrate selectedBitrate;
   qint32 prefBitrateIdx;

    //buffering related settings
    m_pSession->setMinBufferSize(m_pAppSettings->value("QSpot/minBufferLen",MIN_BUFFER_LEN_DEFAULT).toInt()); //update min. buffer length (default to 30 secs)

    if(m_pAppSettings->value("QSpot/preBufferingActive",PRE_BUFFERING_DEFAULT).toInt())
    {
        m_pSession->setPreBufferingActive(true);
    }
    else
    {
        m_pSession->setPreBufferingActive(false);
    }

    //display orientation setting
    setDisplayOrientation(m_pAppSettings->value("QSpot/displayOrientation",ORIENTATION_DEFAULT).toInt());

    //headset related settings
    if(m_pAppSettings->value("QSpot/pauseOnHsetUnplugged",PAUSE_ON_HEADSET_UNPLUG_DEFAULT).toInt())
        m_pQueueView->setPauseOnHSUnplugged(true);
    else
        m_pQueueView->setPauseOnHSUnplugged(false);

    //scrobbling related settings
    m_LastfmScrobblingEnabled = m_pAppSettings->value("QSpot/enableScrobbling",ENABLE_SCROBBLING_DEFAULT).toBool();
    if(m_LastfmScrobblingEnabled)
    {
        //scrobbling enabled, send authentication request in order to retrieve a Last.fm session key
        m_pLastFMScrobbler->setCache(true); //enable caching of scrobbles
        m_pLastFMScrobbler->authenticateSession(m_pAppSettings->value("QSpot/lastfmUid","").toString(),m_pAppSettings->value("QSpot/lastfmPwd","").toString());
    }

    //hide coverart state
    if(m_pAppSettings->value("QSpot/HideCoverart",HIDE_COVERART_DEFAULT).toInt())
    {
        m_pQueueView->setHideCoverart(true);
    }
    else
    {
         m_pQueueView->setHideCoverart(false);
    }

    //set preferred streaming bitrate
    prefBitrateIdx = m_pAppSettings->value("QSpot/PrefBitrate",IDX_BITRATE_160k).toInt();
    if(prefBitrateIdx==IDX_BITRATE_96k)
        selectedBitrate = SP_BITRATE_96k;
    else if(prefBitrateIdx==IDX_BITRATE_320k)
        selectedBitrate = SP_BITRATE_320k;
    else //default bitrate
        selectedBitrate = SP_BITRATE_160k;
    m_pSession->setPreferredBitrate(selectedBitrate);

    //set preferred offline sync. bitrate
    prefBitrateIdx = m_pAppSettings->value("QSpot/PrefSyncBitrate",IDX_BITRATE_160k).toInt();
    if(prefBitrateIdx==IDX_BITRATE_96k)
        selectedBitrate = SP_BITRATE_96k;
    else if(prefBitrateIdx==IDX_BITRATE_320k)
        selectedBitrate = SP_BITRATE_320k;
    else //default bitrate
        selectedBitrate = SP_BITRATE_160k;
    m_pSession->setPreferredSyncBitrate(selectedBitrate);

    //set whether offline sync. over mobile (e.g., 3G) connection should be allowed or not
    if(m_pAppSettings->value("QSpot/allowMobileSync",ALLOW_MOBILE_SYNC_DEFAULT).toInt())
    {
        m_pSession->setAllowMobileSync(true);
    }
    else
    {
         m_pSession->setAllowMobileSync(false);
    }


    //check whether offline mode status should be changed
    qint32 offlineMode = m_pAppSettings->value("QSpot/OfflineMode",FORCE_OFFLINE_MODE_OFF).toInt();
    if(offlineMode && !m_pSession->isOfflineMode())
    { //force offline mode
       m_pSession->setOfflineMode(true,true);
       m_pOnlineStateAction->setChecked(true); //set UI state
    }
}

void MainWindow::OnShowLogin()
{
    m_pLoginDlg->showMaximized(); //show the login dialog
}

void MainWindow::OnPlayingState(bool state)
{
    //playback state changed notification from session instance
    if(state)
    { //playing state
        m_playing = true;
        if(m_update)
            m_pSpectrum->Start();
    }
    else
    { //paused state
        m_pSpectrum->Stop();
        m_playing = false;
    }
}

void MainWindow::OnPlaybackPositionUpdated(qint64 pos)
{
        //set current playback position, and update the track progress part of the now playing infobar

        QString strCurrPlaybackPos, strCurrTrackDuration, strCurrTrackProgressInfo;

        m_currPlaybackPos = pos;

        QTextStream outStream(&strCurrTrackProgressInfo);

        int seconds = ( pos / 1000);
        QTime currTrackPos = QTime( 0, 0, 0, 0 );
        currTrackPos = currTrackPos.addSecs( seconds );

        Spotify::Track* currTrack = m_PlayQueue.at(m_currQueuePos);

        if(currTrack && m_update)
        {

            //set track progress info in format current playback position / total track duration
            if(currTrackPos.hour()>0)
                strCurrPlaybackPos = currTrackPos.toString("h:mm:ss"); //position exceeds 60 mins
            else
                strCurrPlaybackPos = currTrackPos.toString("m:ss");
            strCurrTrackDuration = currTrack->getDurationAsString();
            outStream << "(" << strCurrPlaybackPos << "/" << strCurrTrackDuration << ")";
            m_NowPlayingPositionInfo.setText(strCurrTrackProgressInfo);
        }
}

void MainWindow::keyPressEvent(QKeyEvent *evt)
{
    //overriden keyPress event handler

    if(evt->key()==Qt::Key_Right)
    {
        evt->accept();
        OnNowPlaying(); //display Now Playing window
    }
    else
    {
        QMainWindow::keyPressEvent(evt);
    }
}

void MainWindow::OnNowPlaying()
{
    //show the current playqueue view (now playing)

    if(m_PlayQueue.size()>0) //some tracks in playqueue
    {
        if(m_pQueueView)
        {
            if(m_pQueueView->parent()!=this)
                m_pQueueView->setParent(this,m_pQueueView->windowFlags()); //re-parent the QueueView instance (to this instance)
            m_pQueueView->show();
        }
    }
}

void MainWindow::OnSetPlayQueue(const QList<Spotify::Track*>& queue, qint32 currQueuePos)
{
    //set the current (active) playqueue, and associated playback position within the queue.
    m_PlayQueue = queue;
    m_currQueuePos = currQueuePos;
    if(sender()==m_pTrackView)
    { //queue set from track view
        if(m_pTrackView->getActivePlaylist())
        {
            if(m_pTrackView->getActivePlaylist()->getOfflineStatus()==Spotify::Available) //playlist is available offline
                m_pQueueView->setQueueOffline(true); //queued tracks (playlist) available offline
            else
                m_pQueueView->setQueueOffline(false); //queued tracks (playlist) not available offline
        }
        else
        {
            m_pQueueView->setQueueOffline(false); //queued tracks (e.g., search results) not available offline
        }
    }
    else //queue set from e.g., album view
    {
        m_pQueueView->setQueueOffline(false);
    }
    m_pQueueView->setQueueInitialState(m_PlayQueue, currQueuePos);
    if(m_pArtistView->isVisible())
        m_pArtistView->hide();
    m_pQueueView->show();
}

void MainWindow::OnShowAlbumView(Spotify::Album* pAlbum)
{
    //show the Album View for the specified album
    if(sender()==m_pTrackView && m_pQueueView->parent()!=m_pTrackView)
        m_pQueueView->setParent(m_pTrackView,m_pQueueView->windowFlags()); //re-parent the QueueView instance (to TrackView instance)
    m_pAlbumView->showMaximized();
    m_pAlbumView->setAlbum(pAlbum);
}

void MainWindow::OnShowArtistView(Spotify::Artist* pArtist)
{
    //show the Artist View for the specified artist
    m_pArtistView->showMaximized();
    m_pArtistView->setArtist(pArtist);
}

void MainWindow::OnPlayQueuePosUpdated(qint32 pos)
{
    if(pos>=0)
    {
        m_currQueuePos = pos;

        Spotify::Track* currTrack = m_PlayQueue.at(m_currQueuePos);

        if(currTrack)
        {
            m_strNowPlayingInfo = currTrack->getName();
            if(currTrack->getArtist())
            {
                m_strNowPlayingInfo += ", ";
                m_strNowPlayingInfo += currTrack->getArtist()->getName();
            }
            m_NowPlayingInfo.setText(m_strNowPlayingInfo);
        }
    }
    else //empty now playing infobar (when playqueue has reached end)
    {
        m_NowPlayingInfo.setText("----");
        m_NowPlayingPositionInfo.setText("");
    }
}

void MainWindow::logout()
{//logout the curent user and display the login dialog
    m_pSession->forgetCurrentUser();
    m_pSession->logout(); //logout current user
}

void MainWindow::closeEvent(QCloseEvent *)
{//event received upon application shutdown
    m_pSession->logout(); //logout current user
}

void MainWindow::OnLoginDialogRejected()
{//request to reject the login dialog, exit app
    qApp->quit();
}

void MainWindow::OnLoginResponse(Spotify::Error error)
{
    //Slot invoked when response from login request received (from Spotify library)
    if(error.isError()) //some error occured during login
    {
#ifdef Q_WS_MAEMO_5
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide main window progress indicator
        QMaemo5InformationBox::information(this,tr("Error:") + error.toString());
#else
        Banner::information(this,tr("Error:") + error.toString());
#endif
        m_pLoginDlg->resetState(); //re-enable login button etc.
        emit showLogin(); //display login dialog in case auto-login is enabled and has failed
    }
    else
    { //successful login to Spotify service
#ifdef Q_WS_MAEMO_5
        m_pLoginDlg->setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display main window progress indicator
#endif
        m_pLogoutAction->setText(tr("Log Out") + " " + m_pSession->getCurrentUser()); //setup the Log Out menubar entry
        m_pLoginDlg->hide();
        m_SearchInput.setFocus(); //set default focus to search field
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this,tr("Loading playlists")); //display loading notification
#else
        Banner::information(this,tr("Loading playlists")); //display loading notification
#endif
    }
}

 void MainWindow::OnPlaylistAdded(Spotify::Playlist* playlist, qint32 position)
 {
     //a playlist has been added
     if(playlist)
     {   //connect updated signal
         connect(playlist,SIGNAL(updated()),this,SLOT(OnPlaylistUpdated()));
         m_Playlists.insert(position,playlist); //insert into corresponding pos in list
         updatePlaylistModel(); //update playlists model
     }
 }

  void MainWindow::OnPlaylistRemoved(qint32 position)
  {
      //a playlist has been removed

      if(position<m_Playlists.size())
      {
          Spotify::Playlist* playlist = m_Playlists.takeAt(position);
          if(playlist)
          {
              disconnect(playlist,SIGNAL(updated()),this,SLOT(OnPlaylistUpdated())); //disconnect signal / slot
              updatePlaylistModel(); //update playlists model
          }
      }
  }

void MainWindow::OnPlaylistUpdated()
{
    //a playlist (e.g., metadata) has been updated
    qDebug() << "Playlist updated..";

    Spotify::Playlist* pPlaylist = (Spotify::Playlist*)sender(); //sender playlist instance
    if(pPlaylist)
        pPlaylist->load(); //re-load metadata

    updatePlaylistModel(); //force model update (for changes to be e.g., reflected in playlists view)
}

void MainWindow::updatePlaylistModel()
{
    //update the listmodel from underlying playlist instances

    QPixmap iconMap;
    iconMap.load("/opt/qspot/images/icon-type-list.png"); //load playlist type icon from file
    QIcon listTypeIcon(iconMap);
    iconMap.load("/opt/qspot/images/icon-type-list-offline.png"); //load playlist type (offline) icon from file
    QIcon listTypeOfflineIcon(iconMap);

    m_pPlaylistModel->clear();

    for(qint32 pos=0; pos < m_Playlists.size(); pos++)
    {
        Spotify::Playlist* pl = m_Playlists.at(pos);
        if(pl->getOfflineStatus() != Spotify::Available && m_pSession->isOfflineMode())
        {//we are currently in offline mode, and the playlist has not been synced for offline usage
            continue; //skip list (do not display list in the listview)
        }
        else
        {
            QStandardItem *item = new QStandardItem();
            item->setEditable(false);
            if(pl->getOfflineStatus()!=Spotify::Available) //playlist not available offline
                item->setData(listTypeIcon,ListViewDelegate::iconRole);
            else
                item->setData(listTypeOfflineIcon,ListViewDelegate::iconRole);  //offline playlist
            item->setData(pl->getName(),ListViewDelegate::headerRole);
            item->setData(QString::number(pl->getNumTracks()) + " " + tr("tracks"),ListViewDelegate::subHeaderLeftJustifyRole);
            item->setData(pos,ListViewDelegate::customDataRole); //store associated position in playlist instance list
            m_pPlaylistModel->appendRow(item);
        }
    }
}

void MainWindow::OnPlaylistContainersReady(QList<Spotify::Playlist*> playlists)
{
   //add items to playlists list widget (all user's playlists have successfully loaded)

   m_Playlists = playlists; //store list of playlists
   for(qint32 pos=0; pos < m_Playlists.size(); pos++)
   {
       //setup signal / slot connections (for notification of playlist metadata / state updated)
       Spotify::Playlist* pl = m_Playlists.at(pos);
       connect(pl,SIGNAL(updated()),this,SLOT(OnPlaylistUpdated()));
   }
   updatePlaylistModel(); //update listview data model
   setupInitialState(); //setup initial state

#ifdef Q_WS_MAEMO_5
   setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
#endif
}

void MainWindow::setupInitialState()
{
    //set up the app initial state (audio etc.)

    m_pSession->initAudio(); //init audio interface

    if(!m_pSession->isOfflineMode())
    {
        m_SearchInput.setDisabled(false);
        m_SearchBtn.setDisabled(false);
    }
    m_SearchInput.setFocus(); //set initial (default) focus to the Search input field

    updateSettings(); //set application-related settings
}

void MainWindow::setDisplayOrientation(qint32 orientation)
{
    //set display orientation (e.g., landscape / portrait / automatic mode)

#ifdef Q_WS_MAEMO_5
    Qt::WidgetAttribute dispOrientation = Qt::WA_Maemo5LandscapeOrientation;

    if(orientation==LANDSCAPE_ORIENTATION)
        dispOrientation = Qt::WA_Maemo5LandscapeOrientation;
    else if(orientation==PORTRAIT_ORIENTATION)
        dispOrientation = Qt::WA_Maemo5PortraitOrientation;
    else if(orientation==AUTO_ORIENTATION)
        dispOrientation = Qt::WA_Maemo5AutoOrientation;

    //set display orientation for views / dialogs
    setAttribute(dispOrientation);
    if(m_pLoginDlg)
        m_pLoginDlg->setAttribute(dispOrientation);
    if(m_pTrackView)
        m_pTrackView->setAttribute(dispOrientation);
    if(m_pQueueView)
        m_pQueueView->setAttribute(dispOrientation);
    if(m_pSettingsDlg)
        m_pSettingsDlg->setAttribute(dispOrientation);
    if(m_pAlbumView)
    {
        m_pAlbumView->setAttribute(dispOrientation);
        m_pAlbumView->setReviewDlgOrientation(dispOrientation);
    }
    if(m_pArtistView)
    {
        m_pArtistView->setAttribute(dispOrientation);
        m_pArtistView->setBiographyDlgOrientation(dispOrientation);
    }
    m_AboutDlg.setAttribute(dispOrientation);

#endif
}

void MainWindow::OnPlaylistActivated(QModelIndex index)
{
	//Display the trackview for the activated playlist

        int pos = index.data(ListViewDelegate::customDataRole).toInt();

        if(pos!=-1) //-1 in case offline mode is active, and the particular list has not been synced offline.
        {
            Spotify::Playlist* pPlaylist = m_Playlists.at(pos);

            if(m_pTrackView)
            {
                if(m_pQueueView)
                {
                    if(m_pQueueView->parent()!=m_pTrackView)
                        m_pQueueView->setParent(m_pTrackView,m_pQueueView->windowFlags());//re-parent QueueView instance (to TrackView)
                }
                m_pTrackView->setPlaylist(pPlaylist); //set active playlist for track view
                m_pTrackView->show();
            }
        }
}

void MainWindow::OnMainMenuAction(QAction* action)
{
  //User has selected an item from the app main menu

    if(action->data().toInt()==MENU_ACTION_SETTINGS)
    { //display the app Settings dialog
        //display the Settings dialog
        m_pSettingsDlg->readSettings();
        m_pSettingsDlg->showMaximized();
    }
    else if(action->data().toInt()==MENU_ACTION_FM_TRANSMITTER)
    { //display the FM transmitter control panel dialog (plugin)
        OnDisplayFMTransmitterDialog();
    }
    else if(action->data().toInt()==MENU_ACTION_ONLINE_HELP)
    { //open / display online help contenttp://qspot.garage.maemo.org/qspot_hel
        m_pHelpViewer->setContent("htp");
        m_pHelpViewer->showMaximized();
    }
    else if(action->data().toInt()==MENU_ACTION_ABOUT)
    { //display the app About dialog
        showAboutDialog();
    }
    else if(action->data().toInt()==MENU_ACTION_LOGOUT)
    { //log out the current user (and remove any saved credentials)
        logout();
    }
}


void MainWindow::OnSearch()
{
    //Search request

    Spotify::Error error;
    int maxTracks, maxAlbums, maxArtists;

    if(m_pAppSettings)
    {
        maxTracks = m_pAppSettings->value("QSpot/maxSearchTracks",MAX_TRACKS_DEFAULT).toInt();
        maxAlbums = m_pAppSettings->value("QSpot/maxSearchAlbums",MAX_ALBUMS_DEFAULT).toInt();
        maxArtists = m_pAppSettings->value("QSpot/maxSearchArtists",MAX_ARTISTS_DEFAULT).toInt();

        if(m_SearchInput.text().trimmed()!="")
        {
#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(this,tr("Searching"));
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
#else
            Banner::information(this,tr("Searching"));
#endif
            error = m_pSession->search(m_SearchInput.text(),0,maxTracks,0,maxAlbums,0,maxArtists,NULL);
            if(error.isError())
            { //some error occured
#ifdef Q_WS_MAEMO_5
                QMaemo5InformationBox::information(this,tr("Error: ") + error.toString());
                setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
#else
                Banner::information(this,tr("Error: ") + error.toString());
#endif
            }
        }
    }}

void MainWindow::OnSearchComplete(Spotify::Search* search)
{
    //Search request completed

    if(search)
    {
        Spotify::Error error = sp_search_error(search->getSearch()); //TODO: move into Search class
        if(error.isError())
        { //some error has occured
#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(this,tr("Error: ") + error.toString());
#else
            Banner::information(this,tr("Error: ") + error.toString());
#endif
        }
        else
        {
            if(search->getTracks().size()>0)
            {//some track / tracks returned by search
                if(m_pTrackView)
                {
                    if(m_pQueueView)
                    {
                        if(m_pQueueView->parent()!=m_pTrackView)
                            m_pQueueView->setParent(m_pTrackView,m_pQueueView->windowFlags()); //re-parent QueueView (to TrackView)
                    }
                    m_pTrackView->setSearchResults(search);
                    m_pTrackView->show();
                }
            }
            else
            {
#ifdef Q_WS_MAEMO_5
                QMaemo5InformationBox::information(this,tr("No tracks found!"));
#else
                Banner::information(this,tr("No tracks found!"));
#endif
            }
        }
    }

    m_SearchInput.setText(""); //clear search field
    m_SearchInput.setCursorPosition(0);
    m_SearchInput.setFocus(); //set focus to search field
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
#endif
}

void MainWindow::OnPlaybackTokenLost()
{
    //playback token has been lost (e.g., same account has logged in from somewhere else); noitify user
#ifdef Q_WS_MAEMO_5
     QMaemo5InformationBox::information(this,tr("Playback paused due to the same account being used somewhere else!"));
#else
    Banner::information(this,tr("Playback paused due to the same account being used somewhere else!"));
#endif

}

void MainWindow::OnScrobblingStatus(bool status, const QString& error)
{
    //status received from the Last.fm service

    if(!status)
    {
        //error occured
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this,tr("Last.fm error: ") + error);
#else
         Banner::information(this,tr("Last.fm error: ") + error);
#endif

    }
}

void MainWindow::OnScrobbleUpdateNowPlaying(const QString& strTrackName, const QString& strArtistName, const QString& strAlbumName, const QString& strAlbumArtist)
{
    //update Last.fm now playing info
    if(m_LastfmScrobblingEnabled) //scrobbling enabled
        m_pLastFMScrobbler->updateNowPlaying(strTrackName,strArtistName,strAlbumName,strAlbumArtist);
}

void MainWindow::OnScrobbleTrack()
{
    //scrobble current track to Last.fm (and send current queue of scrobbles to Last.fm)
     if(m_LastfmScrobblingEnabled) //scrobbling enabled
        m_pLastFMScrobbler->scrobbleTrack();
}

void MainWindow::OnOrientationChanged()
{
    //device orientation changed (e.g, screen rotate)

    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry
    if (screenGeometry.width() > screenGeometry.height())
    {
        //switch to landscape mode
        m_currOrientation = LANDSCAPE_ORIENTATION;
        m_NowPlayingInfo.setFixedWidth(510);
        m_AboutDlg.setFixedSize(APP_WIDTH,APP_HEIGHT); //resize dialog to fixed size
    }
    else
    {
        //switch to portrait mode
        m_currOrientation = PORTRAIT_ORIENTATION;
        m_NowPlayingInfo.setFixedWidth(180);
        m_AboutDlg.setFixedSize(APP_HEIGHT,APP_WIDTH); //resize dialog to fixed size
    }
}

void MainWindow::OnDisplayStateChanged(bool state)
{
    //Display state changed notification (true=ON, false=OFF)

    if(state)
    {
        m_update = true; //allow UI updates
        //enable spectrum analyzer etc.
        if(m_playing)
            m_pSpectrum->Start();
    }
    else
    {
        m_update = false; //no UI updates
        //disable spectrum analyzer etc.
        m_pSpectrum->Stop();
#ifdef Q_WS_MAEMO_5
        m_currSysVolume = m_pDBusWrapper->getVolume(); //store the current system volume
#endif
    }
}

#ifdef Q_WS_MAEMO_5
void MainWindow::OnHWZoomKeyPressed(quint32 keycode)
{
    //notification that HW zoom key has been pressed

    if(m_update) //screen unlocked, return directly
        return;

    if(((keycode == MM_KEY_DOWN) && m_currOrientation == LANDSCAPE_ORIENTATION)||
       ((keycode == MM_KEY_UP) && m_currOrientation == PORTRAIT_ORIENTATION))
    {
        if(m_currSysVolume<100)
            m_pDBusWrapper->setVolume(m_currSysVolume++); //set new system volume value
    }
    else if(((keycode == MM_KEY_UP) && m_currOrientation == LANDSCAPE_ORIENTATION)|| //landscape
            ((keycode == MM_KEY_DOWN) && m_currOrientation == PORTRAIT_ORIENTATION)) //portrait
    {
        if(m_currSysVolume>0)
            m_pDBusWrapper->setVolume(m_currSysVolume--); //set new system volume value
    }
}
#endif

void MainWindow::OnLoggedOut()
{
    //logged out notification for current user
    close(); //close down application upon an explicit logout request.
}

void MainWindow::OnContextMenuRequested(QPoint pnt)
{
    //display context menu for marking / un-marking playlist for offline mode
    QMenu ctxMenu;
    QString strActionTxt;

    if(m_pPlaylistView->currentIndex().row()>=0)
    {
        Spotify::Playlist* pPlaylist = m_Playlists.at(m_pPlaylistView->currentIndex().row());
        if(pPlaylist)
        {
            if(pPlaylist->getOfflineStatus()!=Spotify::Available)
            {
                strActionTxt = tr("Set available offline");
            }
            else
            {
                strActionTxt = tr("Unset available offline");
            }
            ctxMenu.addAction(strActionTxt);
            if(pPlaylist->getOfflineStatus()==Spotify::Available||!m_pSession->isOfflineMode())
            {//only show the context menu in case we are online, or in case the activated playlist is available offline
                if(ctxMenu.exec(pnt))
                {//set / unset offline mode for playlist depending on its current offline status
                    m_pSession->setPlaylistOfflineMode(pPlaylist,pPlaylist->getOfflineStatus()!=Spotify::Available);
                    if(pPlaylist->getOfflineStatus()!=Spotify::Available)
                        m_pSyncProgress->setVisible(true); //display progressbar
                }
            }
        }
    }
}

void MainWindow::OnOfflineSyncStatusUpdated(bool newSync, qint32 progress)
{
    //invoked when offline sync status is updated for update of operation progress (first param. specifies if new sync. operation; second param. progress in percentage)

    if(newSync)
    {
        //new sync operation started; display user notification and show the progress indicator
        if(!m_pSyncProgress->isVisible())
            m_pSyncProgress->setVisible(true);
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this,tr("Offline synchronisation of playlists in progress"));
#else
        Banner::information(this,tr("Offline synchronisation of playlists in progress"));
#endif
    }

    if(progress==m_pSyncProgress->maximum())
    {//hide progress bar when sync completed
        m_pSyncProgress->setVisible(false);
        m_pSyncProgress->setValue(0);
    }
    else
    { //update progressbar state
        m_pSyncProgress->setValue(progress);
    }
}

void MainWindow::OnOfflineSyncError(const QString& strError)
{
    //Notification about offline sync. error. Display notification to user.
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this,tr("Error: ") + strError);
#else
        Banner::information(this,tr("Error: ") + strError);
#endif
}

void MainWindow::OnOfflineModeStateChanged(bool offline)
{
    //Notification received when current oflline state has changed (e.g., online -> offline).
    //The parameter online indicates new offline state.

    bool forceShowMainView = false; //true in case the control should be force to the main view

    updatePlaylistModel(); //update playlists model state

    //enable / disble search based on new state
    m_SearchInput.setEnabled(!offline);
    m_SearchBtn.setEnabled(!offline);

    if(m_LastfmScrobblingEnabled)
    {
        //Last.fm scrobbling active; set offline mode on / off
        m_pLastFMScrobbler->setOfflineMode(offline);
        if(!offline) //authenticate new session
            m_pLastFMScrobbler->authenticateSession(m_pAppSettings->value("QSpot/lastfmUid","").toString(),m_pAppSettings->value("QSpot/lastfmPwd","").toString());
    }

    m_pOnlineStateAction->setChecked(offline);

    if(offline)
    { //offline mode activated
        if(!m_pQueueView->isQueueOffline())
        { //tracks in current playqueue are not available offline
            if(m_pQueueView->isVisible())
                m_pQueueView->hide();
            m_PlayQueue.clear(); //clear playqueue list
            OnPlayQueuePosUpdated(-1); //clear now playing infobar
            forceShowMainView = true;
        }
        if(m_pTrackView->isVisible())
        {
            if(m_pTrackView->getActivePlaylist())
            {
                if(m_pTrackView->getActivePlaylist()->getOfflineStatus()!=Spotify::Available) //current playlist is not available offline
                {
                    m_pTrackView->hide();
                    forceShowMainView = true;
                }
            }
            else
            {
                m_pTrackView->hide();
                forceShowMainView = true;
            }
        }
        if(forceShowMainView)
            this->show();
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(this,tr("Offline mode is now activated"));
#else
        Banner::information(this,tr("Offline mode is now activated"));
#endif
    }
}

void MainWindow::OnOfflineModeStateToggled(bool checked)
{
    //The user has manually toggled between offline / online states in UI.
    //In case checked parameter is true online -> offline has been toggled; otherwise offline -> online.
    m_pSession->setOfflineMode(checked,true);

    //remember the current user setting
    if(checked)
        m_pAppSettings->setValue("QSpot/OfflineMode",FORCE_OFFLINE_MODE_ON);
    else
        m_pAppSettings->setValue("QSpot/OfflineMode",FORCE_OFFLINE_MODE_OFF);

}
