//ListViewDelegate.cpp: Implementation of the ListViewDelegate class.

#include "ListViewDelegate.h"

ListViewDelegate::ListViewDelegate(QWidget* parent)
    : QStyledItemDelegate(parent)
{
    m_HeaderFontPointSize = -1;
    m_SubHeaderFontPointSize = -1;
    m_TextHighlightColor = QColor(255,255,255);
}

ListViewDelegate::~ListViewDelegate()
{
}

void ListViewDelegate::setHeaderFontPointSize(int size)
{
    m_HeaderFontPointSize = size;
}

void ListViewDelegate::setSubHeaderFontPointSize(int size)
{
    m_SubHeaderFontPointSize = size;
}

void ListViewDelegate::setItemTextHighlightColor(const QColor& color)
{
    m_TextHighlightColor = color;
}

void ListViewDelegate::setItemTextDisabledColor(const QColor& color)
{
   m_TextDisabledColor = color;
}

QSize ListViewDelegate::sizeHint(const QStyleOptionViewItem & option , const QModelIndex & index) const
{
    //handles size allocation of single item in the associated listview
    QIcon icon = qvariant_cast<QIcon>(index.data(iconRole));
    QSize iconsize = icon.actualSize(option.decorationSize);
    QFont font = QApplication::font();
    if(index.data(headerItemSizeRole).isValid())
    {
        font.setPointSize(index.data(headerItemSizeRole).toInt());
    }
    else
    {
        if(m_HeaderFontPointSize!=-1)
            font.setPointSize(m_HeaderFontPointSize);
    }
    QFontMetrics fm(font);

    return(QSize(iconsize.width(),iconsize.height()+fm.height() + 10));
}

void ListViewDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
 {
    //overriden paint method (handles drawing of custom listview item)

    QStyledItemDelegate::paint(painter,option,index); //call base

    painter->save();

    bool isHeaderItem = false;
    QColor headerItemTextColor;


    QFont font = QApplication::font();
    QFont subFont = QApplication::font();
    font.setBold(true);
    subFont.setWeight(subFont.weight()-2);
    if(index.data(headerItemRole).isValid())
    {
        isHeaderItem = index.data(headerItemRole).toBool();
    }

    if(isHeaderItem)
    {
        headerItemTextColor = qvariant_cast<QColor>(index.data(headerItemColorRole));
        font.setPointSize(index.data(headerItemSizeRole).toInt());
    }
    else
    {
        if(m_HeaderFontPointSize!=-1)
            font.setPointSize(m_HeaderFontPointSize);
        if(m_SubHeaderFontPointSize!=-1)
            subFont.setPointSize(m_SubHeaderFontPointSize);
    }


    QFontMetrics fm(font);
    QFontMetrics fmSub(subFont);

    QIcon icon = qvariant_cast<QIcon>(index.data(iconRole));
    QString headerText = index.data(headerRole).toString();
    QString subTextLeftJustify = index.data(subHeaderLeftJustifyRole).toString();
    QString subTextRightJustify = index.data(subHeaderRightJustifyRole).toString();
    bool highlightItemText = index.data(highlightItemTextRole).toBool();

    QSize iconsize = icon.actualSize(option.decorationSize);

    QRect headerRect = option.rect;
    QRect subheaderRect = option.rect;
    QRect iconRect = subheaderRect;

    iconRect.setRight(iconsize.width()+30);
    iconRect.setTop(headerRect.top());
    headerRect.setLeft(iconRect.right());
    subheaderRect.setLeft(iconRect.right());
    headerRect.setTop(headerRect.top()+15);
    headerRect.setBottom(headerRect.top()+fm.height());

    subheaderRect.setTop(headerRect.bottom()+2);

    //draw item icon
    painter->drawPixmap(QPoint(iconRect.left()+(iconsize.width()/2)-15,iconRect.top()+(iconsize.height()/2)-5),icon.pixmap(iconsize.width(),iconsize.height()));

    //draw item header text
    int maxHeaderTextWidth = headerRect.right() - headerRect.left() + 10; //max. allowed width of header text

    if(highlightItemText) //item text should be highlighted (if it is not currently selected)
    {
        if(!(option.state & QStyle::State_Selected))
            painter->setPen(m_TextHighlightColor);
    }
    else
    {
        if(isHeaderItem)
        {
            painter->setPen(headerItemTextColor); //set header item text color
        }
        else if(!(option.state & QStyle::State_Enabled))
        { //draw item disabled in case not enabled
            painter->setPen(m_TextDisabledColor);
        }
    }

    painter->setFont(font);

    painter->drawText(headerRect.left()-10,headerRect.top()+20,fm.elidedText(headerText,Qt::ElideRight,maxHeaderTextWidth));

    if(!isHeaderItem)
    { //process for non-header item type only
        int subTextRightJustifyLen = fmSub.width(subTextRightJustify); //get length of string in pixels

        //calculate the max. allowed width of the left justified subheader text
        int maxLeftJustifyTextWidth = subheaderRect.right()- subTextRightJustifyLen - (subheaderRect.left()+10);
        if(maxLeftJustifyTextWidth>0)
        {  //draw left justified part of subheader text
            painter->setFont(subFont);
            painter->drawText(subheaderRect.left()-10,subheaderRect.top()+17,fmSub.elidedText(subTextLeftJustify,Qt::ElideRight,maxLeftJustifyTextWidth));
        }

        //draw right justified part of subheader text
        painter->drawText(subheaderRect.right()-subTextRightJustifyLen,subheaderRect.top()+17,subTextRightJustify);
    }

    painter->restore();
 }

