#include "spotifysession.h"
#include "link.h"
#include <QDebug>
#include <QList>
#include <QImage>
#include <QPicture>

#include "artist.h"
#include "album.h"
#include "callbacks.h"



/**
 *  Spotify session class. Handles the session functions. Emits signals for whatever
 *  callback the Spotify library uses.
 */
Spotify::Session* Spotify::Session::callingSession;


Spotify::Session::Session( QObject* parent ):
    QObject( parent ),
    m_currentTrack( 0 ),
    m_playlistcontainer( 0 ),
    m_valid( false ),
    m_currentUser( 0 ),
    m_playbackPos( 0 )
{
    // Meta types:
    qRegisterMetaType<Spotify::Error>("Spotify::Error");
    qRegisterMetaType< QList<Spotify::Track*> >("QList<Spotify::Track*>");

    // C callbacks <-> signal/slot hack:
    Spotify::Session::callingSession = this;


    // Set up signal / slot connections:

    connect( this, SIGNAL(onNotifyMainThread()),
             this, SLOT(OnProcessThreadEvents()),Qt::QueuedConnection);

    connect(this, SIGNAL(rootListContainerLoaded(sp_playlistcontainer*)),
            this, SLOT(OnRootListContainerLoaded(sp_playlistcontainer*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onPlaylistAdded(sp_playlist*,int)),
            this, SLOT(OnPlayListAdded(sp_playlist*,int)),Qt::QueuedConnection);
    connect(this, SIGNAL(onPlaylistRenamed(sp_playlist*)),
            this, SLOT(OnPlayListRenamed(sp_playlist*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onTracksAdded(sp_playlist*,sp_track*const*,int,int)),
            this, SLOT(OnTracksAdded(sp_playlist*,sp_track*const*,int,int)),Qt::QueuedConnection);
    connect(this, SIGNAL(playlistMetadataUpdated(sp_playlist*)),
            this, SLOT(OnPlaylistMetadataUpdated(sp_playlist*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onImageLoaded(sp_image*,void*)),
            this, SLOT(OnImageLoaded(sp_image*, void*)),Qt::QueuedConnection);
    connect(this, SIGNAL(metadataUpdated()),
                    this, SLOT(updatePlaylistsMetadataSlot()),Qt::QueuedConnection);
    connect(this, SIGNAL(onMusicDelivery(int, const sp_audioformat*,QByteArray)),
                    this, SLOT(OnMusicDelivery(int, const sp_audioformat*,QByteArray)),Qt::QueuedConnection);
    connect(this, SIGNAL(onSearchComplete(sp_search*)),
                    this, SLOT(OnSearchComplete(sp_search*)),Qt::QueuedConnection);

    connect(this, SIGNAL(onAlbumBrowserReady(sp_albumbrowse*,qint32*)),
                    this, SLOT(OnAlbumBrowserReady(sp_albumbrowse*,qint32*)),Qt::QueuedConnection);
    connect(this, SIGNAL(onArtistBrowseComplete(sp_artistbrowse*)),
                    this, SLOT(OnArtistBrowseComplete(sp_artistbrowse*)),Qt::QueuedConnection);
    m_pAudioThread = NULL;

    m_preBufActive = true;

    m_bPlaylistsLoading = false;
}

void Spotify::Session::init(const sp_bitrate& bitrate)
{
    //Init the current Spotify session.
    //Note: this method NEEDS to be initially called to set up the session, start audio playback thread etc.

    // Set config:
    m_config.api_version = SPOTIFY_API_VERSION;
    m_config.cache_location = "tmp";
    m_config.settings_location = "tmp";
    m_config.application_key = g_appkey;
    m_config.application_key_size = g_appkey_size;
    m_config.user_agent = "QSpot";
    m_config.callbacks = &g_callbacks;

    // Initiate session:
    m_error = sp_session_init( &m_config, &m_session );
    if( m_error.isError() )
    {
        qDebug() << "FATAL ERROR: Could not initiate session";
    }

   sp_session_preferred_bitrate(m_session, bitrate); //set the preferred streaming bitrate

    m_ProcessEvents = false; //event processing not initiated
    m_DisableNotifyThread = false;

   m_pProcEvtTimer = new QTimer(this);
   connect(m_pProcEvtTimer,SIGNAL(timeout()),this,SLOT(OnProcessThreadEvents()));
}

void Spotify::Session::initAudio()
{
    //Init the audio / playback thread (pulseaudio).
    //Note: Must be called before calling any Spotify session player functions.

    m_pAudioThread = new PlaybackThread(this);

    m_pAudioThread->setBufferingPaused(false);
    m_pAudioThread->setPlaybackPaused(false);

    connect(m_pAudioThread, SIGNAL(resumeBuffering()),
                     this, SLOT(OnResumeBuffering()),Qt::QueuedConnection);
    connect(m_pAudioThread, SIGNAL(playbackPositionUpdated(qint64)),
                     this, SLOT(OnPlaybackPositionUpdated(qint64)),Qt::QueuedConnection);
    connect(m_pAudioThread, SIGNAL(playbackFinished()),
                     this, SLOT(OnPlaybackFinished()),Qt::QueuedConnection);
    connect(m_pAudioThread, SIGNAL(playing(bool)),
                     this, SLOT(OnPlayingState(bool)),Qt::QueuedConnection);

    m_pAudioThread->start(); //start audio playback thread
}

bool Spotify::Session::playlistLoadingInProgress()
{
    //returns true in case playlist loading (getPlaylists) is currently in progress
    return m_bPlaylistsLoading;
}

void Spotify::Session::setMinBufferSize(qint32 secs)
{
    //set the (minimum) size of the internal FIFO used to hold audio data
    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setMinBufferSize(secs);
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::setPreBufferingActive(bool active)
{
    //set whether track pre-buffering is active (i.e., buffer next track data in advance)

    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        qDebug() << "Set pre-buffering state: " << active;
        m_preBufActive = active;
        m_pAudioThread->setPreBufActive(active);
        //m_pAudioThread->setNextTrackStartPos(-1);
        m_pAudioThread->getMutex()->unlock();
    }
}

bool Spotify::Session::getPreBufferingActive()
{
    return m_preBufActive;
}

void Spotify::Session::setPlaybackPos(qint64 pos)
{
    m_playbackPos = pos;
}

void Spotify::Session::setBufferingPos(qint64 pos)
{
    m_bufferedAmount = pos;
}


void Spotify::Session::setPlaying(bool playing)
{
    m_playing = playing;
}

void Spotify::Session::setNextTrackStartPos()
{
    //set the next track start position in the audio FIFO
    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setNextTrackStartPos();
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::setNewTrackStarted(bool started)
{
    //indicate start of new track audio buffer data (in playback thread FIFO)
    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setNewTrackStarted(started);
        m_pAudioThread->getMutex()->unlock();
    }
}


void Spotify::Session::OnPlayingState(bool state)
{
    //emit current playback state (playing / paused)
    emit playing(state);
}

void Spotify::Session::OnProcessThreadEvents()
{
    int next_timeout = -1;
    
   do
   {
       if(m_session)
       {
            sp_session_process_events(m_session,&next_timeout);
       }
   }
   while(!next_timeout);

   if(m_ProcessEvents)
   {
       //QTimer::singleShot(next_timeout,this,SLOT(OnProcessThreadEvents()));
       m_pProcEvtTimer->start(next_timeout);
   }
}

Spotify::Error Spotify::Session::login( QString username, QString password )
{
    // Log user in

    m_DisableNotifyThread = false;

    m_error = sp_session_login( m_session, username.toUtf8(), password.toUtf8() );
    if( m_error.isError() )
    {
        qDebug()<<"Not valid";
        m_valid = false;
    }
    else
    {
        m_username = QString(username);
        m_password = QString(password);
    }

    return m_error;
}

void Spotify::Session::logout()
{
    sp_session_logout( m_session );
}

Spotify::Error Spotify::Session::load( Spotify::Track* track )
{
    Spotify::Error error;
    sp_track* sptrack = track->getTrack();
    if( sptrack )
    {
        qDebug() << "+player load";
        error = sp_session_player_load( m_session, sptrack );
        qDebug() << "-player load";
    }
    return error;
}

void Spotify::Session::OnPlaybackFinished()
{
    emit playbackFinished();
}

void Spotify::Session::OnPlaybackPositionUpdated(qint64 duration)
{

    m_playbackPos += duration;

    emit playbackPositionUpdated(m_playbackPos);
}

void Spotify::Session::resetPlaybackMembers()
{
    //public method for resetting the playback related instance members (and removes any pending events from the playback thread)

    if(m_pAudioThread)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->clearFIFO();
        m_pAudioThread->setFIFOSize(0);
        m_pAudioThread->setBufferingPaused(false);
        m_pAudioThread->setPlaybackPaused(false);
        m_pAudioThread->setEndOfTrack(false);
        qApp->removePostedEvents(this); //remove any pending events for this instance (from e.g., the playback thread)
        m_pAudioThread->getMutex()->unlock();
    }
}


Spotify::Error Spotify::Session::play( Spotify::Track* track, bool preBuffer )
{
    m_playing = false;
    m_currentTrack = track;
    Spotify::Error error;
    QMutex mutex;
    QWaitCondition sleep;

    sleep.wait(&mutex,1300);
	
    // Disregard tracks that are not available:
    if( !track->isAvailable() )
    {
        qDebug() << "ERROR: Asked to play back a unavailable track..";
        return Spotify::Error( SP_ERROR_TRACK_NOT_PLAYABLE );
    }


    // Disregard unloaded of unavailable tracks:
    if( !track->isLoaded() )
    {
        qDebug() << "ERROR: Asked to play back an unloaded track..";
        return Spotify::Error( SP_ERROR_RESOURCE_NOT_LOADED );
    }

    // Load track:
    error = load( track );
    if( error.isError() )
    {
        qDebug() << "ERROR: Could not load track: " << error.toString();
        return error;
    }

    if(!preBuffer) //pre-buffering not active (i.e., audio FIFO should be cleared)
    {
        if(m_pAudioThread)
        {
            m_pAudioThread->getMutex()->lock();
            m_pAudioThread->clearFIFO();
            m_pAudioThread->setFIFOSize(0);
            m_pAudioThread->setBufferingPaused(false);
            m_pAudioThread->setPlaybackPaused(false);
            m_pAudioThread->setEndOfTrack(false);
            m_pAudioThread->setNewTrackStarted(false);
            m_pAudioThread->setNextTrackStartPos(-1);
            m_pAudioThread->getMutex()->unlock();
        }
    }

    //sleep.wait(&mutex,300);

    // Play track:
    qDebug() << "+play";
    error = sp_session_player_play( m_session, true );
    qDebug() << "-play";
    if( !error.isError())
    {
        if(!preBuffer) //track pre-buffer not active
        {
            m_playbackPos = 0;
            m_bufferedAmount = 0;
        }
        m_playing = true;
    }
    else
    {
        qDebug() << "ERROR: Could not play track: " << error.toString();
        return error;
    }

    return Spotify::Error( SP_ERROR_OK );
}

Spotify::Error Spotify::Session::seek( int pos )
{
    Spotify::Error error = sp_session_player_seek( m_session, pos );
    if( !error.isError() )
        m_playbackPos = pos;
    return error;
 }

Spotify::Error Spotify::Session::stop()
{
    Spotify::Error error;

    if( m_playing )
    {
        error = sp_session_player_play( m_session, false );
        if( !error.isError() )
        {
           // sleep.wait(&mutex,1500);
            //sp_session_player_unload( m_session );
            //m_playing = false;
        }
    }
    return error;
}

void Spotify::Session::pauseResumePlayback()
{
    //Pause / resume playback from Spotify (e.g., signal to playback thread to stop / continue writing data to PulseAudio)
    m_pAudioThread->getMutex()->lock();
    if(m_pAudioThread->getPlaybackPaused())
       m_pAudioThread->setPlaybackPaused(false);
    else
        m_pAudioThread->setPlaybackPaused(true);
    m_pAudioThread->getMutex()->unlock();
}

Spotify::Error Spotify::Session::pause()
{
    //Pause music delivery from Spotify

    Spotify::Error error;
    // Is playing:

    if( m_playing )
    {
        error = sp_session_player_play( m_session, false );
        if( !error.isError() )
        {
            m_playing = false;
        }
    }
    // Is paused:
    else if( !m_playing && m_currentTrack->getTrack() && sp_track_is_loaded( m_currentTrack->getTrack() ) )
    {
        error = sp_session_player_play( m_session, true );
        if( !error.isError() )
        {
             m_playing = true;
        }
    }
    return error;
}

Spotify::Error Spotify::Session::search( const QString& query, const int trackOffset, const int trackCount,
                                         const int albumOffset, const int albumCount, const int artistOffset,
                                         const int artistCount, void* userdata )
{
    Spotify::Error error;
    sp_search* search = sp_search_create( m_session, query.toUtf8().data(), trackOffset, trackCount,
                                          albumOffset, albumCount, artistOffset, artistCount,
                                          &searchCompleteCallback, userdata );

    Spotify::Search* s = new Spotify::Search( search, this );
    m_searchHash.insert( search, s );
    connect( s, SIGNAL(destroyed(QObject*)),
             this, SLOT(searchDeletedSlot(QObject*)));
    return error;
}

Spotify::Error Spotify::Session::loadImage(const byte* imageID, Spotify::ImageContainer* pContainer)
{
    //request to load image (identified by imageID) from the Spotify backend.

    Spotify::Error error;
    if( imageID && pContainer )
    {
        sp_image* image = sp_image_create( m_session, imageID );
        error = sp_image_error( image );
        if(image)
            sp_image_add_load_callback( image, &Spotify::Session::imageLoadedCallback, (void*)pContainer );
    }
    else
        error = SP_ERROR_RESOURCE_NOT_LOADED;
    return error;
}

void Spotify::Session::getPlaylistContainer()
{
    //Request (root) playlist container

    qDebug()<<"Fetch root playlist container..";

    //set playlistcontainer callbacks

    sp_playlistcontainer_add_callbacks(
            sp_session_playlistcontainer( m_session ),
             &pc_callbacks,
             NULL );

    // Go fetch container
  sp_session_playlistcontainer( m_session );
}


Spotify::Error Spotify::Session::getPlaylists()
{
  //get the playlists

 Spotify::Error error;
 QWaitCondition sleep;
 QMutex mutex;

 m_ProcessEvents = false;
 m_DisableNotifyThread = true;

 m_bPlaylistsLoading = true;

  for (int i=0; i < sp_playlistcontainer_num_playlists(m_playlistcontainer); i++)
  {
    sp_playlist* pl = sp_playlistcontainer_playlist(m_playlistcontainer, i);

   int next_timeout = -1;

   while(!sp_playlist_is_loaded(pl))
    { //process main app event loop while waiting for playlist to load
       if(m_bStopPlaylistLoading)
           break;
       sp_session_process_events(m_session,&next_timeout);
       qApp->processEvents();
       sleep.wait(&mutex,100); //sleep for 100ms just in case that we would have problem with retrieving the last element in the associated list.
    }

   if(m_bStopPlaylistLoading)
       break;

   if(i>=m_playlistList.size())
   {
       qDebug() << "Adding playlist: " << sp_playlist_name(pl);
       Spotify::Playlist* playlist = new Spotify::Playlist(pl,this);
       m_playlistList.insert(i,playlist); //we could add the following element to the list elements in case of unsupported commands
   }
}
  m_bPlaylistsLoading = false; //indicate that playlist loading done (successfully)

  emit playlistContainersReady(m_playlistList);

  m_DisableNotifyThread = false;

  return error;
}

void Spotify::Session::stopPlaylistLoading()
{
    //set flag for requesting playlist loading to be interrupted.
    m_bStopPlaylistLoading = true;
}

Spotify::Error Spotify::Session::getPlaylistTracks(Spotify::Playlist* pPlaylist)
{
	//get / update tracks for the specified playlist
      QWaitCondition sleep;
      QMutex mutex;

      m_ProcessEvents = false;
      m_DisableNotifyThread = true;

      int next_timeout = -1;

      for(int trackPos = 0; trackPos < pPlaylist->getNumTracks(); trackPos++)
      {
        Spotify::Track* track = pPlaylist->getTrack(trackPos);
        if(track)
        { 
         /* while(!sp_track_is_loaded(track->getTrack()))
          { //loop and process events until track is loaded
            qDebug()<< "Attempting to load unloaded track..";
            sp_session_process_events(m_session,&next_timeout);
            qApp->processEvents();
            sleep.wait(&mutex,100); //sleep for 100ms
          }
          */
          track->load();
          if(track->isAvailable() && track->getAlbum())
          {
            //attempt to load coverart for the track album
            qDebug()<<"Requesting coverart..";
            //getAlbumCover(track->getAlbum());
          }
	}
      }

      emit playlistTracksReady(pPlaylist); //notify track data is ready for the playlist

      m_DisableNotifyThread = false;
}

bool Spotify::Session::addPlaylist( QString name )
{
    if(!m_playlistcontainer) return false;
    sp_playlist* p = sp_playlistcontainer_add_new_playlist( m_playlistcontainer, name.toUtf8().data() );
    return p!=0;
}

bool Spotify::Session::addPlaylist( Spotify::Link link )
{
    if(!m_playlistcontainer) return false;
    if( link.type() == Spotify::Link::PLAYLIST )
    {
        sp_link* l = link.getLink();
        if( l )
        {
            sp_playlist* p = sp_playlistcontainer_add_playlist( m_playlistcontainer, l );
            return p!=0;
        }
    }
    return false;
}

bool Spotify::Session::removePlaylist( const int index )
{
    if( !m_playlistcontainer) return false;
    Spotify::Error error = sp_playlistcontainer_remove_playlist( m_playlistcontainer, index );
    return !error.isError();
}

bool Spotify::Session::movePlaylist( const int index, int position )
{
    if( !m_playlistcontainer) return false;
    if( position > index ) position+=1; // libspotify wants from n to n+2 rather than n to n+1.....
    Spotify::Error error = sp_playlistcontainer_move_playlist( m_playlistcontainer, index, position );
    return !error.isError();
}

void Spotify::Session::updatePlaylistsMetadataSlot()
{
    qDebug() << "metadata updated..";

    for( int i = 0; i < m_playlistList.count(); ++i )
    {
        Spotify::Playlist* playlist = m_playlistList.at(i);
        if( playlist )
        {
            playlist->load();
            playlist->updateTracks();
        }
    }

    // Try to load searches:
    QList<Spotify::Search*> searches = m_searchHash.values();
    for( int i = 0; i < searches.count(); ++i )
    {
        if( !searches.at(i)->isLoaded() )
            searches.at(i)->load();
    }

    // Try to load artist catalogues:
    QList<Spotify::ArtistCatalogue*> artistCatalogues = m_artistBrowseHash.values();
    for( int i = 0; i < artistCatalogues.count(); ++i )
    {
        if( !artistCatalogues.at(i)->isLoaded() )
            artistCatalogues.at(i)->load();
    }
}

void Spotify::Session::searchDeletedSlot( QObject* object )
{
    Spotify::Search* search = static_cast< Spotify::Search* >( object );
    if( search )
        m_searchHash.remove( search->getSearch() );
    else
        qDebug() << "WARNING: Trying to delete non-existing search";
}

bool Spotify::Session::browse( Spotify::Artist* artist )
{
    if( artist )
    {
        sp_artist* spartist = artist->getArtist();
        sp_artistbrowse* ab = sp_artistbrowse_create( m_session, spartist,
                                                      &Spotify::Session::artistBrowseCompleteCallback,
                                                      NULL);
        if(!ab)
           return false;
        Spotify::ArtistCatalogue* ac = new Spotify::ArtistCatalogue( ab, this );
        m_artistBrowseHash.insert( ab, ac );
        connect( ac, SIGNAL(destroyed(QObject*)),
                 this, SLOT(artistCatalogueDeletedSlot(QObject*)));
    }
    else
        return false;

    return true;
}

void Spotify::Session::artistCatalogueDeletedSlot( QObject* object )
{
    qDebug() << "An artistcatalogue was deleted";
    Spotify::ArtistCatalogue* ac = static_cast< Spotify::ArtistCatalogue* >( object );
    if( ac )
        m_artistBrowseHash.remove( ac->getArtistBrowse() );
    else
        qDebug() << "WARNING: Trying to delete non-existing artist browser";
}

bool Spotify::Session::browse( Spotify::Album* album, qint32* pCallerID )
{
    if( album )
    {
        sp_album* spalbum = album->getAlbum();
        sp_albumbrowse* ab = sp_albumbrowse_create( m_session, spalbum,
                                                    &Spotify::Session::albumBrowseCompleteCallback,
                                                    (void*)pCallerID);
        if(!ab)
            return false; //failed to create browse album request

        //error = sp_albumbrowse_error( ab );
        Spotify::AlbumBrowser* alb = new Spotify::AlbumBrowser( ab, this );
        m_albumBrowseHash.insert( ab, alb);
        connect( alb, SIGNAL(destroyed(QObject*)),
                 this, SLOT(albumBrowserDeletedSlot(QObject*)));
    }
    else
        return false;

    return true;
}

void Spotify::Session::albumBrowserDeletedSlot( QObject* object )
{
    qDebug() << "An album browser was deleted";
    Spotify::AlbumBrowser* ab = static_cast< Spotify::AlbumBrowser* >( object );
    if( ab )
        m_albumBrowseHash.remove( ab->getAlbumBrowser() );
    else
        qDebug() << "WARNING: Trying to delete non-existing album browser";
}

Spotify::Session::~Session()
{
    m_ProcessEvents = false;
    if(m_currentUser)
        m_currentUser->deleteLater();
    if(m_pAudioThread)
    {
        m_pAudioThread->stop();
        m_pAudioThread->wait(); //wait for thread execution to finish
    }
    qDebug() << "Session deleted";
}


/********************* CALLBACKS *****************/

void Spotify::Session::artistBrowseCompleteCallback( sp_artistbrowse* ab, void* userdata )
{
    Spotify::Session::callingSession->artistBrowseCompleteWrapper( ab, userdata );
}

void Spotify::Session::artistBrowseCompleteWrapper( sp_artistbrowse* ab, void* userdata )
{
    Q_UNUSED(userdata);
    emit onArtistBrowseComplete(ab);
}

void Spotify::Session::OnArtistBrowseComplete( sp_artistbrowse* ab )
{
    if( ab && m_artistBrowseHash.contains( ab ) )
    {
        qDebug() << "An artist was just browsed!";
        Spotify::ArtistCatalogue* ac = m_artistBrowseHash.value( ab );
        emit artistCatalogueReady( ac );
    }
}

void Spotify::Session::albumBrowseCompleteCallback( sp_albumbrowse* ab, void* userdata )
{
    Spotify::Session::callingSession->albumBrowseCompleteWrapper( ab, userdata );
}

void Spotify::Session::albumBrowseCompleteWrapper( sp_albumbrowse* ab, void* userdata )
{
    qint32* pCallerID = (qint32*)userdata;
    emit onAlbumBrowserReady(ab, pCallerID);
}

void Spotify::Session::OnAlbumBrowserReady(sp_albumbrowse *ab, qint32* pCallerID)
{
    qint32 callerID = -1;

    if(pCallerID)
    {
        callerID = *(qint32*)pCallerID;
    }
    if( ab && m_albumBrowseHash.contains( ab ) )
    {
        qDebug() << "Album browse requeste completed!";
        Spotify::AlbumBrowser* alb = m_albumBrowseHash.value( ab );
        emit albumBrowserReady( alb, callerID);
    }
}

void SP_CALLCONV Spotify::Session::playlistAddedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                                     int position, void* userdata )
{
    qDebug()<<"Playlist added..";
    Spotify::Session::callingSession->playlistAddedWrapper( pc, playlist, position, userdata);
}

void Spotify::Session::playlistAddedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist, int position, void* userdata )
{
    Q_UNUSED( pc );
    Q_UNUSED( userdata );
    emit onPlaylistAdded(playlist,position);
}

void Spotify::Session::OnPlayListAdded(sp_playlist* playlist, int position)
{
    //Set playlist callbacks:
   sp_playlist_add_callbacks( playlist, &pl_callbacks, NULL );

   qDebug()<<"Playlist added: " << sp_playlist_name(playlist);

   Spotify::Playlist* pl = new Spotify::Playlist( playlist );
    pl->setParent( this );
   pl->setPosition(position);
   m_playlistHash.insert( playlist, pl );
   m_playlistList.insert( position, pl );
   emit playlistAdded( pl, position ); //signal that playlist intance has been added
   /*if(isPlaylistContainersReady()) //all playlists have been succesfully loaded
   {
       emit playlistContainersReady(m_playlistList);
    }
    */

}


bool Spotify::Session::isPlaylistContainersReady()
{

    /* returns true in case all playlist containers have been loaded */

    if(m_playlistList.size()!=sp_playlistcontainer_num_playlists(m_playlistcontainer))
        return false;

    //loop through playlists list and check if name is available

    for(int pos = 0; pos < m_playlistList.size(); pos++)
    {
        Spotify::Playlist* pl = m_playlistList.at(pos);
        if(!sp_playlist_is_loaded(pl->getPlaylist()))
            return false;
    }

    qDebug()<<"All playlists are ready..";

    return true;
}

void SP_CALLCONV Spotify::Session::playlistRemovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                     int position, void* userdata )
{
    Spotify::Session::callingSession->playlistRemovedWrapper( pc, playlist, position, userdata );
}

void Spotify::Session::playlistRemovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                               int position, void* userdata )
{
    Q_UNUSED( pc );
    Q_UNUSED( userdata );
    qDebug() << "A playlist was removed";

    /*if( m_playlistHash.contains( playlist ) )
    {
        Spotify::Playlist* pl = m_playlistHash.take( playlist );
        if( pl && m_playlistList.contains( pl ) && position < m_playlistList.count() )
        {
            m_playlistList.takeAt( position );
            pl->deleteLater();
            emit playlistRemoved( position );
        }
    }
    else
        qDebug() << "WARNING: Trying to delete non-existing playlist. Sync inconsistency!!!";
        */
}

void SP_CALLCONV Spotify::Session::playlistMovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                              int oldPosition, int newPosition, void* userdata )
{
   // Spotify::Session::callingSession->playlistMovedWrapper( pc, playlist, oldPosition, newPosition, userdata );
}

void Spotify::Session::playlistMovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                             int oldPosition, int newPosition, void* userdata )
{
    Q_UNUSED( pc );
    Q_UNUSED( playlist );
    Q_UNUSED( userdata );
    qDebug() << "playlist moved..";
    // libspotify workaround:
    if( newPosition > oldPosition ) newPosition -= 1;
    m_playlistList.move( oldPosition, newPosition );
    emit playlistMoved( oldPosition, newPosition );
    qDebug() << "A playlist was moved.";
}

void Spotify::Session::playListContainerLoadedCallback(sp_playlistcontainer *pc, void* userdata)
{
    qDebug()<<"Container loaded callback..";
   Spotify::Session::callingSession->playListContainerLoadedWrapper(pc, userdata);
}

void Spotify::Session::playListContainerLoadedWrapper(sp_playlistcontainer *pc, void* userdata)
{
   Q_UNUSED( userdata );
   qDebug()<< "Playlist container loaded";

   emit rootListContainerLoaded(pc);
}

void Spotify::Session::OnRootListContainerLoaded(sp_playlistcontainer* pc)
{
     qDebug() << "Number of playlists:" << sp_playlistcontainer_num_playlists( pc );

     m_playlistcontainer = pc;

     m_playlistList.clear(); //clear list of playlists

     getPlaylists();

}

void Spotify::Session::tracksAddedCallback( sp_playlist* pl, sp_track* const* tracks, int num_tracks,
                                           int position, void* userdata )
{
    qDebug()<<"Tracks added callback..";
    Spotify::Session::callingSession->tracksAddedWrapper( pl, tracks, num_tracks, position, userdata );
}

void Spotify::Session::tracksAddedWrapper( sp_playlist* pl, sp_track* const* tracks, int num_tracks,
                                          int position, void* userdata )
{
    Q_UNUSED( userdata );
    emit onTracksAdded(pl,tracks,num_tracks,position);
 }

void Spotify::Session::OnTracksAdded(sp_playlist* pl, sp_track* const* tracks, int num_tracks, int position)
{
    if( m_playlistHash.contains( pl ) )
    {
        Spotify::Playlist* playlist = m_playlistHash.value( pl );
        if( playlist )
        {
            qDebug() << "Adding " <<num_tracks << " tracks to playlist " << playlist->getName();
            QList< Spotify::Track* > trackList;
            for( int i = 0; i < num_tracks; ++i )
            {
                // Un-const track:
                sp_track* tr = const_cast< sp_track* >( tracks[i] );
                if( tr )
                {
                    Spotify::Track* track = new Spotify::Track( tr );
                    track->setParent( playlist );
                    trackList.append( track );
                }
            }
            playlist->insertTracks( trackList, position );
        }
    }
}

void Spotify::Session::tracksRemovedCallback( sp_playlist* pl, const int* tracks, int num_tracks, void* userdata )
{
    Spotify::Session::callingSession->tracksRemovedWrapper( pl, tracks, num_tracks, userdata );
}

void Spotify::Session::tracksRemovedWrapper( sp_playlist* pl, const int* tracks, int num_tracks, void* userdata )
{
    Q_UNUSED( userdata );
    qDebug() << "Should remove " << num_tracks << " tracks";
    if( m_playlistHash.contains( pl ) )
    {
        Spotify::Playlist* playlist = m_playlistHash.value( pl );
        if( playlist )
        {
            playlist->removeTracks( tracks, num_tracks );
        }
    }
}

void Spotify::Session::tracksMovedCallback( sp_playlist* pl, const int* tracks, int num_tracks, int new_position,
                                            void* userdata)
{
    Spotify::Session::callingSession->tracksMovedWrapper( pl, tracks, num_tracks, new_position, userdata );
}

void Spotify::Session::tracksMovedWrapper( sp_playlist* pl, const int* tracks, int num_tracks, int new_position,
                                           void* userdata )
{
    Q_UNUSED( userdata );
    qDebug() << "Should move " << num_tracks << " tracks";
    if( m_playlistHash.contains( pl ) )
    {
        Spotify::Playlist* playlist = m_playlistHash.value( pl );
        if( playlist )
        {
            if( new_position > tracks[0] ) new_position -= 1;
            playlist->moveTracks( tracks, num_tracks, new_position );
        }
    }
}

void Spotify::Session::playlistRenamedCallback( sp_playlist* pl, void* userdata )
{
    qDebug()<<"Playlist renamed callback..";
    Spotify::Session::callingSession->playlistRenamedWrapper( pl, userdata );
}

void Spotify::Session::playlistRenamedWrapper( sp_playlist* pl, void* userdata )
{
    Q_UNUSED( userdata );

    emit onPlaylistRenamed(pl);
}

void Spotify::Session::OnPlayListRenamed(sp_playlist* playlist)
{

    if( m_playlistHash.contains( playlist ) )
    {
        Spotify::Playlist* pl = m_playlistHash.value( playlist );
        if( pl )
        {
            pl->updateName();
            if(isPlaylistContainersReady()) //all playlists have been succesfully loaded
            {
                emit playlistContainersReady(m_playlistList);
            }
        }
    }
}


void Spotify::Session::playlistStateChangedCallback( sp_playlist* pl, void* userdata )
{
    Spotify::Session::callingSession->playlistStateChangedWrapper( pl, userdata );
}

void Spotify::Session::playlistStateChangedWrapper( sp_playlist* pl, void*  userdata )
{
    Q_UNUSED( userdata );
    if( m_playlistHash.contains( pl ) )
    {
        Spotify::Playlist* playlist = m_playlistHash.value( pl );
        if( playlist )
        {
            playlist->updateState();
        }
    }
}

void Spotify::Session::playlistUpdateInProgressCallback( sp_playlist* pl, bool done, void* userdata )
{
    Spotify::Session::callingSession->playlistUpdateInProgressWrapper( pl, done, userdata );
}

void Spotify::Session::playlistUpdateInProgressWrapper( sp_playlist* pl, bool done, void* userdata )
{
    Q_UNUSED( pl );
    Q_UNUSED( done );
    Q_UNUSED( userdata );
}

void Spotify::Session::playlistMetadataUpdatedCallback( sp_playlist* pl, void* userdata )
{
    Spotify::Session::callingSession->playlistMetadataUpdatedWrapper( pl, userdata );
}

void Spotify::Session::playlistMetadataUpdatedWrapper( sp_playlist* pl, void* userdata )
{
	 Q_UNUSED( userdata );

	emit playlistMetadataUpdated(pl);
}

void Spotify::Session::OnPlaylistMetadataUpdated(sp_playlist* pl)
{
	 qDebug() << "Playlist metadata updated";
	 if( m_playlistHash.contains( pl ) )
	 {
		 Spotify::Playlist* playlist = m_playlistHash.value( pl );
	     if( playlist )
	     {
	    	 qDebug()<< "Updating tracks for playlist " << sp_playlist_name(pl);
	       playlist->load();
	       playlist->updateTracks();
	     }
	 }

	 if(isPlaylistContainersReady()) //all playlists have been succesfully loaded
	 {
	    emit playlistContainersReady(m_playlistList);
	 }
}

void Spotify::Session::imageLoadedCallback( sp_image* image, void* userdata )
{
    Spotify::Session::callingSession->imageLoadedWrapper( image, userdata );
}

void Spotify::Session::imageLoadedWrapper( sp_image* image, void* userdata )
{
    emit onImageLoaded(image, userdata);
}

void Spotify::Session::OnImageLoaded(sp_image* image, void* userdata)
{

    if( !image || !sp_image_is_loaded( image ) )
    {
        qDebug() << "Image not loaded..";
        return;
    }

    QString format = "Unknown";
    uchar* data = NULL;
    sp_imageformat spformat = sp_image_format(image);
    size_t len = 0;

    // Get image format:
    switch (spformat)
    {
    case SP_IMAGE_FORMAT_JPEG:
        format = "JPEG";
        break;
    case SP_IMAGE_FORMAT_UNKNOWN:
        format = "UNKNOWN";
        break;
    default:
        format = "UNKNOWN";
        break;
    }

    if( format == "UNKNOWN" )
    {
        qDebug()<< "Unknown format!";
        return;
    }

    // Get actual image data
    data = (uchar*)sp_image_data(image,&len);

    // Create QImage instance, for wrapping the image data

    QImage img;

    if(!img.loadFromData(data, (int)len, format.toAscii()))
    {
        qDebug() << "Loading of image data failed!";
    }

    if(userdata)
    { //assign image to associated container
        Spotify::ImageContainer* pContainer = (Spotify::ImageContainer*)userdata;
        pContainer->setImage(img);
    }

    // Release sp_image
    sp_image_release(image);

    //emit coverArtReady(img, callerID);
}

void Spotify::Session::searchCompleteCallback( sp_search* search, void* userdata )
{
    Spotify::Session::callingSession->searchCompleteWrapper( search, userdata );
}

void Spotify::Session::searchCompleteWrapper( sp_search* search, void* userdata )
{
    Q_UNUSED( userdata )

    emit onSearchComplete(search);
}

void Spotify::Session::OnSearchComplete(sp_search* search)
{
      Spotify::Error error;
      error = sp_search_error( search );

     if(search && m_searchHash.contains( search ) )
     {
         Spotify::Search* s = m_searchHash.value( search );
         s->load(); //try to load search results
         emit searchComplete( s );
     }
     else
         qDebug() << "Error performing search: " << error.toString();
}

void Spotify::Session::loggedInCallback( sp_session* session, sp_error error )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->loggedInWrapper( Spotify::Error( error ) );
}

void Spotify::Session::loggedInWrapper( Spotify::Error error )
{

    // Let application know we're logged in (or not):
    m_valid = !error.isError();
   m_currentUser = new Spotify::User( sp_session_user( m_session ) );
   qDebug() <<"User " + m_currentUser->getDisplayName() + " logged in..";
   emit loggedIn( error );
}

void Spotify::Session::loggedOutCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->loggedOutWrapper();
}

void Spotify::Session::loggedOutWrapper()
{
    qDebug() << "Logged out";
    emit loggedOut();
}

void Spotify::Session::metadataUpdatedCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->metadataUpdatedWrapper();
}

void Spotify::Session::metadataUpdatedWrapper()
{
    emit metadataUpdated();
}

void Spotify::Session::connectionErrorCallback( sp_session* session, sp_error error )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->connectionErrorWrapper( Spotify::Error( error ) );
}

void Spotify::Session::connectionErrorWrapper( Spotify::Error error )
{
    Q_UNUSED( error );
    qDebug() << "Connection error: " << error.toString();
}

void Spotify::Session::messageToUserCallback( sp_session* session, const char* message )
{
    Q_UNUSED( session );
   // Spotify::Session::callingSession->messageToUserWrapper( QString( message ) );
}

void Spotify::Session::messageToUserWrapper( QString message )
{
    //qDebug() << "MESSAGE (user): " << message.trimmed();
    //emit messageToUser( message.trimmed() );
}

void Spotify::Session::notifyMainThreadCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->notifyMainThreadWrapper();
}

void Spotify::Session::notifyMainThreadWrapper()
{
    //notification to wake up main thread.

   if(!m_DisableNotifyThread)
    {
        m_ProcessEvents = true;
        emit onNotifyMainThread();
    }
}

int Spotify::Session::musicDeliveryCallback( sp_session* session, const sp_audioformat* format,
                                             const void* frames, int num_frames )
{
    Q_UNUSED( session );
    return Spotify::Session::callingSession->musicDeliveryWrapper( format, frames, num_frames );
}

int Spotify::Session::musicDeliveryWrapper( const sp_audioformat* format, const void* frames, int num_frames )
{

    // No data or invalid format/frames:
    if (num_frames == 0 || !frames || !format || !m_playing)
        return 0;

    int bufSize = num_frames * sizeof(int16_t) * format->channels;

    QByteArray buf((const char*)frames,bufSize);

    emit onMusicDelivery(num_frames,format,buf);

    return num_frames;
}

void Spotify::Session::OnMusicDelivery(int num_frames, const sp_audioformat* format, QByteArray buffer)
{
    //Audio data buffer received, add buffered data at end of audio playback queue

    QByteArray bufPart;

    m_pAudioThread->getMutex()->lock();

    if(m_preBufActive)
    {
        if(m_pAudioThread->getNewTrackStarted())
        {
            qDebug() << "Pre-buffering active; next track started...";
            m_pAudioThread->setNewTrackStarted(false); //reset track started state
            m_pAudioThread->setNextTrackStartPos();
            m_bufferedAmount = 0; //reset buffering amount (newly loaded track)
        }
    }

    m_bufferedAmount += (num_frames * 1000.0)/format->sample_rate; //increase the currently buffered amount

    emit trackBufferingAmountUpdated( m_bufferedAmount );

    //split audio data buffer into max 1 sec chunks

    for(int bufPos = 0; bufPos < buffer.size(); bufPos+=format->sample_rate*sizeof(int16_t)*format->channels)
    {
        bufPart = buffer.mid(bufPos,format->sample_rate*sizeof(int16_t)*format->channels);
        m_pAudioThread->FIFOAppend(bufPart);
    }

    m_pAudioThread->incFIFOSize(buffer.size());


    if(m_pAudioThread->getFIFOSize()>=(m_pAudioThread->getMinBufferSize()+5)*format->sample_rate*sizeof(int16_t)*format->channels) //>=35 secs of data in buffer
    {
        m_pAudioThread->setBufferingPaused(true);
        m_pAudioThread->getMutex()->unlock();
        pause();
    }
    else
    {
        m_pAudioThread->getMutex()->unlock();
    }
}

void Spotify::Session::OnResumeBuffering()
{
    //signal from playback thread that more buffered data is needed
    pause();
}

void Spotify::Session::playTokenLostCallback( sp_session* session )
{
    Spotify::Session::callingSession->playTokenLostWrapper( session );
}

void Spotify::Session::playTokenLostWrapper( sp_session* session )
{
    //emit signal when playback token has been lost (e.g., same account logs in somewhere else)
    Q_UNUSED( session );
    emit playTokenLost();
}

void Spotify::Session::logMessageCallback( sp_session* session, const char* message )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->logMessageWrapper( QString( message ) );
}

void Spotify::Session::logMessageWrapper( QString message )
{
    qDebug() << "MESSAGE (log):  " << message.trimmed();
}

void Spotify::Session::endOfTrackCallback( sp_session* session )
{
    Q_UNUSED( session );
    Spotify::Session::callingSession->endOfTrackWrapper();
}

void Spotify::Session::endOfTrackWrapper()
{
    qDebug()<<"End of track callback..";
    if(!m_preBufActive)
    {
        m_pAudioThread->getMutex()->lock();
        m_pAudioThread->setEndOfTrack(true);
        m_pAudioThread->getMutex()->unlock();
    }
    else
    {
        emit bufferNextTrack(); //pre-buffering active; signal that buffering of following track can be initiated
    }
}

