#include <QDebug>
#include "artistcatalogue.h"
#include "artist.h"
#include "track.h"
#include "album.h"

namespace Spotify
{
    ArtistCatalogue::ArtistCatalogue( sp_artistbrowse* ab, QObject* parent ):
            Spotify::TrackCollection( parent ),
            m_artistbrowse( ab ),
            m_artist( 0 )
    {
        // Impclitily reffed by sp_artistbrowse_create()
    }

    bool ArtistCatalogue::load()
    {
        if( !m_artistbrowse ) return false;

        // Only trigger this once, libspotify guarantees metadata is ready
        if( m_artistbrowse && sp_artistbrowse_is_loaded( m_artistbrowse ) && !m_isLoaded )
        {
            // Set artist:
            if( !m_artist )
            {
                m_artist = new Spotify::Artist( sp_artistbrowse_artist( m_artistbrowse ), false, this );
                if( !m_artist->isLoaded() && !m_artist->load() )
                {
                    qDebug() << "Unable to load artist..";
                    return false;
                }
            }

            // Set portraits:
            if( m_portraits.isEmpty() )
            {
                int numPortraits = sp_artistbrowse_num_portraits( m_artistbrowse );
                for( int i = 0; i < numPortraits; ++i )
                {
                    m_portraits.append( sp_artistbrowse_portrait( m_artistbrowse, i) );
                }
            }

            // Set tracks:
            if( m_tracks.isEmpty() )
            {
                int numTracks = sp_artistbrowse_num_tracks( m_artistbrowse );
                for( int i = 0; i < numTracks; ++i )
                {
                    sp_track* sptrack = sp_artistbrowse_track( m_artistbrowse, i );
                    Spotify::Track* track = new Spotify::Track( sptrack, false, this );
                    addTrack( track );

                    // Tracks should be loaded:
                    if( !track->isLoaded() && !track->load() )
                        return false;
                }
            }

            if( m_similarArtists.isEmpty() )
            {
                int numSimilarArtists = sp_artistbrowse_num_similar_artists( m_artistbrowse );
                for( int i = 0; i < numSimilarArtists; ++i )
                {
                    sp_artist* spartist = sp_artistbrowse_similar_artist( m_artistbrowse, i );
                    Spotify::Artist* artist = new Spotify::Artist( spartist, false, this );
                    m_similarArtists.append( artist );

                    // Artists should be loaded:
                    if( !artist->isLoaded() && !artist->load() )
                        return false;
                }
            }

            if( m_biography.isEmpty() )
            {
                const char* charData = sp_artistbrowse_biography( m_artistbrowse );
                m_biography = QString::fromUtf8( charData, strlen( charData ) );
            }

            for( int i = 0; i < m_tracks.count(); ++i )
            {
                Spotify::Track* t = m_tracks.at(i);
                Spotify::Album* a = t->getAlbum();
                Spotify::Album::Type type = a->type();
                if( type == Spotify::Album::ALBUM  )
                {
                    QList< Spotify::Album* > albumList = m_albums.keys();
                    bool added = false;
                    for( int j = 0; j < albumList.count(); ++j )
                    {
                        // Already got it:
                        if( albumList.at(j)->getAlbum() == a->getAlbum() )
                        {
                            added = true;
                            m_albums.value( albumList.at(j) )->append( m_tracks.at(i) );
                            break;
                        }
                    }
                    if( !added && a->isAvailable() && a->getArtist()->getArtist() == getArtist()->getArtist())
                    {
                        QList< Spotify::Track* >* tracks = new QList< Spotify::Track* >();
                        tracks->append( m_tracks.at(i));
                        m_albums.insert( a, tracks );
                    }
                }
                else if( type == Spotify::Album::SINGLE && !m_singles.contains( a ) )
                {
                    QList< Spotify::Album* > albumList = m_singles.keys();
                    bool added = false;
                    for( int j = 0; j < albumList.count(); ++j )
                    {
                        // Already got it:
                        if( albumList.at(j)->getAlbum() == a->getAlbum() )
                        {
                            added = true;
                            m_singles.value( albumList.at(j) )->append( m_tracks.at(i) );
                            break;
                        }
                    }
                    if( !added && a->isAvailable() && a->getArtist()->getArtist() == getArtist()->getArtist() )
                    {
                        QList< Spotify::Track* >* tracks = new QList< Spotify::Track* >();
                        tracks->append( m_tracks.at(i));
                        m_singles.insert( a, tracks );
                    }
                }
                else if( type == Spotify::Album::COMPILATION && !m_compilations.contains( a ) )
                {
                    QList< Spotify::Album* > albumList = m_compilations.keys();
                    bool added = false;
                    for( int j = 0; j < albumList.count(); ++j )
                    {
                        // Already got it:
                        if( albumList.at(j)->getAlbum() == a->getAlbum() )
                        {
                            added = true;
                            m_compilations.value( albumList.at(j) )->append( m_tracks.at(i) );
                            break;
                        }
                    }
                    if( !added && a->isAvailable() && a->getArtist()->getArtist() == getArtist()->getArtist() )
                    {
                        QList< Spotify::Track* >* tracks = new QList< Spotify::Track* >();
                        tracks->append( m_tracks.at(i));
                        m_compilations.insert( a, tracks );
                    }

                }
                else if( type == Spotify::Album::UNKNOWN && !m_others.contains( a ) )
                {
                    QList< Spotify::Album* > albumList = m_others.keys();
                    bool added = false;
                    for( int j = 0; j < albumList.count(); ++j )
                    {
                        // Already got it:
                        if( albumList.at(j)->getAlbum() == a->getAlbum() )
                        {
                            added = true;
                            m_others.value( albumList.at(j) )->append( m_tracks.at(i) );
                            break;
                        }
                    }
                    if( !added && a->isAvailable() && a->getArtist()->getArtist() == getArtist()->getArtist() )
                    {
                        QList< Spotify::Track* >* tracks = new QList< Spotify::Track* >();
                        tracks->append( m_tracks.at(i));
                        m_others.insert( a, tracks );
                    }
                }
            }

            m_isLoaded = true;
            emit loaded();
        }
        return m_isLoaded;
    }

    ArtistCatalogue::~ArtistCatalogue()
    {
        qDebug() << "Artist catalogue deleted along with " << this->children().count() << " children";
        QList< QList< Spotify::Track* >* > albums;
        for( int i = 0; i < albums.count(); ++i )
            delete albums.at(i);

        QList< QList< Spotify::Track* >* > singles;
        for( int i = 0; i < singles.count(); ++i )
            delete singles.at(i);

        QList< QList< Spotify::Track* >* > compilations;
        for( int i = 0; i < compilations.count(); ++i )
            delete compilations.at(i);

        QList< QList< Spotify::Track* >* > others;
        for( int i = 0; i < others.count(); ++i )
            delete others.at(i);

        if( m_artistbrowse )
            sp_artistbrowse_release( m_artistbrowse );
    }

} // namespace Spotify
