//AlbumView.cpp: Implementation of the AlbumView class.

#include "AlbumView.h"

AlbumView::AlbumView(Spotify::Session* pSession, QWidget *pParent)
    : QDialog(pParent)
{
    //default constructor, set up the UI elements etc.

    setWindowTitle(tr("Browse Album"));

    m_pAlbumBrowser = NULL;

    m_CallerID = ALBUMVIEW_ID;

    m_pSession = pSession; //set Spotify session instance pointer

    //setup signal / slot connections
    connect(m_pSession,SIGNAL(albumBrowserReady(Spotify::AlbumBrowser*,qint32)),SLOT(OnAlbumBrowserReady(Spotify::AlbumBrowser*,qint32)));
    connect(m_pSession,SIGNAL(coverArtReady(QImage,qint32)),SLOT(OnAlbumCoverReady(QImage,qint32)));

    m_pAlbumTrackView = new QListView(this);
    m_pAlbumTrackDelegate = new ListViewDelegate(this);
    m_pAlbumTrackDelegate->setHeaderFontPointSize(15);
    m_pAlbumTrackDelegate->setSubHeaderFontPointSize(14);
    m_pAlbumTrackModel = new QStandardItemModel(this);

    //connect view to item delegate and model
    m_pAlbumTrackView->setModel(m_pAlbumTrackModel);
    m_pAlbumTrackView->setItemDelegate(m_pAlbumTrackDelegate);
    connect(m_pAlbumTrackView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));

    QFont captionFont;
    captionFont.setFamily(QApplication::font().family());
    captionFont.setPointSize(19);
    m_AlbumCaption.setFont(captionFont);
    m_AlbumCaption.setWordWrap(true);

    m_AlbumCoverContainer.setFixedSize(290,290);

    m_ArtistViewBtn.setFixedWidth(290);
    m_ArtistViewBtn.setDisabled(true); //currently disabled

    m_LeftSubLayout.addWidget(&m_AlbumCoverContainer);
    m_LeftSubLayout.addWidget(&m_ArtistViewBtn);
    m_LeftSubLayout.setSpacing(10);

    m_MainLayout.addLayout(&m_LeftSubLayout,0,0,2,1);
    m_MainLayout.addWidget(&m_AlbumCaption,0,1,1,1);
    m_MainLayout.addWidget(m_pAlbumTrackView,1,1,1,1);

    setLayout(&m_MainLayout);
}

AlbumView::~AlbumView()
{
}

void AlbumView::setAlbum(Spotify::Album* pAlbum)
{
    //public method for setting the album instance to be displayed
    bool status;

    if(m_pSession && pAlbum)
    {
        m_AlbumCoverContainer.clear();

        m_pAlbumTrackModel->clear(); //clear model rows

        m_AlbumCaption.setText("");

        status = m_pSession->browse(pAlbum,&m_CallerID); //request to browse specified album details

        if(!status)
        {
            //error occured
            QMaemo5InformationBox::information(this,tr("Fetching of album details failed!"));
        }
        else
        {
            //enable progress indication, browse call succeeded
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true);
        }
    }
}

void AlbumView::OnAlbumBrowserReady(Spotify::AlbumBrowser* browser, qint32 callerID)
{
     //Album browser request complete, set up the Album View

    Spotify::Error error;

    QFont artistBtnFont;
    artistBtnFont.setFamily(QApplication::font().family());
    artistBtnFont.setPointSize(15);
    QFontMetrics fm(artistBtnFont);

    if(callerID==m_CallerID) //check that results was requested by album view
    {
        if(browser)
        {
            error = sp_albumbrowse_error(browser->getAlbumBrowser());
            if(error.isError())
            {
                //error occured (album browse request failed)
                QMaemo5InformationBox::information(this,tr("Fetching of album details failed: ") + error.toString());
            }
            else
            {
                m_pAlbumBrowser = browser;
                m_pAlbumBrowser->load();
                m_pAlbumBrowser->sortByDiscIndex();
                if(m_pAlbumBrowser->getAlbum())
                {
                    m_pSession->getAlbumCover(m_pAlbumBrowser->getAlbum(),&m_CallerID); //request album coverart
                    m_AlbumCaption.setText(m_pAlbumBrowser->getAlbum()->getName() + " (" + QString::number(m_pAlbumBrowser->getAlbum()->getYear()) + ")"); //set album title
                }
                if(m_pAlbumBrowser->getArtist())
                {
                  m_ArtistViewBtn.setText(fm.elidedText(m_pAlbumBrowser->getArtist()->getName(),Qt::ElideRight,290));
                }
                //add tracks to listview
                addTracks();
            }
        }

        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
    }
}

void AlbumView::addTracks()
{
    //add album tracks to the listview

    QPixmap iconMap;

    iconMap.load("/usr/share/images/icon-type-song-color.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    for(int pos = 0; pos < m_pAlbumBrowser->getTrackCount(); pos++)
    {
        Spotify::Track* currTrack = m_pAlbumBrowser->getTrack(pos);
        if(currTrack)
        {
            QStandardItem* trackItem = new QStandardItem();
            if(!currTrack->isAvailable())
                trackItem->setEnabled(false); //not possible to play unavailable track, but display it anyway in the listview
            trackItem->setEditable(false);
            trackItem->setData(trackTypeIcon,ListViewDelegate::iconRole);
            trackItem->setData(currTrack->getName(),ListViewDelegate::headerRole);
            if(currTrack->getArtist())
            {
                trackItem->setData(currTrack->getArtist()->getName(),ListViewDelegate::subHeaderLeftJustifyRole);
            }
            trackItem->setData(currTrack->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
            trackItem->setData(pos,Qt::UserRole); //assign position in internal album track list
            m_pAlbumTrackModel->appendRow(trackItem);
        }
    }

    if(m_pAlbumTrackModel->rowCount()>0)
    { //scroll to top, no current / selected row by default
        m_pAlbumTrackView->scrollToTop();
        QModelIndex index = m_pAlbumTrackModel->index(-1,-1);
        m_pAlbumTrackView->setCurrentIndex(index);
    }
}

void AlbumView::OnTrackActivated(QModelIndex trackIndex)
{
    //A track has been activated (e.g., selected) in the list of album tracks

    Q_UNUSED(trackIndex)

    m_pAlbumTrackView->setEnabled(false); //disable listview temporarily

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //show progress indicator
    setPlayQueue();
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator

    m_pAlbumTrackView->setEnabled(true); //re-enable the listview
 }

void AlbumView::setPlayQueue()
{
    //Set playqueue (i.e., add all tracks from selected track to last track in list to playqueue)

    QList<Spotify::Track*> PlayQueue;
    Spotify::Track* track;

    for(int pos = m_pAlbumTrackView->currentIndex().row(); pos < m_pAlbumTrackModel->rowCount(); pos++)
    {
        int trackPos = m_pAlbumTrackModel->item(pos)->data(Qt::UserRole).toInt();
         if(m_pAlbumBrowser) //track from playlist
             track = m_pAlbumBrowser->getTrack(trackPos);
         if(track)
         {
             if(track->isAvailable()) //only add track if it is available (playable)
                PlayQueue.append(track);
         }
    }

    emit playQueueSet(PlayQueue); //signal that playqueue has been set / updated
    hide();
}

void AlbumView::OnAlbumCoverReady(QImage image, qint32 callerID)
{
    //Album coverart has been loaded (for current album),
    //assign it to the album coverart container.
    if(callerID==m_CallerID)
    {
         m_AlbumCoverContainer.setPixmap(QPixmap::fromImage(image).scaled(290,290));
    }
}
