import QtQuick 1.0
import QtMultimediaKit 1.1
import "components"
import "scripts/dateandtime.js" as DT
import "scripts/createobject.js" as ObjectCreator

Page {
    id: root

    property bool videoPlaying: true
    property variant currentVideo: []
    property int playlistPosition: 0
    property bool gettingVideoUrl: false

    signal playbackStopped

    function setPlaylist(videoList) {
        playbackQueue.clear();
        for (var i = 0; i < videoList.length; i++) {
            playbackQueue.append(videoList[i]);
        }
        playlistPosition = 0;
        currentVideo = playbackQueue.get(0);
        startPlayback();
    }

    function appendPlaylist(videoList) {
        if (playbackQueue.count == 0) {
            playlistPosition = 0;
            currentVideo = videoList[0];
        }
        for (var i = 0; i < videoList.length; i++) {
            playbackQueue.append(videoList[i]);
        }
    }

    function startPlayback() {
        videoPlayer.stop();
        videoPlayer.source = "";
        if (currentVideo.archive) {
            videoPlayer.setVideo(currentVideo.filePath);
        }
        else {
            gettingVideoUrl = true;
            if (currentVideo.youtube) {
                YouTube.getVideoUrl(currentVideo.videoId);
            }
            else if (currentVideo.dailymotion) {
                Dailymotion.getVideoUrl(currentVideo.id);
            }
            else if (currentVideo.xvideos) {
                XTube.getXVideosVideoUrl(currentVideo.videoId);
            }
            else if (currentVideo.youporn) {
                XTube.getYouPornVideoUrl(currentVideo.videoId);
            }
            else if (currentVideo.xhamster) {
                XTube.getXHamsterVideoUrl(currentVideo.videoId);
            }
        }
    }

    function previous() {
        /* Play the previous item in the playlist */

        if (playlistPosition > 0) {
            playlistPosition--;
            getNextVideo();
        }
    }

    function next() {
        /* Play the next item in the playlist */

        if (playlistPosition < playbackQueue.count - 1) {
            playlistPosition++;
            getNextVideo();
        }
        else if ((Settings.showRelatedVideos) && (!currentVideo.archive)) {
            if (currentVideo.youtube) {
                showYouTubeRelatedVideos();
            }
            else if (currentVideo.dailymotion) {
                showDailymotionRelatedVideos();
            }
        }
        else {
            stopPlayback();
        }
    }

    function getNextVideo() {
        if (playlistPosition < playbackQueue.count) {
            currentVideo = playbackQueue.get(playlistPosition);
            startPlayback();
        }
    }

    function stopPlayback() {
        videoPlayer.stop();
        videoPlayer.source = "";
        currentVideo = [];
        playbackStopped();
        pageStack.pop();
        MainWindow.setScreenOrientation(Settings.screenOrientation);
        Utils.doNotDisturb(false);
    }

    function showYouTubeRelatedVideos() {
        var dialog = ObjectCreator.createObject(Qt.resolvedUrl("YTRelatedVideosDialog.qml"), appWindow);
        dialog.videoId = currentVideo.videoId;
        dialog.playVideo.connect(setPlaylist);
        dialog.closed.connect(stopPlayback);
        dialog.open();
    }

    function showDailymotionRelatedVideos() {
        var dialog = ObjectCreator.createObject(Qt.resolvedUrl("DMRelatedVideosDialog.qml"), appWindow);
        dialog.videoId = currentVideo.id;
        dialog.playVideo.connect(setPlaylist);
        dialog.closed.connect(stopPlayback);
        dialog.open();
    }

    Component.onCompleted: {
        MainWindow.setScreenOrientation("landscape");
        Utils.doNotDisturb(true);
    }

    Connections {
        target: YouTube
        onGotVideoUrl: {
            gettingVideoUrl = false;
            videoPlayer.setVideo(videoUrl);
            var cv = currentVideo;
            cv["videoUrl"] = videoUrl;
            currentVideo = cv;
        }
        onVideoUrlError: {
            gettingVideoUrl = false;
            next();
        }
    }

    Connections {
        target: Dailymotion
        onGotVideoUrl: {
            gettingVideoUrl = false;
            videoPlayer.setVideo(videoUrl);
            var cv = currentVideo;
            cv["videoUrl"] = videoUrl;
            currentVideo = cv;
        }
        onVideoUrlError: {
            gettingVideoUrl = false;
            next();
        }
    }

    Connections {
        target: XTube
        onGotVideoUrl: {
            gettingVideoUrl = false;
            videoPlayer.setVideo(videoUrl);
            var cv = currentVideo;
            cv["videoUrl"] = videoUrl;
            currentVideo = cv;
        }
        onVideoUrlError: {
            gettingVideoUrl = false;
            next();
        }
    }

    SelectionDialog {
        id: queueDialog

        titleText: qsTr("Playback queue")
        model: ListModel {
            id: playbackQueue
        }
        destroyOnClosing: false
        emitValueChosenSignal: false
        onAccepted: {
            playlistPosition = selectedIndex;
            getNextVideo();
        }
        onStateChanged: if (state == "open") selectedIndex = playlistPosition;
    }

    ToolBar {
        id: toolBar

        property bool show: false
        property bool pressed: (stopButton.pressed) || (playButton.pressed) || (aspectRatioButton.pressed) || (queueButton.pressed) || (seekMouseArea.pressed)

        z: 10
        height: 70
        anchors { left: parent.left; right: parent.right; top: parent.bottom }

        states: State {
            name: "show"
            when: toolBar.show
            AnchorChanges { target: toolBar; anchors { top: undefined; bottom: parent.bottom } }
        }

        transitions: Transition {
            AnchorAnimation { easing.type: Easing.OutQuart; duration: 300 }
        }

        tools: Item {
            anchors.fill: parent

            Row {
                id: buttonRow

                height: parent.height
                anchors { left: parent.left; right: parent.right; verticalCenter: parent.verticalCenter }

                spacing: 10

                ToolIcon {
                    id: stopButton

                    anchors.verticalCenter: parent.verticalCenter
                    iconSource: "images/back-white.png"
                    onClicked: stopPlayback()
                }

                ToolIcon {
                    id: playButton

                    anchors.verticalCenter: parent.verticalCenter
                    iconSource: videoPlayer.paused ? "images/play-white.png" : "images/pause-white.png"
                    onClicked: videoPlayer.paused = !videoPlayer.paused
                }

                Item {
                    height: 70
                    width: 526

                    ProgressBar {
                        id: progressBar

                        useTheme: false
                        width: 400
                        anchors.centerIn: parent
                        value: toolBar.show ? videoPlayer.position / videoPlayer.duration : 0
                    }

                    Label {
                        anchors { verticalCenter: progressBar.verticalCenter; right: progressBar.left; rightMargin: 10 }
                        font.pixelSize: _SMALL_FONT_SIZE
                        color: "white"
                        text: (!toolBar.show) || (videoPlayer.status == Video.Loading) || (videoPlayer.status == Video.Buffering) ? "0:00" : DT.getTime(videoPlayer.position)
                    }

                    Label {
                        anchors { verticalCenter: progressBar.verticalCenter; left: progressBar.right; leftMargin: 10 }
                        font.pixelSize: _SMALL_FONT_SIZE
                        color: "white"
                        text: toolBar.show ? DT.getTime(videoPlayer.duration) : "0:00"
                    }

                    SeekBubble {
                        id: seekBubble

                        z: 1000
                        anchors.bottom: progressBar.top
                        opacity: value != "" ? 1 : 0
                        value: (seekMouseArea.drag.active) && (seekMouseArea.posInsideDragArea) ? DT.getTime(Math.floor((seekMouseArea.mouseX / seekMouseArea.width) * videoPlayer.duration)) : ""
                    }

                    MouseArea {
                        id: seekMouseArea

                        property bool posInsideMouseArea: false
                        property bool posInsideDragArea: (seekMouseArea.mouseX >= 0) && (seekMouseArea.mouseX <= (seekMouseArea.drag.maximumX - 10))

                        width: progressBar.width
                        height: 70
                        anchors.centerIn: progressBar
                        drag.target: seekBubble
                        drag.axis: Drag.XAxis
                        drag.minimumX: 0
                        drag.maximumX: width + 10
                        enabled: !currentVideo.dailymotion ? true : false
                        onExited: posInsideMouseArea = false
                        onEntered: posInsideMouseArea = true
                        onPressed: {
                            posInsideMouseArea = true;
                            seekBubble.x = mouseX + 10;
                        }
                        onReleased: if (posInsideMouseArea) videoPlayer.position = Math.floor((mouseX / width) * videoPlayer.duration);
                    }
                }

                ToolIcon {
                    id: aspectRatioButton

                    property variant ratios: ["stretch", "fit", "crop"]

                    anchors.verticalCenter: parent.verticalCenter
                    iconSource: "images/aspect-" + ratios[videoPlayer.aspectRatio] + ".png"
                    onClicked: aspectControls.visible = !aspectControls.visible
                }

                ToolIcon {
                    id: queueButton

                    anchors.verticalCenter: parent.verticalCenter
                    iconSource: "images/list-white.png"
                    onClicked: queueDialog.open()
                }
            }
        }
    }

    Timer {
        id: controlsTimer

        running: ((toolBar.show) || (aspectControls.visible)) && !((toolBar.pressed) || (aspectControls.pressed))
        interval: 3000
        onTriggered: {
            toolBar.show = false;
            aspectControls.visible = false;
        }
    }

    Video {
        id: videoPlayer

        property bool repeat: false // True if playback of the current video is to be repeated
        property int aspectRatio // fillMode is a non-NOTIFYable property

        function setVideo(videoUrl) {
            videoPlayer.source = videoUrl;
            videoPlayer.play();
        }

        width: 800
        height: 480
        anchors.centerIn: parent
        opacity: position > 0 ? 1 : 0
        onSourceChanged: aspectRatio = 1
        onAspectRatioChanged: fillMode = aspectRatio
        onError: {
            if ((error != Video.NoError) && (videoPlayer.source != "")) {
                if (error == Video.NetworkError) {
                    messages.displayMessage(messages._NETWORK_ERROR);
                }
                else if (error == Video.FormatError) {
                    messages.displayMessage(messages._FORMAT_ERROR);
                }
                next();
            }
        }
        onPositionChanged: {
            if ((videoPlayer.position > 2000) && ((videoPlayer.duration - videoPlayer.position) < 50)) {
                if (videoPlayer.repeat) {
                    videoPlayer.position = 0;
                    videoPlayer.play();
                }
                else {
                    next();
                }
            }
        }
    }

    MouseArea {
        id: videoMouseArea

        property int xPos

        anchors.fill: videoPlayer
        onPressed: xPos = mouseX
        onReleased: {
            if (videoPlayer.paused) {
                videoPlayer.paused = false;
            }
            else if (((xPos - mouseX) > 100) && (playlistPosition < playbackQueue.count - 1)) {
                next();
            }
            else if ((mouseX - xPos) > 100) {
                previous();
            }
            else {
                aspectControls.visible = false;
                toolBar.show = !toolBar.show;
            }
        }
    }

    AspectRatioControls {
        id: aspectControls

        anchors.centerIn: videoPlayer
        visible: false
    }

    Label {
        anchors.centerIn: videoPlayer
        font.pixelSize: _LARGE_FONT_SIZE
        color: "white"
        text: qsTr("Loading...")
        visible: ((videoPlayer.status == Video.Loading) || (videoPlayer.status == Video.Buffering) || (gettingVideoUrl)) && (!videoPlayer.paused)
    }

    Image {
        anchors.centerIn: videoPlayer
        source: videoMouseArea.pressed ? "images/play-button-" + Settings.activeColorString + ".png" : "images/play-button.png"
        visible: videoPlayer.paused
    }
}
