import QtQuick 1.0
import "components"
import "scripts/youtube.js" as YT
import "scripts/createobject.js" as ObjectCreator

Page {
    id: root

    property string username
    property bool isSubscribed
    property string subscriptionId
    property string subscriberCount
    property string videoCount
    property string userThumbnail
    property string about
    property string age
    property string firstName
    property string lastName
    property string gender
    property string website

    function getUserProfile(user) {
        if (user == "default") {
            subscriptionButton.visible = false;
        }
        else {
            username = user;

            var i = 0;
            while ((!isSubscribed) && (i < youtubeSubscriptionsModel.count)) {
                if (youtubeSubscriptionsModel.get(i).title == username) {
                    subscriptionId = youtubeSubscriptionsModel.get(i).subscriptionId.split(":")[5];
                    isSubscribed = true;
                }
                i++;
            }
        }
        YT.getUserProfile(user);
    }

    function showUploadDialog() {
        var uploadDialog = ObjectCreator.createObject(Qt.resolvedUrl("YTUploadDialog.qml"), appWindow);
        uploadDialog.open();
    }

    function goToWatchLaterPlaylist() {
        pageStack.push(Qt.resolvedUrl("YTPlaylistVideosPage.qml"));
        pageStack.currentPage.showWatchLaterPlaylist();
    }

    title: qsTr("Profile")
    tools: Menu {

        menuItems: [
            MenuItem {
                text: qsTr("Upload video")
                onClicked: showUploadDialog()
            },

            MenuItem {
                text: qsTr("View 'watch later' playlist")
                onClicked: goToWatchLaterPlaylist()
            }
        ]
    }

    Connections {
        target: YouTube
        onSubscribed: {
            if (id == subscriptionId) {
                isSubscribed = true;
                messages.displayMessage(qsTr("You have subscribed to") + " '" + username + "'");
            }
        }
        onUnsubscribed: {
            if (id == subscriptionId) {
                isSubscribed = false;
                messages.displayMessage(qsTr("You have unsubscribed to") + " '" + username + "'");
            }
        }
    }

    Flickable {
        id: flicker

        anchors.fill: parent
        clip: true
        contentWidth: parent.width
        contentHeight: 250 + col1.height + col2.height
        flickableDirection: Flickable.VerticalFlick
        visible: !(videoCount == "")

        Label {
            id: titleText

            anchors { top: parent.top; left: parent.left; right: parent.right; margins: 10 }
            
            font.bold: true
            
            wrapMode: Text.WordWrap
            text: username
        }

        Image {
            id: cover

            width: 160
            height: 120
            anchors { left: titleText.left; top: titleText.bottom; topMargin: 10 }
            source: userThumbnail
            sourceSize.width: width
            sourceSize.height: height
            smooth: true
            onStatusChanged: if (status == Image.Error) source = "images/avatar.png";
        }

        Button {
            id: subscriptionButton

            anchors { right: parent.right; top: cover.bottom; margins: 10 }
            text: isSubscribed ? qsTr("Unsubscribe") : qsTr("Subscribe")
            visible: (YouTube.userSignedIn) && (username != YouTube.username)
            onClicked: {
                toggleBusy(true);
                if (isSubscribed) {
                    YouTube.unsubscribeToChannel(subscriptionId);
                }
                else {
                    YouTube.subscribeToChannel(username);
                }
            }
        }

        Column {
            id: col1

            anchors { left: cover.right; leftMargin: 10; top: cover.top; right: parent.right; rightMargin: 10 }

            Label {
                width: parent.width
                
                
                elide: Text.ElideRight
                text: videoCount + " " + qsTr("uploads")
                visible: !(text == "")
            }

            Label {
                width: parent.width
                
                
                elide: Text.ElideRight
                text: subscriberCount + " " + qsTr("subscribers")
                visible: !(text == "")
            }
        }

        SeparatorLabel {
            id: separator

            anchors { left: parent.left; right: parent.right; top: subscriptionButton.visible ? subscriptionButton.bottom : cover.bottom; margins: 10 }
            text: qsTr("Profile")
        }

        Column {
            id: col2

            anchors { left: parent.left; right: parent.right; top: separator.bottom; margins: 10 }

            Label {
                width: parent.width
                
                
                text: qsTr("Name") + ": " + firstName + " " + lastName
                visible: !(firstName == "")
            }

            Label {
                width: parent.width
                
                
                text: qsTr("Age") + ": " + age
                visible: !(age == "")
            }

            Label {
                width: parent.width
                
                
                text: gender == "m" ? qsTr("Gender") + ": " + qsTr("Male") : qsTr("Gender") + ": " + qsTr("Female")
                visible: !(gender == "")
            }

            Row {
                width: parent.width

                Label {
                    id: webLabel

                    
                    
                    text: qsTr("Website") + ": "
                    visible: !(website == "")
                }

                Label {
                    width: parent.width - webLabel.width
                    
                    color: Settings.activeColor
                    elide: Text.ElideRight
                    text: website
                    opacity: websiteMouseArea.pressed ? 0.5 : 1
                    visible: !(text == "")

                    MouseArea {
                        id: websiteMouseArea

                        anchors.fill: parent
                        onClicked: Qt.openUrlExternally(website)
                    }
                }
            }

            SeparatorLabel {
                id: separator2

                width: parent.width
                text: qsTr("About")
            }

            Label {
                width: parent.width
                
                
                wrapMode: Text.WordWrap
                text: (about == "") ? qsTr("No info") : about.replace(/\r/g, "")
                visible: !(text == "")
            }
        }
    }

    BusyDialog {
        anchors.centerIn: parent
        visible: videoCount == ""
    }
}
