#include "dailymotion.h"
#include "json.h"
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QString>
#include <QUrl>
#include <QStringList>
#include <QDateTime>
#include <QTimer>

#define EXCLUDED_CHARS " \n\t#[]{}=+$&*()<>@|',/!\":;?"

Dailymotion::Dailymotion(QObject *parent) :
    QObject(parent), clientId("c1eda48dcc5e41f23cbb"), clientSecret("66d40e4bf9949dea9313722059acb53e53017f8b") {
    emit clientIdChanged();
    emit clientSecretChanged();
    refreshTimer = new QTimer(this);
    refreshTimer->setSingleShot(false);
    connect(refreshTimer, SIGNAL(timeout()), this, SLOT(refreshAccessToken()));
}

void Dailymotion::setNetworkAccessManager(QNetworkAccessManager *manager) {
    nam = manager;
}

void Dailymotion::setBusyMessage(const QString &message) {
    busyMessage = message;
    emit busyMessageChanged();
}

void Dailymotion::clearBusyMessage() {
    busyMessage = QString();
    emit busyMessageChanged();
}

void Dailymotion::setAccount(const QString &user, const QString &aToken, const QString &rToken, int expiry) {
    if ((!rToken.isEmpty()) && (expiry == 0)) {
        refreshAccessToken(rToken);
    }
    else {
        setUsername(user);
        setAccessToken(aToken);
        setRefreshToken(rToken, expiry);
    }
}

void Dailymotion::setUsername(const QString &user) {
    username = user;
    emit usernameChanged();
}

void Dailymotion::setAccessToken(const QString &token) {
    accessToken = token;
    emit accessTokenChanged();
    if ((accessToken.isEmpty()) || (!refreshTimer->isActive())) {
        emit userSignedInChanged();
    }
}

void Dailymotion::setRefreshToken(const QString &token, int expiry) {
    refreshToken = token;
    if ((!token.isEmpty()) && (expiry > 0)) {
        setTokenExpiry(expiry);
    }
}

void Dailymotion::setTokenExpiry(int expiry) {
    tokenExpiry = expiry * 900;
    refreshTimer->start(tokenExpiry);
}

void Dailymotion::refreshAccessToken(QString token) {
    if (token.isEmpty()) {
        token = refreshToken;
    }
    else {
        setBusyMessage(tr("Refreshing access token..."));
    }
    QUrl url("https://api.dailymotion.com/oauth/token");
    QByteArray data("grant_type=refresh_token&client_id=" + clientId.toAscii()
                    + "&client_secret=" + clientSecret.toAscii()
                    + "&refresh_token=" + token.toAscii());
    QNetworkRequest request(url);
    QNetworkReply* reply = nam->post(request, data);
    connect(reply, SIGNAL(finished()), this, SLOT(checkForToken()));
}

void Dailymotion::checkForToken() {
    clearBusyMessage();
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit alert(tr("Cannot sign in to Dailymotion"));
        return;
    }

    using namespace QtJson;

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();
    if (!ok) {
        QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
        emit alert(statusText);
    }
    else {
        QVariantMap error = result.value("error").toMap();
        if (!error.isEmpty()) {
            QString errorString = error.value("message").toString();
            emit alert(errorString);
        }
        else {
            QString user = result.value("uid").toString();
            QString aToken = result.value("access_token").toString();
            QString rToken = result.value("refresh_token").toString();
            int expiry = result.value("expires_in").toInt();
            if ((aToken.isEmpty()) || (rToken.isEmpty()) || (expiry == 0)) {
                emit alert(tr("Cannot sign in to Dailymotion"));
            }
            else {
                signedIn(user, aToken, rToken, expiry);
            }
        }
    }
    reply->deleteLater();
}

void Dailymotion::getVideoUrl(const QString &id) {
    QString url = "http://iphone.dailymotion.com/video/" + id;
    QNetworkReply* reply = nam->get(QNetworkRequest(QUrl(url)));
    connect(reply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void Dailymotion::parseVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (reply->error()) {
        emit alert("Unable to play video");
        emit videoUrlError();
    }
    else {
        QString response(reply->readAll());
        QString videoUrl = response.split("type=\"video/x-m4v\" href=\"").at(1).split("\"").at(0);
        if (!videoUrl.startsWith("http")) {
            emit alert(tr("Unable to play video"));
            emit videoUrlError();
        }
        else {
            emit gotVideoUrl(videoUrl);
        }
    }
    reply->deleteLater();
}
