import QtQuick 1.0
import "components"
import "scripts/youtube.js" as YT
import "scripts/createobject.js" as ObjectCreator

Page {
    id: root

    property string username
    property bool isSubscribed
    property string subscriptionId
    property string subscriberCount
    property string videoCount
    property string userThumbnail
    property string about
    property string age
    property string firstName
    property string lastName
    property string gender
    property string website

    function getUserProfile(user) {
        if (user == "default") {
            subscriptionButton.visible = false;
        }
        else {
            username = user;

            var i = 0;
            while ((!isSubscribed) && (i < youtubeSubscriptionsModel.count)) {
                if (youtubeSubscriptionsModel.get(i).title == username) {
                    subscriptionId = youtubeSubscriptionsModel.get(i).subscriptionId.split(":")[5];
                    isSubscribed = true;
                }
                i++;
            }
        }
        YT.getUserProfile(user);
    }

    function showUploadDialog() {
        var uploadDialog = ObjectCreator.createObject(Qt.resolvedUrl("YTUploadDialog.qml"), appWindow);
        uploadDialog.open();
    }

    function goToWatchLaterPlaylist() {
        pageStack.push(Qt.resolvedUrl("YTPlaylistVideosPage.qml"));
        pageStack.currentPage.showWatchLaterPlaylist();
    }

    title: qsTr("Profile")
    tools: Menu {

        menuItems: [
            MenuItem {
                text: qsTr("Upload video")
                onClicked: showUploadDialog()
            },

            MenuItem {
                text: qsTr("View 'watch later' playlist")
                onClicked: goToWatchLaterPlaylist()
            }
        ]
    }

    Connections {
        target: YouTube
        onSubscribed: subIndicator.visible = false
        onUnsubscribed: subIndicator.visible = false
    }

    Flickable {
        id: flicker

        anchors.fill: parent
        clip: true
        contentWidth: parent.width
        contentHeight: 200 + col1.height + col2.height
        flickableDirection: Flickable.VerticalFlick
        visible: !(videoCount == "")

        Text {
            id: titleText

            anchors { top: parent.top; left: parent.left; right: parent.right; margins: 10 }
            font.pixelSize: _STANDARD_FONT_SIZE
            font.bold: true
            color: _TEXT_COLOR
            wrapMode: Text.WordWrap
            text: username
        }

        Image {
            id: cover

            width: 160
            height: 120
            anchors { left: titleText.left; top: titleText.bottom; topMargin: 10 }
            source: userThumbnail
            sourceSize.width: width
            sourceSize.height: height
            smooth: true
            onStatusChanged: if (status == Image.Error) source = "images/avatar.png";
        }

        Button {
            id: subscriptionButton

            anchors { right: parent.right; top: cover.bottom; margins: 10 }
            text: isSubscribed ? qsTr("Unsubscribe") : qsTr("Subscribe")
            visible: (YouTube.userSignedIn) && (username != YouTube.username)
            onClicked: {
                subIndicator.visible = true;
                if (isSubscribed) {
                    YouTube.unsubscribeToChannel(subscriptionId);
                }
                else {
                    YouTube.subscribeToChannel(username);
                }
            }
        }

        Column {
            id: col1

            anchors { left: cover.right; leftMargin: 10; top: cover.top; right: parent.right; rightMargin: 10 }

            Text {
                width: parent.width
                font.pixelSize: _STANDARD_FONT_SIZE
                color: _TEXT_COLOR
                elide: Text.ElideRight
                text: videoCount + " " + qsTr("uploads")
                visible: !(text == "")
            }

            Text {
                width: parent.width
                font.pixelSize: _STANDARD_FONT_SIZE
                color: _TEXT_COLOR
                elide: Text.ElideRight
                text: subscriberCount + " " + qsTr("subscribers")
                visible: !(text == "")
            }
        }

        SeparatorLabel {
            id: separator

            anchors { left: parent.left; right: parent.right; top: subscriptionButton.visible ? subscriptionButton.bottom : cover.bottom; margins: 10 }
            text: qsTr("Profile")
        }

        Column {
            id: col2

            anchors { left: parent.left; right: parent.right; top: separator.bottom; margins: 10 }

            Text {
                width: parent.width
                font.pixelSize: _STANDARD_FONT_SIZE
                color: _TEXT_COLOR
                text: qsTr("Name") + ": " + firstName + " " + lastName
                visible: !(firstName == "")
            }

            Text {
                width: parent.width
                font.pixelSize: _STANDARD_FONT_SIZE
                color: _TEXT_COLOR
                text: qsTr("Age") + ": " + age
                visible: !(age == "")
            }

            Text {
                width: parent.width
                font.pixelSize: _STANDARD_FONT_SIZE
                color: _TEXT_COLOR
                text: gender == "m" ? qsTr("Gender") + ": " + qsTr("Male") : qsTr("Gender") + ": " + qsTr("Female")
                visible: !(gender == "")
            }

            Row {
                width: parent.width

                Text {
                    id: webLabel

                    font.pixelSize: _STANDARD_FONT_SIZE
                    color: _TEXT_COLOR
                    text: qsTr("Website") + ": "
                    visible: !(website == "")
                }

                Text {
                    width: parent.width - webLabel.width
                    font.pixelSize: _STANDARD_FONT_SIZE
                    color: Settings.activeColor
                    elide: Text.ElideRight
                    text: website
                    opacity: websiteMouseArea.pressed ? 0.5 : 1
                    visible: !(text == "")

                    MouseArea {
                        id: websiteMouseArea

                        anchors.fill: parent
                        onClicked: Qt.openUrlExternally(website)
                    }
                }
            }

            SeparatorLabel {
                id: separator2

                width: parent.width
                text: qsTr("About")
            }

            Text {
                width: parent.width
                font.pixelSize: _STANDARD_FONT_SIZE
                color: _TEXT_COLOR
                wrapMode: Text.WordWrap
                text: (about == "") ? qsTr("No info") : about.replace(/\r/g, "")
                visible: !(text == "")
            }
        }
    }

    BusyDialog {
        anchors.centerIn: parent
        visible: videoCount == ""
    }
}
