import QtQuick 1.0
import "scripts/settings.js" as Settings

Item {
    id: dialog

    property string playbackQuality
    property string downloadQuality
    property string downloadStatus
    property string downloadPath
    property string safeSearch
    property string screenOrientation
    property string mediaPlayer
    property string theme
    property string language

    property variant playbackSettings : [qsTr("Mobile"), qsTr("High Quality")]
    property variant downloadSettings : [qsTr("Mobile"), qsTr("High Quality"), "360p", "480p", "720p"]
    property variant downloadStatusSettings : [qsTr("Automatically"), qsTr("Manually")]
    property variant safeSearchSettings : [qsTr("On"), qsTr("Off")]
    property variant orientationSettings : [qsTr("Automatic"), qsTr("Landscape"), qsTr("Portrait")]
    property variant mediaPlayerSettings
    property variant themeSettings : [qsTr("Light"), qsTr("Dark")]
    property variant languageSettings : [qsTr("English"), qsTr("Finnish"), qsTr("German"), qsTr("Italian")]

    property string settingToBeChanged

    signal close

    function getSettings() {
        /* Retrieve relevent settings from the database
          and populate the dialog */

        mediaPlayerSettings = Controller.getInstalledMediaPlayers();
        playbackQuality = Settings.getSetting("playbackQuality");
        downloadQuality = Settings.getSetting("downloadQuality");
        var status = Settings.getSetting("downloadStatus");
        if (status == qsTr("Queued")) {
            downloadStatus = qsTr("Automatically");
        }
        else {
            downloadStatus = qsTr("Manually");
        }
        var safe = Settings.getSetting("safeSearch");
        if (safe == "strict") {
            safeSearch = qsTr("On");
        }
        else {
            safeSearch = qsTr("Off");
        }
        downloadPath = Settings.getSetting("downloadPath");
        screenOrientation = Settings.getSetting("screenOrientation");
        mediaPlayer = Settings.getSetting("mediaPlayer");
        theme = Settings.getSetting("theme");
        language = Settings.getSetting("language");
    }

    function saveSettings() {
        /* Save all settings to the database */

        Settings.setSetting("playbackQuality", playbackQuality);
        Settings.setSetting("downloadQuality", downloadQuality);
        Settings.setSetting("screenOrientation", screenOrientation);
        Settings.setSetting("mediaPlayer", mediaPlayer);
        Settings.setSetting("downloadPath", downloadPath);
        Settings.setSetting("theme", theme);
        Settings.setSetting("language", language);
        Controller.setOrientation(screenOrientation);
        Controller.setMediaPlayer(mediaPlayer);
        YouTube.setPlaybackQuality(playbackQuality);
        DownloadManager.setDownloadQuality(downloadQuality);
        if (downloadStatus == qsTr("Automatically")) {
            Settings.setSetting("downloadStatus", qsTr("Queued"));
        }
        else {
            Settings.setSetting("downloadStatus", qsTr("Paused"));
        }
        if (safeSearch == qsTr("On")) {
            Settings.setSetting("safeSearch", "strict");
        }
        else {
            Settings.setSetting("safeSearch", "none");
        }
        cuteTubeTheme = theme;
        displayMessage(qsTr("Your settings have been saved"));
        close();
    }

    function showDownloadPathDialog() {
        settingToBeChanged = qsTr("Download Path");
        settingsLoader.source = "FileChooserDialog.qml";
        settingsLoader.item.title = qsTr("Download Location");
        settingsLoader.item.showButton = true;
        settingsLoader.item.showFiles = false;
        settingsLoader.item.folder = downloadPath;
        dialog.state = "showChild"
    }

    function showSettingsList(title, list, currentSetting) {
        /* Show the settings list dialog */

        settingToBeChanged = title;
        settingsLoader.source = "SettingsListDialog.qml";
        settingsLoader.item.setSettingsList(title, list, currentSetting);
        dialog.state = "showChild"
    }

    function changeSetting(setting) {
        /* Change the appropriate setting in the dialog */

        if (settingToBeChanged == qsTr("Playback Quality")) {
            playbackQuality = setting;
        }
        else if (settingToBeChanged == qsTr("Download Quality")) {
            downloadQuality = setting;
        }
        else if (settingToBeChanged == qsTr("Start Downloads")) {
            downloadStatus = setting;
        }
        else if (settingToBeChanged == qsTr("Download Path")) {
            downloadPath = setting;
        }
        else if (settingToBeChanged == qsTr("Safe Search")) {
            safeSearch = setting;
        }
        else if (settingToBeChanged == qsTr("Screen Orientation")) {
            screenOrientation = setting;
        }
        else if (settingToBeChanged == qsTr("Media Player")) {
            mediaPlayer = setting;
        }
        else if (settingToBeChanged == qsTr("Theme")) {
            theme = setting;
        }
        else if (settingToBeChanged == qsTr("Language")) {
            language = setting;
        }
    }

    function clearSearches() {
        /* Delete all saved searches from the database */

        if (Settings.clearSearches() == "OK") {
            displayMessage(qsTr("Your saved searches have been cleared"));
        }
        else {
            displayMessage(qsTr("Database error. Unable to clear searches"));
        }
    }

    width: parent.width
    anchors { right: parent.left; top: parent.top; bottom: parent.bottom }

    Component.onCompleted: getSettings()

    Connections {
        target: dialog.parent

        onDialogClose: {
            dialog.state = "";
            dialog.destroy(600);
        }
    }

    Loader {
        id: settingsLoader

        width: parent.width
        anchors { right: parent.left; top: parent.top; bottom: parent.bottom }

        Connections {
            target: settingsLoader.item
            onClose: dialog.state = "show"
            onSettingChosen: changeSetting(setting)
        }
    }

    Item {
        id: background

        anchors.fill: dialog

        Rectangle {
            anchors.fill: background
            color: _BACKGROUND_COLOR
            opacity: 0.5
            smooth: true
        }

        Text {
            id: title

            anchors { horizontalCenter: background.horizontalCenter; top: background.top; topMargin: 10 }
            font.pixelSize: _SMALL_FONT_SIZE
            color: _TEXT_COLOR
            text: qsTr("Settings")
            smooth: true
        }

        Flickable {
            id: flicker

            anchors { fill: background; topMargin: 50; leftMargin: 10; rightMargin: 10; bottomMargin: 90 }
            contentWidth: flicker.width
            contentHeight: divider.height
            clip: true
            flickableDirection: Flickable.VerticalFlick
            boundsBehavior: Flickable.DragOverBounds

            Rectangle {
                id: divider

                width: 1
                height: settingColumn.height
                anchors { top: parent.top; horizontalCenter: parent.horizontalCenter }
                color: _ACTIVE_COLOR_LOW


                Column {
                    id: settingColumn

                    anchors { right: divider.left; rightMargin: 10; top: divider.top }
                    spacing: 10

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Playback quality")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Download quality")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Start downloads")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Download location")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Safe Search")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Screen orientation")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Media player")
                        smooth: true
                    }

                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Theme")
                        smooth: true
                    }


                    Text {
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: _TEXT_COLOR
                        horizontalAlignment: Text.AlignRight
                        text: qsTr("Language")
                        smooth: true
                    }

                }

                Column {
                    id: valueColumn

                    anchors { left: divider.right; leftMargin: 10; top: divider.top }
                    spacing: 10

                    Text {
                        id: playbackText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: playbackMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: playbackQuality
                        smooth: true

                        MouseArea {
                            id: playbackMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Playback Quality"), playbackSettings, playbackQuality)
                        }
                    }

                    Text {
                        id: downloadText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: downloadMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: downloadQuality
                        smooth: true

                        MouseArea {
                            id: downloadMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Download Quality"), downloadSettings, downloadQuality)
                        }
                    }

                    Text {
                        id: downloadStatusText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: downloadStatusMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: downloadStatus
                        smooth: true

                        MouseArea {
                            id: downloadStatusMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Start Downloads"), downloadStatusSettings, downloadStatus)
                        }
                    }

                    Text {
                        id: downloadPathText

                        width: (dialog.parent.state == "") ? 240 : 180
                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: downloadPathMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        elide: Text.ElideRight
                        text: downloadPath
                        smooth: true

                        MouseArea {
                            id: downloadPathMouseArea

                            width: downloadPathText.width
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showDownloadPathDialog()
                        }
                    }

                    Text {
                        id: safeSearchText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: safeSearchMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: safeSearch
                        smooth: true

                        MouseArea {
                            id: safeSearchMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Safe Search"), safeSearchSettings, safeSearch)
                        }
                    }

                    Text {
                        id: orientationText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: orientationMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: screenOrientation
                        smooth: true

                        MouseArea {
                            id: orientationMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Screen Orientation"), orientationSettings, screenOrientation)
                        }
                    }

                    Text {
                        id: playerText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: playerMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: mediaPlayer
                        smooth: true

                        MouseArea {
                            id: playerMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Media Player"), mediaPlayerSettings, mediaPlayer)
                        }
                    }

                    Text {
                        id: themeText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: themeMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: theme
                        smooth: true

                        MouseArea {
                            id: themeMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Theme"), themeSettings, theme)
                        }
                    }

                    Text {
                        id: languageText

                        font.pixelSize: _STANDARD_FONT_SIZE
                        color: languageMouseArea.pressed ? _ACTIVE_COLOR_HIGH : _ACTIVE_COLOR_LOW
                        text: language
                        smooth: true

                        MouseArea {
                            id: languageMouseArea

                            width: 200
                            height: 30
                            anchors.centerIn: parent
                            onClicked: showSettingsList(qsTr("Language"), languageSettings, language)
                        }
                    }
                }
            }
        }

        PushButton {
            id: clearSearchesButton

            width: (dialog.parent.state == "") ? 300 : dialog.width - 20
            anchors { left: background.left; leftMargin: 10; bottom: background.bottom; bottomMargin: (dialog.parent.state == "") ? 10 : 90 }
            showIcon: false
            showText: true
            name: qsTr("Clear saved searches")
            onButtonClicked: clearSearches()
        }

        PushButton {
            id: saveButton

            width: (dialog.parent.state == "") ? 150 : dialog.width - 20
            anchors { right: background.right; bottom: background.bottom; margins: 10 }
            icon: "ui-images/tick.png"

            Connections {
                onButtonClicked: saveSettings()
            }
        }
    }

    CloseButton {
        onButtonClicked: close()
    }

    MouseArea {

        property real xPos

        z: -1
        anchors.fill: dialog
        onPressed: xPos = mouseX
        onReleased: {
            if (xPos - mouseX > 100) {
                close();
            }
        }
    }

    states: [
        State {
            name: "show"
            AnchorChanges { target: dialog; anchors.right: parent.right }
        },

        State {
            name: "showChild"
            AnchorChanges { target: dialog; anchors { left: parent.right; right: undefined } }
        }
    ]

    transitions: Transition {
        AnchorAnimation { easing.type: Easing.OutQuart; duration: 500 }
    }
}
