/*
 *  qorientationinfo_p.h
 *  author: Timo Härkönen
 */

#ifndef QORIENTATIONINFO_P_H
#define QORIENTATIONINFO_P_H

#include "qorientationinfo.h"

#include <QtDBus>
#include <QtCore>

#include <mce/mode-names.h>
#include <mce/dbus-names.h>

namespace Maemo {

class QOrientationInfoPrivate
{
    MAEMO_DECLARE_PUBLIC(QOrientationInfo);

public:
    QOrientationInfoPrivate();
    ~QOrientationInfoPrivate();
    void orientationUpdate(const QString &rotation, const QString &stand, const QString &face, int x, int y, int z);
    void startListening();
    void stopListening();
    bool isListening() const;

private:
    bool m_listening;
};

/* begin private implementation */

QOrientationInfoPrivate::QOrientationInfoPrivate() :
    m_listening(false)
{

}

QOrientationInfoPrivate::~QOrientationInfoPrivate()
{

}

void QOrientationInfoPrivate::orientationUpdate(const QString &rotation, const QString &stand, const QString &face, int x, int y, int z)
{
    MAEMO_PUBLIC(QOrientationInfo);

    Q_UNUSED(rotation)
    Q_UNUSED(stand)
    Q_UNUSED(face)

    qreal pitch = atan2(static_cast<qreal>(x), static_cast<qreal>(-y));
    qreal roll = atan2(static_cast<qreal>(y), static_cast<qreal>(-z));

    emit pub->orientationChanged(pitch, roll);
}

void QOrientationInfoPrivate::startListening()
{
    if(!m_listening) {
        //request to start accelerometer
        QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                                                         MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ));
        MAEMO_PUBLIC(QOrientationInfo);

        //connect accelerometer signal to orientation update slot
        QDBusConnection::systemBus().connect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
            MCE_DEVICE_ORIENTATION_SIG, pub, SLOT(orientationUpdate(QString, QString, QString, int, int, int)));

        m_listening = true;
    }
}

void QOrientationInfoPrivate::stopListening()
{
    if(m_listening) {
        MAEMO_PUBLIC(QOrientationInfo);

        //disconnect update slot from accelerometer signal
        QDBusConnection::systemBus().disconnect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
            MCE_DEVICE_ORIENTATION_SIG, pub, SLOT(orientationUpdate(QString,QString,QString, int, int, int)));

        //request to stop accelerometer
        QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                                                             MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));

        m_listening = false;
    }
}

bool QOrientationInfoPrivate::isListening() const
{
    return m_listening;
}

} // namespace
#endif

