/**********************************************************************************************
    Copyright (C) 2009 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

**********************************************************************************************/

#ifndef CBACKLIGHT_H
#define CBACKLIGHT_H

#include <QObject>
#include <QDir>
#include <QTime>

class QTimer;


class CBacklight : public QObject
{
    Q_OBJECT;
    public:
        virtual ~CBacklight();

        static CBacklight& self(){return *m_self;}

        /// set the brightness in levels of 0..10
        /**
            0 backlight and screen off

            1 lowest backlight

            10 brightest backlight
        */
        void setBrightness(int level);

        /// get the brightness level 0..10
        /**
            0 backlight and screen off

            1 lowest backlight

            10 brightest backlight
        */
        int getBrightness();

        /// toggle between lcd off and last brightness level
        void toggle();

        /// set time until "lcd off"
        /**
            @param t time in seconds, -1 switches to manual toggle mode
        */
        void setTimeout(int t);

        /// read current "lcd off" timeout
        int getTimeout();

        /// read expired time until "lcd off"
        int getTimer();

    protected:
        bool eventFilter(QObject *obj, QEvent *ev);

    private slots:
        void slotTimeout();

    private:
        friend class CCanvas;

        CBacklight(QObject * parent);

        void resetTimeout();


        static CBacklight * m_self;

        QDir dirDeviceBacklight;
        QDir dirDeviceLcd;

        int maxBrightness;
        int actBrightness;
        int oldBrightness;

        int levels[11];

        inline int level2brightness(int level)
        {
            if(level > 10) level = 10;
            if(level < 0)  return 0;
            return levels[level];
        }

        inline int brightness2level(int level)
        {
            if(level < 1)        return 0;

            int n = 1;
            int step = maxBrightness / 10;
            for(int i = step; i < maxBrightness; i += step, ++n){
                if(level < (i + 1)) return n;
            }
            return 10;
        }

        QTimer * timer;
        QTime time;
        int timeout;

        QTimer * timerFalseTrigger;

        bool noService;
};

#endif //CBACKLIGHT_H

