/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QtTest>
#include <QApplication>
#include <QProcess>
#include <QDBusConnection>
#include <QtConcurrentRun>

#include <qimsysabstractipcobject.h>
#include "qimsysobject.h"

static const int loopCount = 3;

class QimsysAbstractIpcObjectTest : public QObject
{
    Q_OBJECT
public:
    QimsysAbstractIpcObjectTest(QObject *parent = 0)
        : QObject(parent)
    {
        bool ret = QDBusConnection::sessionBus().registerService(QIMSYS_DBUS_SERVICE);
        if (ret) {
            object = new AbstractIpcObject(this, QimsysAbstractIpcObject::Server);
            QStringList args = QApplication::arguments();
            args.removeFirst();
            client = QtConcurrent::run(QProcess::execute, QApplication::applicationFilePath(), args);
        } else {
            object = new AbstractIpcObject(this, QimsysAbstractIpcObject::Client);
        }
        object->init();
    }

    ~QimsysAbstractIpcObjectTest() {
        if (client.isStarted()) {
            client.waitForFinished();
        }
    }

public slots:
    void initTestCase() {
        qDebug() << object->type();
    }

    void init() {
        expected.clear();
        signalReceived = false;
    }

    void cleanup() {
        signalReceived = false;
        wait();
    }

protected slots:
    void boolValueChanged(bool value);
    void intValueChanged(int value);
    void uintValueChanged(uint value);
    void qulonglongValueChanged(qulonglong value);

    void stringValueChanged(QString value);
    void pointValueChanged(QPoint value);
    void sizeValueChanged(QSize value);
    void rectValueChanged(QRect value);
    void fontValueChanged(QFont value);

    void enumValueChanged(AbstractIpcObject::Enum value);

private slots:
    void boolValue_data();
    void boolValue();
    void intValue_data();
    void intValue();
    void uintValue_data();
    void uintValue();
    void qulonglongValue_data();
    void qulonglongValue();

    void stringValue_data();
    void stringValue();
    void pointValue_data();
    void pointValue();
    void sizeValue_data();
    void sizeValue();
    void rectValue_data();
    void rectValue();
    void fontValue_data();
    void fontValue();

    void enumValue_data();
    void enumValue();

private:
    void wait() {
        QTime timer;
        timer.start();
        while (!signalReceived) {
            QTest::qWait(50);
            if (timer.elapsed() > 500) break;
        }
    }
private:
    AbstractIpcObject *object;
    QFuture<int> client;
    bool signalReceived;
    QVariant expected;
};

#define TEST( T, func, ... ) \
    void QimsysAbstractIpcObjectTest::func##ValueChanged( T func##Value ) \
    { \
        T value = expected.value<T>(); \
        QCOMPARE( func##Value, value ); \
        signalReceived = true; \
    } \
    void QimsysAbstractIpcObjectTest::func##Value_data() \
    { \
        QTest::addColumn<T>( "data" ); \
        \
        __VA_ARGS__ \
    } \
    void QimsysAbstractIpcObjectTest::func##Value() \
    { \
        QFETCH( T, data ); \
        expected.setValue( data ); \
        connect( object, SIGNAL( func##ValueChanged(T)), this, SLOT( func##ValueChanged(T)) ); \
        switch( object->type() ) \
        { \
        case QimsysAbstractIpcObject::Server: \
            signalReceived = false; \
            wait(); \
            QCOMPARE( object->func##Value(), data ); \
            break; \
        case QimsysAbstractIpcObject::Client: \
            signalReceived = false; \
            object->func##Value( data ); \
            QCOMPARE( object->func##Value(), data ); \
            wait(); \
            break;\
        } \
        disconnect( object, SIGNAL( func##ValueChanged(T)), this, SLOT( func##ValueChanged(T)) ); \
    }

TEST(bool, bool,
     QTest::newRow("True") << true;
     QTest::newRow("False") << false;
    )
TEST(int, int,
     for (int i = 0; i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << i;
}
    )
TEST(uint, uint,
     for (int i = 0; i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << (uint)i;
}
    )
TEST(qulonglong, qulonglong,
     for (int i = 0; i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << (qulonglong)i;
}
    )

TEST(QString, string,
     QString str("I am a pen.");
     for (int i = 0; i < str.length() && i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << str.left(i);
}
    )
TEST(QPoint, point,
     for (int i = 0; i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << QPoint(i, i);
}
    )
TEST(QSize, size,
     for (int i = 0; i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << QSize(i, i);
}
    )
TEST(QRect, rect,
     for (int i = 0; i < loopCount; i++)
{
QTest::newRow(QString::number(i).toUtf8().data()) << QRect(i, i, i, i);
}
    )
TEST(QFont, font,
     QTest::newRow("default") << QFont();
     QTest::newRow("arial") << QFont("Arial");
    )

TEST(AbstractIpcObject::Enum, enum,
     QTest::newRow("0") << AbstractIpcObject::Enum0;
     QTest::newRow("1") << AbstractIpcObject::Enum1;
     QTest::newRow("2") << AbstractIpcObject::Enum2;
     QTest::newRow("3") << AbstractIpcObject::Enum3;
    )

#undef TEST

QTEST_MAIN(QimsysAbstractIpcObjectTest)
#include "tst_qimsysabstractobject.moc"
