/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QtTest>
#include <QApplication>
#include <QProcess>
#include <QDBusConnection>
#include <QtConcurrentRun>

#include "qimsyspreedit.h"

class QimsysPreeditTest : public QObject
{
    Q_OBJECT
public:
    QimsysPreeditTest(QObject *parent = 0)
        : QObject(parent)
    {
        bool ret = QDBusConnection::sessionBus().registerService(QIMSYS_DBUS_SERVICE);
        if (ret) {
            preedit = new QimsysPreedit(this, QimsysPreedit::Server);
            client = QtConcurrent::run(QProcess::execute, QApplication::applicationFilePath());
        } else {
            preedit = new QimsysPreedit(this, QimsysPreedit::Client);
        }
        preedit->init();
    }

    ~QimsysPreeditTest() {
        if (client.isStarted()) {
            client.waitForFinished();
        }
    }

public slots:
    void initTestCase() {
        qDebug() << preedit->type();
    }

    void init() {
        switch (preedit->type()) {
        case QimsysPreedit::Server:
            break;
        case QimsysPreedit::Client:
            break;
        }
        expected.clear();
        signalReceived = false;
    }

    void cleanup() {
        signalReceived = false;
        wait();
    }

protected slots:
    void itemsChanged(const QimsysPreeditItemList &items) {
        QCOMPARE(items, expected.value<QimsysPreeditItemList>());
        signalReceived = true;
    }
    void rectChanged(const QRect &rect) {
        QCOMPARE(rect, expected.toRect());
        signalReceived = true;
    }

    void fontChanged(const QFont &font) {
        QCOMPARE(font, expected.value<QFont>());
    }

private slots:
    void setItems_data() {
        QTest::addColumn<QimsysPreeditItemList>("data");

        QimsysPreeditItemList items;
        QString alphabet = QString::fromUtf8("abcdefgあいうえお");
        for (int i = 1; i < alphabet.length(); i++) {
            QimsysPreeditItem item;
            item.text = alphabet.left(i);;
            item.cursor = 0;
            item.selection = 0;
            item.underline = QTextCharFormat::NoUnderline;
            items.append(item);
            QTest::newRow(QString("alphabet(%1)").arg(i).toAscii().data()) << items;
        }
    }

    void setItems() {
        QFETCH(QimsysPreeditItemList, data);
        expected = qVariantFromValue(data);
        connect(preedit, SIGNAL(itemsChanged(QimsysPreeditItemList)), this, SLOT(itemsChanged(QimsysPreeditItemList)));
        switch (preedit->type()) {
        case QimsysPreedit::Server:
            signalReceived = false;
            wait();
            QCOMPARE(preedit->items(), data);
            break;
        case QimsysPreedit::Client:
            signalReceived = false;
            preedit->setItems(data);
            QCOMPARE(preedit->items(), data);
            wait();
            break;
        }
        disconnect(preedit, SIGNAL(itemsChanged(QimsysPreeditItemList)), this, SLOT(itemsChanged(QimsysPreeditItemList)));
    }

    void setRect_data() {
        QTest::addColumn<QRect>("data");

        for (int i = 0; i < 10; i++) {
            QTest::newRow(QString::number(i).toAscii().data()) << QRect(i, i, i, i);
        }
    }

    void setRect() {
        QFETCH(QRect, data);
        expected = data;
        connect(preedit, SIGNAL(rectChanged(QRect)), this, SLOT(rectChanged(QRect)));
        switch (preedit->type()) {
        case QimsysPreedit::Server:
            signalReceived = false;
            wait();
            QCOMPARE(preedit->rect(), data);
            break;
        case QimsysPreedit::Client:
            signalReceived = false;
            preedit->setRect(data);
            QCOMPARE(preedit->rect(), data);
            wait();
            break;
        }
        disconnect(preedit, SIGNAL(rectChanged(QRect)), this, SLOT(rectChanged(QRect)));
    }

    void setFont_data() {
        QTest::addColumn<QFont>("data");

        QTest::newRow("") << QFont();
        QTest::newRow("Arial") << QFont("Arial");
    }

    void setFont() {
        QFETCH(QFont, data);
        expected = data;
        connect(preedit, SIGNAL(fontChanged(QFont)), this, SLOT(fontChanged(QFont)));
        switch (preedit->type()) {
        case QimsysPreedit::Server:
            signalReceived = false;
            wait();
            QCOMPARE(preedit->font(), data);
            break;
        case QimsysPreedit::Client:
            signalReceived = false;
            preedit->setFont(data);
            QCOMPARE(preedit->font(), data);
            wait();
            break;
        }
        disconnect(preedit, SIGNAL(fontChanged(QFont)), this, SLOT(fontChanged(QFont)));
    }

private:
    void wait() {
        QTime timer;
        timer.start();
        while (!signalReceived) {
            QTest::qWait(50);
            if (timer.elapsed() > 250) break;
        }
    }
private:
    QimsysPreedit *preedit;
    QFuture<int> client;
    bool signalReceived;
    QVariant expected;
};

QTEST_MAIN(QimsysPreeditTest)
#include "tst_qimsyspreedit.moc"
