/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QtTest>
#include <QApplication>
#include <QProcess>
#include <QDBusConnection>
#include <QtConcurrentRun>

#include "qimsyscandidates.h"

class QimsysCandidatesTest : public QObject
{
    Q_OBJECT
public:
    QimsysCandidatesTest(QObject *parent = 0)
        : QObject(parent)
    {
        bool ret = QDBusConnection::sessionBus().registerService(QIMSYS_DBUS_SERVICE);
        if (ret) {
            candidates = new QimsysCandidates(this, QimsysCandidates::Server);
            client = QtConcurrent::run(QProcess::execute, QApplication::applicationFilePath());
        } else {
            candidates = new QimsysCandidates(this, QimsysCandidates::Client);
        }
        candidates->init();
    }

    ~QimsysCandidatesTest() {
        if (client.isStarted()) {
            client.waitForFinished();
        }
    }

public slots:
    void initTestCase() {
        qDebug() << candidates->type();
    }

    void init() {
        switch (candidates->type()) {
        case QimsysCandidates::Server:
            break;
        case QimsysCandidates::Client:
            break;
        }
        expected.clear();
        signalReceived = false;
    }

    void cleanup() {
        signalReceived = false;
        wait();
    }

protected slots:
    void candidatesChanged(const QimsysConversionItemList &candidates) {
        QCOMPARE(candidates, expected.value<QimsysConversionItemList>());
        signalReceived = true;
    }

    void currentIndexChanged(int currentIndex) {
        QCOMPARE(currentIndex, expected.toInt());
        signalReceived = true;
    }

private slots:
    void setCandidates_data() {
        QTest::addColumn<QimsysConversionItemList>("data");

        QimsysConversionItemList candidates;
        QString alphabet = "abcde";
        for (int i = 0; i < alphabet.length() + 1; i++) {
            QTest::newRow(QString("candidates(%1)").arg(i).toAscii().data()) << candidates;
            QimsysConversionItem elem;
            elem.from = QLatin1String("from:") + alphabet.left(i + 1);
            elem.to = QLatin1String("to:") + alphabet.left(i + 1);
            candidates.append(elem);
        }
    }

    void setCandidates() {
        QFETCH(QimsysConversionItemList, data);
        expected.setValue(data);
        connect(candidates, SIGNAL(candidatesChanged(QimsysConversionItemList)), this, SLOT(candidatesChanged(QimsysConversionItemList)));
        switch (candidates->type()) {
        case QimsysCandidates::Server:
            signalReceived = false;
            wait();
            QCOMPARE(candidates->candidates(), data);
            break;
        case QimsysCandidates::Client:
            signalReceived = false;
            candidates->setCandidates(data);
            QCOMPARE(candidates->candidates(), data);
            wait();
            break;
        }
        disconnect(candidates, SIGNAL(candidatesChanged(QimsysConversionItemList)), this, SLOT(candidatesChanged(QimsysConversionItemList)));
    }

    void setCurrentIndex_data() {
        QTest::addColumn<int>("data");

        for (int i = 0; i < 2; i++) {
            QTest::newRow(QString::number(i).toAscii().data()) << i;
        }
    }

    void setCurrentIndex() {
        QFETCH(int, data);
        expected = data;
        connect(candidates, SIGNAL(currentIndexChanged(int)), this, SLOT(currentIndexChanged(int)));
        switch (candidates->type()) {
        case QimsysCandidates::Server:
            signalReceived = false;
            wait();
            QCOMPARE(candidates->currentIndex(), data);
            break;
        case QimsysCandidates::Client:
            signalReceived = false;
            candidates->setCurrentIndex(data);
            QCOMPARE(candidates->currentIndex(), data);
            wait();
            break;
        }
        disconnect(candidates, SIGNAL(currentIndexChanged(int)), this, SLOT(currentIndexChanged(int)));
    }

private:
    void wait() {
        QTime timer;
        timer.start();
        while (!signalReceived) {
            QTest::qWait(50);
            if (timer.elapsed() > 250) break;
        }
    }
private:
    QimsysCandidates *candidates;
    QFuture<int> client;
    bool signalReceived;
    QVariant expected;
};

QTEST_MAIN(QimsysCandidatesTest)
#include "tst_qimsyscandidates.moc"
