/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QtTest>
#include <QtGui>

#include "qimsysdebug.h"
#include "qimsyskeymanager.h"
#include "qimsysapplicationmanager.h"
#include "qimsyspreedit.h"
#include "qimsyscandidates.h"

class JapaneseTest : public QObject
{
    Q_OBJECT
private slots:
    void initTestCase() {
        bool found = false;
        QDir dir(QCoreApplication::applicationDirPath());
        while (!found && dir.cdUp()) {
            foreach(const QString &subdir, dir.entryList(QDir::Dirs)) {
                if (subdir == QLatin1String("bin")) {
                    dir.cd("lib");
                    dir.cd("qimsys");
                    dir.cd("plugins");
                    dir.cd("inputmethods");
                    found = true;
                    break;
                }
            }
        }
        QVERIFY(found);
        QPluginLoader loader(dir.absoluteFilePath("libqimsys-immodule.so"));
        QObject *object = loader.instance();
        QInputContext *ic = 0;
        if (object) {
            QInputContextPlugin* plugin = qobject_cast<QInputContextPlugin*>(object);
            if (plugin && plugin->keys().contains("qimsys")) {
                ic = plugin->create("qimsys");
            } else {
                delete object;
            }
        }
        QVERIFY(ic);
        qApp->setInputContext(ic);

        keyio.init();
        if (keyio.hasError()) {
            QFAIL("Run qimsys by yourself first.");
        }
        manager.init();
        preedit.init();
        candidates.init();
        widget = new QPlainTextEdit;
        widget->show();
#ifdef Q_WS_X11
        qt_x11_wait_for_window_manager(widget);
#endif
        widget->setFocus();
    }

    void init() {
        key("a", Qt::Key_A);
        if (preedit.items().isEmpty()) {
            key(QString::null, Qt::Key_Zenkaku_Hankaku);
        } else {
            key(QString::null, Qt::Key_Return);
        }
        key(QString::null, Qt::Key_F6);
        widget->clear();
    }

    void cleanup() {
        if (QTest::currentTestFailed()) {
            QTest::qWait(50000);
        }
        preedit.commit();
        manager.exec(QimsysApplicationManager::Reset);
//  preedit.reset();
        widget->clear();
        key(QString::null, Qt::Key_Zenkaku_Hankaku);
    }

private slots:
    void functionality_data() {
        QTest::addColumn<QStringList>("sequence");

        QFile file(":/data/functionality.txt");
        QVERIFY(file.open(QIODevice::ReadOnly | QIODevice::Text));
        QTextStream stream(&file);

        QString name;
        QStringList sequence;
        while (!stream.atEnd()) {
            QString line = stream.readLine();
            if (line.isEmpty()) {
                if (name.isNull()) continue;
                QTest::newRow(name.toLatin1().data()) << sequence;
                name.clear();
                sequence.clear();
            } else if (line.startsWith("//")) {
            } else if (name.isNull()) {
                name = line;
            } else {
                sequence.append(line);
            }
        }
        file.close();
    }

    void functionality() {
        QFETCH(QStringList, sequence);

        foreach(const QString& line, sequence) {
            QChar ch = line.at(0);
            QString data = line.mid(2);
            switch (ch.unicode()) {
            case 'I':
                foreach(const QChar& ch, data) {
                    key(ch, 0);
                }
                break;
            case 'P':
                QCOMPARE(preedit.preeditString(), data);
                break;
            case 'p': {
                int min = preedit.cursorPosition();
                min = qMin(min, min + preedit.selectionLength());
                QCOMPARE(preedit.preeditString().mid(min, qAbs(preedit.selectionLength())), data);
                break;
            }
            case 'r': {
                key("Space");
                QimsysConversionItemList list = candidates.candidates();
                QString from = list.first().from;
                for (int i = 0; i < data.length() - from.length(); i++) {
                    key("Shift+Right");
                }
                for (int i = 0; i < from.length() - data.length(); i++) {
                    key("Shift+Left");
                }
                break;
            }
            case 'c': {
                int min = preedit.cursorPosition();
                min = qMin(min, min + preedit.selectionLength());
                QimsysConversionItemList list = candidates.candidates();
                for (int i = 0; i < list.count(); i++) {
                    if (list[i].to == data) {
                        candidates.setCurrentIndex(i);
                        break;
                    }
                }
                QCOMPARE(preedit.preeditString().mid(min, qAbs(preedit.selectionLength())), data);
                break;
            }
            case 'T':
                QCOMPARE(widget->toPlainText(), data.replace("\t", "\n"));
                break;
            case 'K':
                key(data);
                break;
            case 'R':
                QCOMPARE(preedit.preeditString(), data);
                key("Return");
//     QCOMPARE( widget->toPlainText(), data );
                break;
            case 'S':
                QCOMPARE(qAbs(preedit.selectionLength()), data.toInt());
                break;
            case 'C':
                QCOMPARE(preedit.cursorPosition(), data.toInt());
                break;
            default:
                DEBUG() << ch << data;
                break;
            }
        }
    }

private:
    void key(const QString& text, int keycode, int modifiers = Qt::NoModifier) {
        keyio.filter(text, keycode, modifiers, true, false);
        QTest::qWait(10);
        keyio.filter(text, keycode, modifiers, false, false);
        QTest::qWait(10);
    }

    void key(const QString& data) {
        QKeySequence seq(data);
        int keycode = seq[0];
        int modifiers = 0;
        if (keycode & Qt::CTRL) {
            modifiers |= Qt::CTRL;
            keycode &= ~Qt::CTRL;
        }
        if (keycode & Qt::ALT) {
            modifiers |= Qt::ALT;
            keycode &= ~Qt::ALT;
        }
        if (keycode & Qt::SHIFT) {
            modifiers |= Qt::SHIFT;
            keycode &= ~Qt::SHIFT;
        }
//  DEBUG() << data << keycode << modifiers;
        key(QString::null, keycode, modifiers);
    }

    void wait() {
        QTime timer;
        timer.start();
        while (waiting) {
            QTest::qWait(50);
            if (timer.elapsed() > 250) break;
        }
        QVERIFY(!waiting);
    }
private:
    QPlainTextEdit* widget;
    QimsysKeyManager keyio;
    QimsysApplicationManager manager;
    QimsysPreedit preedit;
    QimsysCandidates candidates;
    bool waiting;
    QVariant expected;
};

QTEST_MAIN(JapaneseTest)
#include "main.moc"
