/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "toolbar.h"
#include "ui_toolbar.h"

#include "qimsysdebug.h"
#include "translator.h"
#include "setproperty.h"

#include "qimsysapplicationmanager.h"
#include "qimsyspluginmanager.h"
#include "qimsysconverter.h"
#include "qimsysengine.h"

#include <qbinding.h>

#include <QDesktopWidget>
#include <QMenu>
#include <QMouseEvent>
#include <QSettings>
#include <QTimer>

#include <QStateMachine>
#include <QSignalTransition>
#include <QParallelAnimationGroup>
#include <QSequentialAnimationGroup>
#include <QPropertyAnimation>
#include <QGraphicsOpacityEffect>
#include <QGraphicsBlurEffect>
#include <QEasingCurve>
#include "qboolsignal.h"

class ToolBar::Private : private QObject
{
    Q_OBJECT
public:
    Private(QimsysAbstractPluginObject *object, ToolBar *parent);
    ~Private();

    void setEnter(bool enter);

public slots:
    void settingsUpdated();
    void positionChanged();

private slots:
    void init();
    void setupUi();
    void setupStateMachine();
    void aboutToHide();

    void focusChanged(uint focus);
    void setInputLanguage(QAction *action);
    void inputLanguageChanged(const QString &inputLanguage);
    void currentEngineChanged(const QString &currentEngine);
    void setCurrentEngine(QAction *action);

    void userDictionary();
    void settings();


signals:
    void enterChanged(bool on);
    void focusChanged(bool on);
    void aboutToShow();
    void minimize();
    void normalize();

private:
    ToolBar *q;
    Ui::ToolBar ui;
    QStateMachine *machine;
    QimsysAbstractPluginObject *plugin;

    QimsysApplicationManager manager;
    QimsysEngineDictionary *currentDictionary;

    QActionGroup *langGroup;
    QActionGroup *engineGroup;
    QBoolSignal *enter;
public:
    QPoint lastPos;
    QSize normalSize;
};

ToolBar::Private::Private(QimsysAbstractPluginObject *object, ToolBar *parent)
    : QObject(parent)
    , q(parent)
    , machine(0)
    , plugin(object)
    , currentDictionary(0)
{
    Translator::begin();
    ui.setupUi(q);
    Translator::ui(q);
    Translator::end();

    ui.icon->setPixmap(qApp->windowIcon().pixmap(22));

    connect(ui.userDictionary, SIGNAL(clicked()), this, SLOT(userDictionary()));
    connect(ui.settings, SIGNAL(clicked()), this, SLOT(settings()));

    q->setWindowFlags(Qt::Window | Qt::WindowStaysOnTopHint | Qt::X11BypassWindowManagerHint | Qt::FramelessWindowHint);
    q->setFocusPolicy(Qt::NoFocus);
    q->setWindowOpacity(0.80);

    metaObject()->invokeMethod(this, "init", Qt::QueuedConnection);
    q->setStyleSheet("*{ background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1, stop:0 rgba(255, 255, 255, 255), stop:1 rgba(128, 128, 128, 255)); }"
                     "QMenu::Item:selected { background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1, stop:0 rgba(128, 128, 128, 128), stop:0.33 rgba(0, 0, 0, 255), stop:0.66 rgba(0, 0, 0, 255), stop:1 rgba(128, 128, 128, 128)); }"
                     "QToolButton:pressed { background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1, stop:0 rgba(128, 128, 128, 128), stop:0.5 rgba(0, 0, 0, 255), stop:1 rgba(128, 128, 128, 128)); }"
                     "QToolButton:hover { background-color: qlineargradient(spread:pad, x1:0, y1:0, x2:0, y2:1, stop:0 rgba(128, 128, 128, 128), stop:0.5 rgba(0, 0, 0, 255), stop:1 rgba(128, 128, 128, 128)); }"
                    );
}

ToolBar::Private::~Private()
{
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    settings.setValue("Geometry", q->saveGeometry());
}

void ToolBar::Private::init()
{
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());

    manager.init();
    connect(&manager, SIGNAL(focusChanged(uint)), this, SLOT(focusChanged(uint)));
    connect(&manager, SIGNAL(inputLanguageChanged(QString)), this, SLOT(inputLanguageChanged(QString)));
    connect(&manager, SIGNAL(currentEngineChanged(QString)), this, SLOT(currentEngineChanged(QString)));

    setupUi();
    settingsUpdated();

    focusChanged(manager.focus());
    q->restoreGeometry(settings.value("Geometry", q->saveGeometry()).toByteArray());
    setupStateMachine();

    q->restoreGeometry(settings.value("Geometry", q->saveGeometry()).toByteArray());
    q->show();
    q->restoreGeometry(settings.value("Geometry", q->saveGeometry()).toByteArray());
}

void ToolBar::Private::setupStateMachine()
{
    qimsysDebugIn();

    if (machine)
        machine->deleteLater();
    QStateMachine *machine = new QStateMachine(this);

    QBoolSignal *focus = new QBoolSignal(this, SIGNAL(focusChanged(bool)), manager.focus(), machine);
    enter = new QBoolSignal(this, SIGNAL(enterChanged(bool)), false, machine);
    QBoolSignal *composing = new QBoolSignal(&manager, SIGNAL(composingChanged(bool)), manager.composing(), machine);
    QBoolsSignal *combination = new QBoolsSignal(focus, composing, QBoolsSignal::And, machine);

    QGraphicsBlurEffect *effect = new QGraphicsBlurEffect;
    ui.widget->setGraphicsEffect(effect);

    QState *parentState = new QState(QState::ParallelStates);
    machine->addState(parentState);
    machine->setInitialState(parentState);

    QState *sizeStates = new QState(parentState);
    QState *visibleStates = new QState(parentState);
    QState *activeStates = new QState(parentState);

    QState *invisibleState = new QState(visibleStates);
    QState *visibleState = new QState(visibleStates);
    QState *smallState = new QState(sizeStates);
    QState *hoverState = new QState(sizeStates);
    QState *popupState = new QState(sizeStates);
    QState *normalState = new QState(sizeStates);
    QState *activeState = new QState(activeStates);
    QState *inactiveState = new QState(activeStates);

    invisibleState->assignProperty(q, "visible", false);
    visibleState->assignProperty(q, "visible", true);

    QSize smallSize = ui.icon->size();
    smallState->setProperty("smallness", true);
    smallState->assignProperty(q, "size", smallSize);
    smallState->assignProperty(effect, "blurRadius", 5);
    hoverState->setProperty("smallness", false);
    new QBinding(q, "normalSize", hoverState, q, "size");
    hoverState->assignProperty(effect, "blurRadius", 0);
    normalState->setProperty("smallness", false);
    new QBinding(q, "normalSize", normalState, q, "size");
    normalState->assignProperty(effect, "blurRadius", 0);

    inactiveState->assignProperty(ui.icon, "enabled", false);
    activeState->assignProperty(ui.icon, "enabled", true);

    QSettings settings;

    settings.beginGroup(plugin->metaObject()->className());
    if (settings.value("Always", true).toBool()) {
        qimsysDebug();
        visibleStates->setInitialState(visibleState);
    } else if (settings.value("Available", true).toBool()) {
        qimsysDebug();
        visibleStates->setInitialState(invisibleState);
        invisibleState->addTransition(focus, SIGNAL(on()), visibleState);
        visibleState->addTransition(focus, SIGNAL(off()), invisibleState);
    } else if (settings.value("Active", true).toBool()) {
        qimsysDebug();
        visibleStates->setInitialState(invisibleState);
        invisibleState->addTransition(composing, SIGNAL(on()), visibleState);
        visibleState->addTransition(composing, SIGNAL(off()), invisibleState);
    }

#define TRANSITION(t) \
    { \
        QAbstractTransition *transition = t; \
        QSequentialAnimationGroup *seq = new QSequentialAnimationGroup; \
        QParallelAnimationGroup *para = new QParallelAnimationGroup; \
        QPropertyAnimation *animation = new QPropertyAnimation(q, "size", transition); \
        animation->setEasingCurve(QEasingCurve::InOutCirc); \
        para->addAnimation(animation); \
        animation = new QPropertyAnimation(q, "pos"); \
        animation->setEasingCurve(QEasingCurve::InOutCirc); \
        para->addAnimation(animation); \
        seq->addAnimation(para); \
        seq->addAnimation(new QPropertyAnimation(effect, "blurRadius")); \
        transition->addAnimation(seq); \
    }

    if (settings.value("Inactive", true).toBool()) {
        qimsysDebug();
        sizeStates->setInitialState(smallState);
        TRANSITION(smallState->addTransition(combination, SIGNAL(on()), normalState));
        TRANSITION(smallState->addTransition(enter, SIGNAL(on()), hoverState));
        TRANSITION(smallState->addTransition(this, SIGNAL(aboutToShow()), popupState));
        TRANSITION(hoverState->addTransition(enter, SIGNAL(off()), smallState));
        TRANSITION(hoverState->addTransition(this, SIGNAL(aboutToShow()), popupState));
        TRANSITION(popupState->addTransition(this, SIGNAL(minimize()), smallState));
        TRANSITION(popupState->addTransition(this, SIGNAL(normalize()), hoverState));
        TRANSITION(normalState->addTransition(combination, SIGNAL(off()), smallState));
    } else {
        if (settings.value("Unavailable", true).toBool()) {
            qimsysDebug();
            sizeStates->setInitialState(smallState);
            TRANSITION(smallState->addTransition(focus, SIGNAL(on()), normalState));
            TRANSITION(smallState->addTransition(enter, SIGNAL(on()), hoverState));
            TRANSITION(smallState->addTransition(this, SIGNAL(aboutToShow()), popupState));
            TRANSITION(hoverState->addTransition(enter, SIGNAL(off()), smallState));
            TRANSITION(hoverState->addTransition(this, SIGNAL(aboutToShow()), popupState));
            TRANSITION(popupState->addTransition(this, SIGNAL(minimize()), smallState));
            TRANSITION(popupState->addTransition(this, SIGNAL(normalize()), hoverState));
            TRANSITION(normalState->addTransition(focus, SIGNAL(off()), smallState));
        } else {
            qimsysDebug();
            sizeStates->setInitialState(normalState);
        }
    }
#undef TRANSITION
    activeStates->setInitialState(inactiveState);
    inactiveState->addTransition(focus, SIGNAL(on()), activeState);
    activeState->addTransition(focus, SIGNAL(off()), inactiveState);
    machine->start();
    positionChanged();
    qimsysDebugOut();
//    qimsysDebugOff();
}

void ToolBar::Private::positionChanged()
{
    qimsysDebugIn();
    QSettings settings;
    settings.beginGroup(q->metaObject()->className());
    settings.setValue("Position", q->pos());

    foreach (QState *state, findChildren<QState*>()) {
        if (state->dynamicPropertyNames().contains("smallness")) {
            QRect desktop = QApplication::desktop()->availableGeometry();
            QRect rect = q->geometry();
            if (state->property("smallness").toBool()) {
                if (q->width() > ui.icon->width()) {
                    if (rect.center().x() < desktop.center().x()) {
                        qimsysDebug() << q->pos();
                        state->assignProperty(q, "pos", q->pos());
                    } else {
                        qimsysDebug() << QPoint(q->x() + ui.widget->width(), q->y());
                        state->assignProperty(q, "pos", QPoint(q->x() + ui.widget->width(), q->y()));
                    }
                } else {
                    qimsysDebug() << q->pos();
                    state->assignProperty(q, "pos", q->pos());
                }
            } else {
                if (q->width() > ui.icon->width()) {
                    qimsysDebug() << q->pos();
                    state->assignProperty(q, "pos", q->pos());
                } else {
                    if (rect.center().x() < desktop.center().x()) {
                        qimsysDebug() << q->pos();
                        state->assignProperty(q, "pos", q->pos());
                    } else {
                        qimsysDebug() << QPoint(q->x() - ui.widget->width(), q->y());
                        state->assignProperty(q, "pos", QPoint(q->x() - ui.widget->width(), q->y()));
                    }
                }
            }
        }
    }
    qimsysDebugOut();
}

void ToolBar::Private::aboutToHide()
{
    if (enter->value())
        emit normalize();
    else
        QTimer::singleShot(1000, this, SIGNAL(minimize()));
}

void ToolBar::Private::setEnter(bool enter)
{
    emit enterChanged(enter);
}

void ToolBar::Private::focusChanged(uint focus)
{
    emit focusChanged(focus > 0);
}

void ToolBar::Private::settingsUpdated()
{
    QSettings settings;
    settings.beginGroup(plugin->metaObject()->className());
    ui.language->setVisible(settings.value("Language", false).toBool());
    ui.extension->setVisible(settings.value("Language Specific", true).toBool());
    ui.engine->setVisible(settings.value("Engine", false).toBool());
    ui.userDictionary->setVisible(settings.value("User Dictionary", true).toBool());
    ui.settings->setVisible(settings.value("Settings", true).toBool());
    ui.widget->resize(ui.widget->sizeHint());
    q->setNormalSize(QSize(ui.widget->x() + ui.widget->sizeHint().width(), ui.widget->sizeHint().height()));
    setupStateMachine();
}

void ToolBar::Private::setupUi()
{
    QString inputLanguage = manager.inputLanguage();
    {
        QMenu *langMenu = new QMenu(q);
        connect(langMenu, SIGNAL(aboutToShow()), this, SIGNAL(aboutToShow()));
        connect(langMenu, SIGNAL(aboutToHide()), this, SLOT(aboutToHide()));
        langGroup = new QActionGroup(langMenu);

        foreach(QimsysConverter *lang, QimsysPluginManager::objects<QimsysConverter>()) {
            QAction *action = new QAction(langGroup);
            new QBinding(lang, "icon", action, "icon");
            new QBinding(lang, "name", action, "text");
            action->setData(lang->identifier());
            action->setCheckable(true);

            if (lang->identifier() == inputLanguage) {
                action->setChecked(true);
                ui.language->setIcon(action->icon());
            }
            langMenu->addAction(action);
        }
        ui.language->setMenu(langMenu);
        connect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setInputLanguage(QAction*)));
    }

    QString currentEngine = manager.currentEngine();
    {
        QMenu *engineMenu = new QMenu(ui.engine);
        connect(engineMenu, SIGNAL(aboutToShow()), this, SIGNAL(aboutToShow()));
        connect(engineMenu, SIGNAL(aboutToHide()), this, SLOT(aboutToHide()));
        engineGroup = new QActionGroup(this);

        foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>()) {
            QAction *action = new QAction(engineGroup);
            new QBinding(engine, "icon", action, "icon");
            new QBinding(engine, "name", action, "text");
            new QBinding(engine, "available", action, "enabled");
            new QBinding(engine, "enabled", action, "visible");

            action->setData(engine->identifier());
            action->setCheckable(true);

            if (currentEngine == engine->identifier()) {
                action->setChecked(true);
                ui.engine->setIcon(action->icon());
                ui.userDictionary->setEnabled(engine->dictionary(this));
            }
            engineMenu->addAction(action);
        }
        ui.engine->setMenu(engineMenu);
        connect(engineGroup, SIGNAL(triggered(QAction*)), this, SLOT(setCurrentEngine(QAction*)));
    }
    inputLanguageChanged(inputLanguage);
    currentEngineChanged(currentEngine);
}

void ToolBar::Private::setInputLanguage(QAction *action)
{
    manager.setInputLanguage(action->data().toString());
}

void ToolBar::Private::inputLanguageChanged(const QString &inputLanguage)
{
    qimsysDebugIn() << inputLanguage;

    foreach (QAction *lang, langGroup->actions()) {
        if (lang->data().toString() == inputLanguage) {
            lang->setChecked(true);
            break;
        }
    }

    foreach(QWidget *widget, ui.extension->findChildren<QWidget*>()) {
        widget->deleteLater();
    }

    QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
    foreach(QimsysConverter *converter, converters) {
        if (converter->identifier() == inputLanguage) {
            ui.language->setIcon(converter->icon());
            foreach(QAction *action, converter->actions()) {
                QToolButton *button = new QToolButton(ui.extension);
                new QBinding(action, "icon", button, "icon");
                new QBinding(action, "text", button, "text");
                new QBinding(action, "enabled", button, "enabled");
                new QBinding(action, "checkable", button, "checkable");
                new QBinding(action, "checked", button, "checked");
                button->setIconSize(QSize(16, 16));
                button->setPopupMode(QToolButton::InstantPopup);
                button->setAutoRaise(true);
                ui.extension->layout()->addWidget(button);

                QList<QAction*> childActions = action->findChildren<QAction*>();
                if (childActions.isEmpty()) {
                    if (action->isCheckable())
                        connect(button, SIGNAL(toggled(bool)), action, SLOT(setChecked(bool)));
                    else
                        connect(button, SIGNAL(clicked()), action, SLOT(trigger()));
                } else {
                    QMenu *extMenu = new QMenu(button);
                    connect(extMenu, SIGNAL(aboutToShow()), this, SIGNAL(aboutToShow()));
                    connect(extMenu, SIGNAL(aboutToHide()), this, SLOT(aboutToHide()));
                    foreach(QAction *a, childActions) {
                        if (a->text() == "-") {
                            extMenu->addSeparator();
                        } else {
                            extMenu->addAction(a);
                        }
                    }
                    button->setMenu(extMenu);
                }
                button->show();
            }
            ui.engine->setVisible(converter->useEngine());
            ui.userDictionary->setVisible(converter->useEngine());
            break;
        }
    }

    ui.widget->resize(ui.widget->sizeHint());
    q->setNormalSize(QSize(ui.widget->x() + ui.widget->width(), ui.widget->height()));
    metaObject()->invokeMethod(this, "positionChanged", Qt::QueuedConnection);
    qimsysDebugOut();
}

void ToolBar::Private::currentEngineChanged(const QString &currentEngine)
{
    ui.engine->setIcon(QIcon());
    ui.engine->setEnabled(false);
    foreach(QAction *action, engineGroup->actions()) {
        if (currentEngine == action->data().toString()) {
            action->setChecked(true);
            ui.engine->setIcon(action->icon());
            ui.engine->setEnabled(true);
            break;
        }
    }
    ui.userDictionary->setEnabled(false);
    foreach(QimsysEngine *engine, QimsysPluginManager::objects<QimsysEngine>(this)) {
        if (engine->identifier() == currentEngine) {
            ui.userDictionary->setEnabled(engine->dictionary(this));
            break;
        }
    }
    ui.widget->resize(ui.widget->sizeHint());
    q->setNormalSize(QSize(ui.widget->x() + ui.widget->sizeHint().width(), ui.widget->sizeHint().height()));
}

void ToolBar::Private::setCurrentEngine(QAction *action)
{
    manager.setCurrentEngine(action->data().toString());
}

void ToolBar::Private::userDictionary()
{
    manager.exec(QimsysApplicationManager::ShowDictionary);
}

void ToolBar::Private::settings()
{
    manager.exec(QimsysApplicationManager::ShowSettings);
}

ToolBar::ToolBar(QimsysAbstractPluginObject *plugin, QWidget *parent)
    : QFrame(parent)
{
    d = new Private(plugin, this);
}

ToolBar::~ToolBar()
{
    delete d;
}

const QSize &ToolBar::normalSize() const
{
    return d->normalSize;
}

void ToolBar::setNormalSize(const QSize &normalSize)
{
    if (d->normalSize == normalSize) return;
    d->normalSize = normalSize;
    emit normalSizeChanged(normalSize);
}

void ToolBar::settingsUpdated()
{
    d->settingsUpdated();
}

void ToolBar::enterEvent(QEvent *e)
{
    d->setEnter(true);
    QFrame::enterEvent(e);
}

void ToolBar::leaveEvent(QEvent *e)
{
    QFrame::leaveEvent(e);
    d->setEnter(false);
}

void ToolBar::mousePressEvent(QMouseEvent *e)
{
    d->lastPos = e->pos();
}

void ToolBar::mouseMoveEvent(QMouseEvent *e)
{
    if (d->lastPos.isNull()) return;
    move(pos() - d->lastPos + e->pos());
}

void ToolBar::mouseReleaseEvent(QMouseEvent *e)
{
    mouseMoveEvent(e);
    d->lastPos = QPoint();
    d->positionChanged();
}

void ToolBar::moveEvent(QMoveEvent *e)
{
    qimsysDebugIn() << e->pos() << e->oldPos();
    QFrame::moveEvent(e);
    qimsysDebugOut();
}

void ToolBar::resizeEvent(QResizeEvent *e)
{
    qimsysDebugIn() << e->size() << e->oldSize();
    QFrame::resizeEvent(e);
    qimsysDebugOut();
}

#include "toolbar.moc"
