#include "ryushare.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>
#include <QStringList>
#include <QRegExp>

RyuShare::RyuShare(QObject *parent) :
    ServicePlugin(parent),
    m_waitTimer(new QTimer(this)),
    m_waitTime(0)
{
    this->connect(m_waitTimer, SIGNAL(timeout()), this, SLOT(updateWaitTime()));
}

QRegExp RyuShare::urlPattern() const {
    return QRegExp("http(s|)://(www.|)ryushare.com/\\w+", Qt::CaseInsensitive);
}

bool RyuShare::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void RyuShare::login(const QString &username, const QString &password) {
    QString data = QString("op=login&logFormSubmit=Login&login=%1&password=%2").arg(username).arg(password);
    QUrl url("http://ryushare.com/");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkLogin()));
}

void RyuShare::checkLogin() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    emit loggedIn((statusCode == 200) || (statusCode == 302));

    reply->deleteLater();
}

void RyuShare::checkUrl(const QUrl &webUrl) {
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void RyuShare::checkUrlIsValid() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();
    QRegExp re("http://\\w+.ryushare.com/files/\\d+/\\w+/[^'\"]+");

    if ((!redirect.isEmpty()) && (re.indexIn(redirect) == -1)) {
        this->checkUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());
        QString fileName = response.section("fname\" value=\"", 1, 1).section('"', 0, 0);

        if (fileName.isEmpty()) {
            emit urlChecked(false);
        }
        else {
            QString urlString = reply->request().url().toString();

            if (!urlString.endsWith(fileName)) {
                if (!urlString.endsWith('/')) {
                    urlString.append('/');
                }

                urlString.append(fileName);
            }

            QUrl url(urlString);
            emit urlChecked(true, url, this->serviceName(), fileName);
        }
    }

    reply->deleteLater();
}

void RyuShare::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    m_url = webUrl;
    QString urlString(m_url.toString());
    QString id = urlString.section('/', -2, -2);
    QString fileName = urlString.section('/', -1);
    QString data = QString("op=download1&id=%1&fname=%2&method_free=Low Speed Download").arg(id).arg(fileName);
    QNetworkRequest request(m_url);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(onWebPageDownloaded()));
}

void RyuShare::onWebPageDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://\\w+.ryushare.com/files/\\d+/\\w+/[^'\"]+");
    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (re.indexIn(redirect) == 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else if (!redirect.isEmpty()) {
        this->getDownloadUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (re.indexIn(response) >= 0) {
            QNetworkRequest request;
            request.setUrl(QUrl(re.cap()));
            emit downloadRequestReady(request);
        }
        else {
            m_fileId = response.section("id\" value=\"", 1, 1).section('"', 0, 0);
            m_rand = response.section("rand\" value=\"", 1, 1).section('"', 0, 0);
            QString codeBlock = response.section("<td align=right>", 1, 1).section("</span></div>", 0, 0);
            QStringList codeList = codeBlock.split(">&#", QString::SkipEmptyParts);

            if (!codeList.isEmpty()) {
                codeList.removeFirst();
            }

            int i = 1;

            while ((!codeList.isEmpty()) && (i < 5)) {
                m_code.append(QString::number(codeList.takeFirst().left(2).toInt() - 48));
                i++;
            }

            if ((m_fileId.isEmpty()) || (m_rand.isEmpty()) || (m_code.isEmpty())) {
                QString errorString = response.section("<div class=\"err\">", 1, 1).section('<', 0, 0);

                if (!errorString.isEmpty()) {
                    if (errorString.startsWith("You have to wait")) {
                        int mins = errorString.section(" minutes", 0, 0).section(' ', -1).toInt();
                        int secs = errorString.section(" seconds", 0, 0).section(' ', -1).toInt();
                        this->startWait((mins * 60000) + (secs * 1000));
                        this->connect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
                    }
                    else {
                        emit error(UnknownError);
                    }
                }
                else {
                    emit error(UnknownError);
                }
            }
            else {
                this->startWait(30000);
                this->connect(this, SIGNAL(waitFinished()), this, SLOT(submitCaptcha()));
            }
        }
    }

    reply->deleteLater();
}

void RyuShare::submitCaptcha() {
    QString data = QString("op=download2&id=%1&rand=%2&method_free=Low Speed Download&code=%3&down_direct=1").arg(m_fileId).arg(m_rand).arg(m_code);
    QNetworkRequest request(m_url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(onCaptchaSubmitted()));
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(submitCaptcha()));
}

void RyuShare::onCaptchaSubmitted() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://\\w+.ryushare.com/files/\\d+/\\w+/[^'\"]+");
    QString response(reply->readAll());

    if (re.indexIn(response) >= 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else {
        emit error(UnknownError);
    }

    reply->deleteLater();
}

void RyuShare::startWait(int msecs) {
    if (msecs > 30000) {
        emit statusChanged(LongWait);
    }
    else {
        emit statusChanged(ShortWait);
    }

    emit waiting(msecs);
    m_waitTime = msecs;
    m_waitTimer->start(1000);
}

void RyuShare::updateWaitTime() {
    m_waitTime -= m_waitTimer->interval();
    emit waiting(m_waitTime);

    if (m_waitTime <= 0) {
        m_waitTimer->stop();
        emit waitFinished();
    }
}

void RyuShare::onWaitFinished() {
    emit statusChanged(Ready);
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
}

Q_EXPORT_PLUGIN2(ryushare, RyuShare)
