#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import ecore
import logging
from evas import ClippedSmartObject
from kinetic import KineticMouse

__all__ = ("CellRenderer", "BaseGrid", "KineticGrid")

log = logging.getLogger("terra.ui.grid")


class CellRenderer(object):
    def __init__(self, canvas):
        pass

    def value_set(self, v):
        raise NotImplemented("value_set")


class BaseGrid(ClippedSmartObject):
    """Basic vertical grid of items.

    Cell renderers should implement the interface provided by L{CellRenderer},
    that is, a C{value_set()} method that will get the model it should
    use. These renderers should be returned by C{renderer_new}, that will
    be called with the current L{Canvas} as parameter.

    This is a base class and implements no event management.

    @note: this grid does not move items by itself, it rely on another
       element to call L{Grid.move_offset()} or other means to manipulate
       the current position, like L{KineticGrid}.

    @see: L{KineticGrid}
    """
    click_constant = 20
    move_constant = 10
    click_block_time = 0.5

    def __init__(self, canvas, renderer_new, elements, renderer_size=(32, 32),
                 v_align=0.5):
        if not callable(renderer_new):
            raise TypeError("renderer_new must be callable")
        ClippedSmartObject.__init__(self, canvas)
        self.elements = elements
        if elements:
            self.current = 0
        else:
            self.current = None
        self._dirty = True
        self._frozen = 0
        self.renderer_new = renderer_new
        self.renderers = []
        self.renderer_size = renderer_size
        self.renderer_width = renderer_size[0]
        self.renderer_height = renderer_size[1]
        self.spare_renderers_cols = 2
        self._position_changed_cb = None
        self._clicked_cb = None
        self.last_top_left_visible = None
        self.offset = -renderer_size[0]
        self.last_offset = None
        self.v_align = v_align
        self.inc_y = 0
        self.offset_y = 0

    def freeze(self):
        self._frozen += 1

    def thaw(self):
        if self._frozen > 1:
            self._frozen -= 1
        elif self._frozen == 1:
            self._frozen = 0
            if self._dirty:
                self._reconfigure_renderers(*self.geometry_get())
        else:
            log.warning("thaw more than freeze!")

    def model_updated(self):
        "Notifies that model was updated and grid need to be redrawn."
        self.request_reconfigure(*self.geometry_get())

    def request_reconfigure(self, x, y, w, h):
        "Request object to reconfigure its children to fit given geometry."
        self._dirty = True
        if self._frozen == 0:
            self._reconfigure_renderers(x, y, w, h)

    def resize(self, w, h):
        x, y, old_w, old_h = self.geometry_get()
        if (old_w, old_h) == (w, h):
            return
        self.request_reconfigure(x, y, w, h)

    def position_changed_cb_set(self, cb, *a, **ka):
        """Set callback used to notify position changes.

        Signature: C{function(grid, percent, *a, **ka)}
        """
        if cb is None:
            self._position_changed_cb = None
        elif callable(cb):
            self._position_changed_cb = (cb, a, ka)
        else:
            raise TypeError("cb must be callable or None")

    def position_changed_cb_get(self):
        return self._position_changed_cb

    position_changed_cb = property(position_changed_cb_get,
                                   position_changed_cb_set)

    def emit_position_changed(self):
        "Emit position_changed."
        if self._position_changed_cb is None or len(self.elements) <= 1:
            return

        percent = self.position_get()
        cb, a, ka = self._position_changed_cb
        cb(self, percent, *a, **ka)

    def position_get(self):
        "Get position in grid, from 0.0 (left) to 1.0 (right)."
        if self.last_top_left_visible < 1:
            return 0.0
        else:
            rel_offset = self.offset + self.renderer_width
            pos = self.current + float(rel_offset) / -self.renderer_width
            return float(pos) / self.last_top_left_visible

    def position_set(self, value):
        if value < 0.0:
            log.warning("value (%s) < 0.0, set to 0.0", value)
            value = 0.0
        elif value > 1.0:
            log.warning("value (%s) > 1.0, set to 1.0", value)
            value = 1.0

        max_x = self.last_top_left_visible * self.renderer_width
        x = int(max_x * value)
        self.current = x / self.renderer_width
        self.offset = -self.renderer_width - (x % self.renderer_width)
        self._reposition_renderers()
        self._refill_renderers()
        self.emit_position_changed()

    def clicked_cb_set(self, cb, *a, **ka):
        """Set callback used to notify click events.

        Signature: C{function(grid, index, *a, **ka)}
        """
        if cb is None:
            self._clicked_cb = None
        elif callable(cb):
            self._clicked_cb = (cb, a, ka)
        else:
            raise TypeError("cb must be callable or None")

    def clicked_cb_get(self):
        return self._clicked_cb

    clicked_cb = property(clicked_cb_get, clicked_cb_set)

    def emit_clicked(self, x, y):
        "Emit clicked at the given position (relative to canvas)."
        if not self._clicked_cb:
            return

        idx = self._index_at_xy(x, y)
        if idx >= 0:
            cb, a, ka = self._clicked_cb
            cb(self, idx, *a, **ka)

    def visible_rows_count(self):
        "Return the number of visible rows."
        return self.h_items

    def visible_cols_count(self):
        "Return the number of visible columns."
        if not self.renderers or not self.renderers[0]:
            return 0

        c = len(self.renderers[0])
        e = len(self.elements)
        if c == e:
            return e - 1
        elif c > e:
            return e
        else:
            return max(c - self.spare_renderers_cols, 0)

    def visible_cols_scale_get(self):
        """Return the scale of visible cols compared to total number.

        This is compared to the largest row.

        @rtype: float
        """
        v = self.visible_cols_count()
        e = self.items_per_row
        if v >= e or e == 0:
            return 1.0
        else:
            return float(v) / e

    def renderer_for_index(self, index):
        """Get the renderer of the element at index or None if it's not visible.

        @return: row renderer or None if index is not visible.
        """
        if self.current is None:
            return None
        if not (0 <= index < len(self.elements)):
            return None

        col_idx = index % self.items_per_row
        base = col_idx - self.current
        if 0 <= self.current <= self.visible_cols_count():
            row_idx = index / self.items_per_row
            base += 1
            return self.renderers[row_idx][base]

    def _index_at_xy(self, x, y):
        ox, oy = self.pos
        x -= ox + self.offset
        y -= oy

        col_idx = x / self.renderer_width
        row_idx = y / self.renderer_height
        idx = row_idx * self.items_per_row + col_idx + (self.current - 1)

        if 0 <= idx < len(self.elements):
            return idx
        else:
            return -1

    def index_at_xy(self, x, y):
        """Return the index of the element being displayed at given position.

        @return: -1 if out of visible area or index otherwise.
        """
        if (x, y) not in self.rect:
            return -1
        return self._index_at_xy(x, y)

    def _renderer_new_get(self):
        o = self.renderer_new(self.evas)
        self.member_add(o)
        return o

    def _apply_to_renders_row(self, row_idx, func):
        for c in self.renderers[row_idx]:
            func(c)

    def _apply_to_renders_col(self, col_idx, func):
        for row in self.renderers:
            cell = row[col_idx]
            func(cell)

    def _reconfigure_renderers(self, x, y, w, h):
        """Reconfigure renderers based on new geometry.

        Create, delete or change renderers to fit the new geometry.

        @precondition: C{self._dirty == True}
        """
        if not self._dirty:
            return

        c_width, c_height = self.renderer_size
        self.w_items = w_items = max(int(w / c_width), 1)
        self.h_items = h_items = max(int(h / c_height), 1)
        self.n_items = n_items = w_items * h_items

        if self.v_align < 0 and h_items > 1:
            self.offset_y = 0
            self.inc_y = int((h - h_items * c_height) / (h_items - 1))
        else:
            self.offset_y = int(self.v_align * (h - h_items * c_height))
            self.inc_y = 0

        n_elements = len(self.elements)
        if n_items >= n_elements:
            self.current = 0
            self.offset = -c_width

        if n_items >= n_elements:
            self.items_per_row = w_items
            self.last_top_left_visible = 0
        else:
            self.items_per_row = n_elements / h_items
            if n_elements % h_items:
                self.items_per_row += 1
            self.last_top_left_visible = self.items_per_row - w_items

        if w % c_width:
            self.spare_renderers_cols = 3
        else:
            self.spare_renderers_cols = 2
        w_items += self.spare_renderers_cols

        if self.last_top_left_visible < 0:
            self.last_top_left_visible = 0

        # delete unneeded
        if len(self.renderers):
            # delete unrequired rows of renderers
            while h_items < len(self.renderers):
                for r in self.renderers.pop():
                    r.delete()

            # delete unrequired columns of renderers
            if len(self.renderers) and w_items < len(self.renderers[0]):
                while w_items < len(self.renderers[0]):
                    for row in self.renderers:
                        cell = row.pop()
                        cell.delete()

        # reposition existing
        cy = y + self.offset_y
        for row in self.renderers:
            cx = x + self.offset
            for c in row:
                c.move(cx, cy)
                cx += c_width
            cy += c_height + self.inc_y

        # create required columns of renderers
        if len(self.renderers):
            if w_items > len(self.renderers[0]):
                cx = x + len(self.renderers[0]) * c_width + self.offset
                while w_items > len(self.renderers[0]):
                    cy = y + self.offset_y
                    for row in self.renderers:
                        c = self._renderer_new_get()
                        c.geometry_set(cx, cy, c_width, c_height)
                        row.append(c)
                        cy += c_height + self.inc_y
                    cx += c_width

        # create required rows of renderers
        cy = y + len(self.renderers) * c_height + self.offset_y
        while h_items > len(self.renderers):
            cx = x + self.offset
            row = []
            for i in xrange(w_items):
                c = self._renderer_new_get()
                c.geometry_set(cx, cy, c_width, c_height)
                row.append(c)
                cx += c_width
            self.renderers.append(row)
            cy += c_height + self.inc_y

        self._refill_renderers()
        self._dirty = False

    def _hide_all_renderers(self):
        for row in self.renderers:
            for c in row:
                c.hide()

    def _hide_left_spare_col(self):
        self._apply_to_renders_col(0, lambda c: c.hide())

    def _hide_right_spare_cols(self):
        def hide(c):
            c.hide()
        for i in xrange(-self.spare_renderers_cols + 1, 0):
            self._apply_to_renders_col(i, hide)

    def _hide_spare_cols(self):
        self._hide_left_spare_col()
        self._hide_right_spare_cols()

    def _refill_renderers_col(self, col_idx, idx):
        n_elements = len(self.elements)
        items_per_row = self.items_per_row
        for row in self.renderers:
            c = row[col_idx]
            if idx < n_elements:
                c.value_set(self.elements[idx])
                c.show()
            else:
                c.hide()
            idx += items_per_row

    def _refill_renderers_single_case(self):
        self._hide_spare_cols()
        w_items = self.w_items
        renderers = self.renderers
        for idx, e in enumerate(self.elements):
            i = idx / w_items
            j = idx % w_items + 1
            c = renderers[i][j]
            c.value_set(e)
            c.show()
        for idx in xrange(idx + 1, self.n_items):
            i = idx / w_items
            j = idx % w_items + 1
            c = renderers[i][j]
            c.hide()

    def _refill_renderers(self):
        """Setup renderers to use the correct models.

        This synchronizes the view and model, also taking care to hide
        renderers not being used.
        """
        if not self.renderers or not self.renderers[0]:
            return

        n_elements = len(self.elements)
        if n_elements < 1 or self.current is None:
            return self._hide_all_renderers()
        if self.n_items >= n_elements:
            return self._refill_renderers_single_case()

        if self.current == 0:
            self._hide_left_spare_col()
            col_idx_start = 1
            base_start = 0
        else:
            col_idx_start = 0
            base_start = -1

        if self.current == self.last_top_left_visible:
            col_idx_end = self.w_items + 1
            self._hide_right_spare_cols()
        else:
            col_idx_end = len(self.renderers[0])
            if self.current == self.last_top_left_visible - 1 \
               and self.spare_renderers_cols > 2:
                col_idx_end -= 1
                self._apply_to_renders_col(-1, lambda c: c.hide())

        n_rows = n_elements / self.items_per_row
        cols = range(col_idx_start, col_idx_end)
        base = base_start
        for row_idx in xrange(n_rows):
            idx = base + self.current
            row = self.renderers[row_idx]
            for col_idx in cols:
                c = row[col_idx]
                c.value_set(self.elements[idx])
                c.show()
                idx += 1
            base += self.items_per_row

        partial_row = n_elements % self.items_per_row
        if partial_row:
            row_idx = n_rows
            base = partial_row - (self.current + base_start)
            if base <= 0:
                self._apply_to_renders_row(row_idx, lambda c: c.hide())
                return

            row = self.renderers[row_idx]
            idx = row_idx * self.items_per_row + (self.current + base_start)
            for col_idx in cols[:base]:
                c = row[col_idx]
                c.value_set(self.elements[idx])
                c.show()
                idx += 1
            for col_idx in cols[base:]:
                c = row[col_idx]
                c.hide()

    def _reposition_renderers(self):
        """Reposition renderers based on new offset.

        This uses self.offset to reposition renderers.
        """
        if self.offset == self.last_offset:
            return

        self.last_offset = self.offset
        x, y = self.pos
        x += self.offset
        c_width, c_height = self.renderer_size
        cy = y + self.offset_y
        for row in self.renderers:
            cx = x
            for c in row:
                c.move(cx, cy)
                cx += c_width
            cy += c_height + self.inc_y

    def _rotate_renders_right(self):
        for r in self.renderers:
            c = r.pop()
            r.insert(0, c)

    def _rotate_renders_left(self):
        for r in self.renderers:
            c = r.pop(0)
            r.append(c)

    def move_offset(self, offset):
        """Move (scroll) contents by the given offset.

        @precondition: C{offset != 0 and self.current is not None}
        @return: True if operation was successful or False if the value was
           out of boundaries. The offset will be restricted to respect
           these limits.
        """
        if offset == 0 or self.current is None \
               or not self.renderers or not self.renderers[0]:
            return False

        max_x = self.last_top_left_visible * self.renderer_width
        old_x = (self.current - 1) * self.renderer_width - self.offset
        x = old_x - offset

        if x < 0 or x > max_x:
            if x < 0:
                self.current = 0
            else:
                self.current = self.last_top_left_visible
            self.offset = -self.renderer_width
            self._refill_renderers()
            self._reposition_renderers()
            self.emit_position_changed()
            return False

        idx = x / self.renderer_width
        items_over = idx - self.current
        if items_over == 0:
            self.offset += offset
        elif items_over in (-1, 1):
            self.current += items_over
            self.offset = (self.current - 1) * self.renderer_width - x
            if items_over == 1:
                self._rotate_renders_left()
                col_idx = -1
                new_idx = self.current + len(self.renderers[0]) - 2
            else:
                self._rotate_renders_right()
                col_idx = 0
                new_idx = self.current - 1

            if 0 <= new_idx < self.items_per_row:
                self._refill_renderers_col(col_idx, new_idx)
            else:
                self._apply_to_renders_col(col_idx, lambda c: c.hide())

        else:
            self.current += items_over
            self.offset = (self.current - 1) * self.renderer_width - x
            self._refill_renderers()

        self._reposition_renderers()
        self.emit_position_changed()
        return True


class KineticGrid(BaseGrid):
    """Grid using kinetics.

    Events will not be handled to row renderers, they will be handled by
    event_area, an object that will be laid out on top of every renderer.

    Behavior is based on B{click_constant}: maximum vertical motion to
    consider as click, values greater that (in absolute terms) are considered
    drag.
    """
    click_constant = 20

    def __init__(self, canvas, renderer_new, elements, renderer_size=(32, 32),
                 v_align=0.5):
        BaseGrid.__init__(self, canvas, renderer_new, elements, renderer_size,
                          v_align)
        self.event_area = self.Rectangle(color=(0, 0, 0, 0))
        self.event_area.show()
        self.event_area.on_mouse_down_add(self._cb_on_mouse_down)
        self.event_area.on_mouse_up_add(self._cb_on_mouse_up)
        self.event_area.on_mouse_move_add(self._cb_on_mouse_move)
        self.kinetic = KineticMouse(self.move_offset)
        self.is_drag = False
        self.mouse_down_pos = None
        self.clicks_blocked = False
        self._unblock_clicks_timer = None

    def _renderer_new_get(self):
        o = BaseGrid._renderer_new_get(self)
        self.event_area.raise_()
        return o

    def resize(self, w, h):
        BaseGrid.resize(self, w, h)
        self.event_area.resize(w, h)

    def _cb_on_mouse_down(self, obj, event):
        if event.button == 1:
            x = event.position.canvas.x
            self.mouse_down_pos = x
            self.is_drag = not self.kinetic.mouse_down(x)

    def _is_click_possible(self, x):
        if self.is_drag or self.mouse_down_pos is None:
            return False
        else:
            return abs(x - self.mouse_down_pos) <= self.click_constant

    def _cb_on_mouse_up(self, obj, event):
        if event.button == 1:
            x = event.position.canvas.x
            if self._is_click_possible(x):
                if not self.clicks_blocked:
                    self.emit_clicked(x, event.position.canvas.y)
                self.kinetic.mouse_cancel()
            else:
                self.kinetic.mouse_up(x)

    def _cb_on_mouse_move(self, obj, event):
        if event.buttons == 1:
            x = event.position.canvas.x
            if not self._is_click_possible(x):
                self.is_drag = True
            if abs(self.mouse_down_pos - x) > self.move_constant:
                self.kinetic.mouse_move(x)

    def emit_position_changed(self):
        BaseGrid.emit_position_changed(self)

        if self.is_drag:
            if self._unblock_clicks_timer is not None:
                self._unblock_clicks_timer.delete()

            def cb_unblock_clicks():
                self.clicks_blocked = False

            self.clicks_blocked = True
            self._unblock_clicks_timer = ecore.timer_add(self.click_block_time,
                                                         cb_unblock_clicks)
