from StreamAnalyzer import StreamAnalyzer
from PySide import QtCore

import gobject
import time
import urllib




# idle timeout in milliseconds
_IDLE_TIMEOUT = 1000 * 60 * 3


class AbstractBackend(QtCore.QObject):
    """
    Abstract base class for media player backend implementations.
    Backends deriving from this class only have to implement a minimal set of
    simple methods, while all the logic is contained in this base class.
    This way, backend implementation becomes easy and straight-forward.
    """
    __pyqtSignals__ = ( "eof", "pos_changed(int,int)", "volume(int)", "tag_discovered(*args)", "status_change(int)", "state_change(int)" )

    # player states
    STATUS_CONNECTING = 0
    STATUS_BUFFERING = 1
    STATUS_PLAYING = 2
    STATUS_STOPPED = 3
    STATUS_EOF = 4

    # error codes
    ERR_INVALID = 0
    ERR_NOT_FOUND = 1
    ERR_CONNECTION_TIMEOUT = 2
    ERR_NOT_SUPPORTED = 3
    ERR_SERVER_FULL = 4
    
    EVENT_STARTED = "event-started"
    EVENT_KILLED = "event-killed"
    EVENT_SUSPENDED = "event-suspended"
    EVENT_ERROR = "event-error"
    
    EVENT_STATUS_CHANGED = "event-status-changed"
    EVENT_VOLUME_CHANGED = "event-volume-changed"
    EVENT_POSITION_CHANGED = "event-position-changed"
    EVENT_ASPECT_CHANGED = "event-aspect-changed"
    EVENT_TAG_DISCOVERED = "event-tag-discovered"
   
    # remote control keys that can be sent to backends (for DVD navigation etc.)
    KEY_UP = 0
    KEY_DOWN = 1
    KEY_LEFT = 2
    KEY_RIGHT = 3
    KEY_SELECT = 10
    KEY_MENU1 = 11
    KEY_MENU2 = 12
    KEY_MENU3 = 13

    # playback modes so that the backends can behave accordingly
    MODE_AUDIO = 0

    __context_id_cnt = [0]
    
    
    def __init__(self):
        """
        Constructor to be invoked by subclasses. Do not instantiate this class
        directly.
        """
        
        # mode: MODE_AUDIO
        self.__mode = self.MODE_AUDIO

        # ID tags
        self.__tags = {}

        # URI of the current file
        self.__uri = ""
        
        # point to resume from
        self.__suspension_point = None

        # current position and total length
        self.__position = (0, 0)
        
        # whether we are at end-of-file
        self.__eof_reached = False

        # whether the player is currently playing
        self.__playing = False
        
        # current volume level (0..100)
        #self.__volume = 50
        
        self.__context_id = 0
                
        self.__position_handler = None
        self.__idle_handler = None
        
        self.__stream_analyzer = StreamAnalyzer()
        
        QtCore.QObject.__init__(self)

    def __repr__(self):
        """
        Returns a readable string representation of this player object.
        """
    
        return self.__class__.__name__


    def _new_context_id(self):
        """
        Generates and returns a new context ID.
        
        @return: a new context ID
        """
    
        self.__context_id_cnt[0] += 1
        ctx_id = self.__context_id_cnt[0]
        return ctx_id

    def __on_eof(self):
        """
        Reacts on end-of-file.
        """

        self.__playing = False
        self.__eof_reached = True
        self.__position = (0,0)
        #self.__suspension_point = (self.__uri, 0)
        self.emit(QtCore.SIGNAL("eof"))


    def __on_idle_timeout(self):
        """
        Reacts on idle timeout and suspends the player.
        """

        gobject.source_remove(self.__idle_handler)
        self.__idle_handler = None
        
        self.__playing = False
        pos, total = self.__position
        self.__suspension_point = (self.__uri, pos)
        self._close()


    def __resume_if_necessary(self):
    
        if (self.__suspension_point):
            uri, pos = self.__suspension_point
            self.__suspension_point = None

            self._ensure_backend()

            self._load(uri)
            #self._set_volume(self.__volume)
            gobject.idle_add(self._seek, pos)
            self._stop()
            self.__watch_progress()
            
        elif (self.__eof_reached):
            self.__eof_reached = False
            
            self._load(self.__uri)
            #self._set_volume(self.__volume)
            self._stop()
            self.__position = (0, 0)
            self.__watch_progress()
            #end if


    def __watch_progress(self):
        """
        Starts watching the time position progress.
        """
    
        if (not self.__position_handler):
            self.__position_handler = \
                  gobject.timeout_add(0, self.__update_position, 0, time.time())
            

    def __update_position(self, beginpos, timestamp):
        """
        Regularly updates the position in the file.
        """
	
        if (self.__playing):
            pos, total = self.__position
            if (pos < 3):
                pos, total = self._get_position()
                timestamp = time.time()
                beginpos = pos
            else:
                # we don't ask the backend for position every time because
                # this could be inefficient with some backends
                pos = beginpos + (time.time() - timestamp)

            self.__position = (pos, total)
            if (pos != 0 and total > 0):
                if (pos >= 0):
                    self.emit(QtCore.SIGNAL("pos_changed(int,int)"),pos,total)

            if (total > 0 and total - pos < 1):
                delay = 200
            else:
                delay = 500
            self.__position_handler = \
                gobject.timeout_add(delay, self.__update_position,
                                      beginpos, timestamp)

            # detect EOF
            if (pos > 1 and total > 0 and self._is_eof()):
                self.__on_eof()
          
        else:
            # stop updating when not playing
            self.__position_handler = None

        # reset idle timeout
        if (self.__idle_handler):
            gobject.source_remove(self.__idle_handler)
        self.__idle_handler = gobject.timeout_add(_IDLE_TIMEOUT,
                                                  self.__on_idle_timeout)          


    def __parse_playlist(self, url):
        """
        Parses the playlist at the given URL and returns a list of URLs.
        """
    
        uris = []
        try:
            fd = urllib.urlopen(url)
        except:
            return uris
            
        data = fd.read()
        fd.close()

        filelines = [ l for l in data.splitlines()
                      if l.startswith("File") ]
        httplines = [ l for l in data.splitlines()
                      if l.startswith("http") ]
        for line in filelines:
            idx = line.find("=")
            uri = line[idx + 1:].strip()
            uris.append(uri)
        #end for

        for line in httplines:
            uri = line.strip()
            uris.append(uri)
        #end for

        return uris

        
    def set_window(self, xid):
        """
        Sets the window for video output.
        
        @param xid: Xid of the window
        """
    
        self._set_window(xid)
        
        
    def send_key(self, key):
    
        return self._send_key(key)


    def load_audio(self, uri):
    
        self.__mode = self.MODE_AUDIO
        return self.__load(uri)
        
        
    def load_video(self, uri):
    
        self.__mode = self.MODE_VIDEO
        return self.__load(uri)

        
    def __load(self, uri, ctx_id = -1):
        """
        Loads and plays the given file.
        
        @param uri: URI of the file
        @param ctx_id: context ID to use; for internal use only
        @return: context ID
        """

        if (uri.startswith("http") and not uri.startswith("http://127.0.0.1")):
            s_type = self.__stream_analyzer.analyze(uri)
        else:
            s_type = StreamAnalyzer.STREAM

        if (s_type == StreamAnalyzer.PLAYLIST):
            uris = self.__parse_playlist(uri)
            if (uris):
                uri = uris[0]
            else:
                uri = ""

        self._ensure_backend()
        
        self.__tags.clear()
        self.__uri = uri
        self.__playing = False
        self.__eof_reached = False
        self.__suspension_point = None
       
        self.stop()

        self._load(uri)
        #self._set_volume(self.__volume)
        
        #print "DELAY PLAY", uri
        gobject.timeout_add(0, self.play)

        if (ctx_id != -1):
            self.__context_id = ctx_id
        else:
            self.__context_id = self._new_context_id()

        return self.__context_id


    def _report_state(self, state):
        self.emit(QtCore.SIGNAL("state_change(int)"),state)

    def _report_volume(self, vol):
        """
        The subclass calls this to report the sound volume.
        
        @param vol: volume level between 0 and 100
        """
        
        self.emit(QtCore.SIGNAL("volume(int)"),vol)

    def _report_tag(self, tag, value):
        """
        The subclass calls this to report ID tags.
        """
        
        self.__tags[tag] = value
        self.emit(QtCore.SIGNAL("tag_discovered(PyQt_PyObject)"),self.__tags)
    
    
    def _report_connecting(self):
        """
        The subclass calls this to report connecting to a server.
        """
        
        self.emit(QtCore.SIGNAL("status_change(int)"),self.STATUS_CONNECTING)
        
    
    def _report_buffering(self, value):
        """
        The subclass calls this to report stream buffering.
        """
        
        self.emit(QtCore.SIGNAL("status_change(int)"),self.STATUS_BUFFERING)
        
        
    def _report_error(self, err, message):
        """
        The subclass calls this to report errors.
        
        @param err: error code
        """
        
        self.stop()
        self.emit(QtCore.SIGNAL("error(PyQt_PyObject)"),err)
        self.__eof_reached = True
       
        
        
    def play(self):
        """
        Starts playback.
        """
        
        self.__resume_if_necessary()

        if (not self.__playing):
            self.__position = (0, 0)
            self._play()

        self.__playing = True
        print "PLAY"
        self.emit(QtCore.SIGNAL("status_change(int)"),self.STATUS_PLAYING)
        self.__watch_progress()
        
        
    def pause(self):
        """
        Pauses playback or starts it if it was paused.
        """

        self.__resume_if_necessary()

        if (self.__playing):
            self._stop()
            self.__playing = False
            self.emit(QtCore.SIGNAL("status_change(int)"),self.STATUS_STOPPED)
        else:
            self._play()
            self.__playing = True
            self.emit(QtCore.SIGNAL("status_change(int)"),self.STATUS_PLAYING)
            self.__watch_progress()
        
        
    def stop(self):
        """
        Stops playback.
        """

        if (self.__playing):
            self._stop()
            self.emit(QtCore.SIGNAL("status_change(int)"),self.STATUS_STOPPED)
        self.__playing = False
        
        
    def close(self):
        """
        Closes the player.
        """
    
        self._close()
        self.__context_id = 0
        
        
    def set_volume(self, volume):
        """
        Sets the audio volume.
        
        @param volume: volume as a value between 0 and 100
        """

        #self._ensure_backend()
        #self.__volume = volume
        self._set_volume(volume)    

    def _ensure_backend(self):
        """
        To be implemented by the backend.
        """
    
        raise NotImplementedError

    
    def _close(self):
        """
        To be implemented by the backend.
        """
        
        raise NotImplementedError

    
    def _set_window(self, xid):
        """
        May be implemented by the backend.
        """
        
        pass
        
        
    def _send_key(self, key):
        """
        May be implemented by the backend.
        """
        
        return False


    def _load(self, uri):
        """
        To be implemented by the backend.
        """
        
        raise NotImplementedError


    def _is_eof(self):
        """
        To be implemented by the backend.
        """

        raise NotImplementedError

    
    def _play(self):
        """
        To be implemented by the backend.
        """
        
        raise NotImplementedError
        
        
    def _stop(self):
        """
        To be implemented by the backend.
        """

        raise NotImplementedError


    def _seek(self, pos):
        """
        To be implemented by the backend.
        """
    
        raise NotImplementedError
        
        
    def _set_volume(self, vol):
        """
        To be implemented by the backend.
        """
    
        raise NotImplementedError
        
        
    def _get_position(self):
        """
        To be implemented by the backend.
        """
    
        raise NotImplementedError

