#include "gamewindow.h"
#include <QPainter>
#include <QTimerEvent>
#include <QDebug>
#include <QDateTime>
#include <QKeyEvent>
#include <QTimer>

#include "pair.h"
#include "board.h"
#include "motionsenser.h"
#include "gamethread.h"

GameWindow::GameWindow(QWidget *parent)
    : QWidget(parent),mThread(0),mPair(0),mBoard(0),mGameState(GameNotStated)
{
    //qDebug() << "Window::Window()";
    setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
    Smily::loadImages();
    mSenser = new MotionSenser();
    mRenderTimer = new QTimer(this);
    connect(mRenderTimer, SIGNAL(timeout()), this, SLOT(render()));
    this->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    this->resize( 480,720);
}

GameWindow::~GameWindow()
{    
    delete mPair;
    delete mBoard;
    delete mThread;
    delete mSenser;
    Smily::unloadImages();
}


void GameWindow::newGame()
{
    qDebug() << "GameWindow::newGame()";

    if( mBoard ) {
        delete mBoard;
    }

    if ( mPair ) {
        delete mPair;
    }

    if(mThread ) {
        delete mThread;
    }

    mPair = new Pair();
    mBoard = new Board(QSize(8,12));
    QObject::connect(mBoard,SIGNAL(pointsEarned(int)),this,SIGNAL(pointsEarned(int)));

    mThread = new GameThread( mBoard,mPair);
    QObject::connect(mThread,SIGNAL(gameEnded()),this,SLOT(slotGameEnded()));
    mThread->startGame();

    mGameState = GameStarted;
    mSenser->setObserver(this);    
    mRenderTimer->start( 200 );
}

void GameWindow::resumeGame()
{    
    mThread->startGame();
    mRenderTimer->start( 200 );
    mGameState = GameStarted;
    mSenser->setObserver(this);
}

void GameWindow::stopGame()
{
    mSenser->setObjectName(0);
    mGameState = GamePaused;
    mThread->pauseGame();    
}

void GameWindow::paintEvent(QPaintEvent* /*aPaintEvent*/ )
{    
    QPainter painter(this);

    painter.setRenderHint(QPainter::Antialiasing);
    mPair->draw(painter);
    mBoard->draw(painter);
}

void GameWindow::render()
{    
    update();
}

void GameWindow::keyPressEvent( QKeyEvent* aKeyEvent )
{
    if( mGameState != GameStarted ) {
        aKeyEvent->ignore();
        return;
    }

    switch( aKeyEvent->key() )  {
    case Qt::Key_Left:
        {            
            mPair->slide( *mBoard,-1);
            repaint();
            //update();
            break;
        }
    case Qt::Key_Right:
        {                   
            mPair->slide( *mBoard, 1);
            repaint();
            //update();
            break;
        }
    case Qt::Key_G:
        {
            qDebug() << "G";
            mBoard->drawCellNo();
            repaint();
            //update();
            break;
        }
    case Qt::Key_Up:
    case Qt::Key_Down:
        {
            mPair->rotate(*mBoard);
            repaint();
            break;
        }
    }
}

void GameWindow::mousePressEvent( QMouseEvent* /*aMouseEvent*/ )
{   
    if( mGameState == GameEnded ){
        emit gameEnded();
    } else {
        stopGame();
        emit pauseGame();
    }
}

QSize GameWindow::sizeHint() const
{
    return QSize(480,720);
}

void GameWindow::slotGameEnded()
{
    mSenser->setObjectName(0);
    mGameState = GameEnded;
    mThread->pauseGame();
    emit gameEnded();
}




