#include <stdio.h>
#include <stdlib.h>

#include <SDL.h>
#include <SDL_gles.h>
#include <GLES/gl.h>

#include "override.h"
#include "private.h"
#include "debug.h"

#define DEBUG_DOMAIN "SDLGL"

enum {
	SDL_GL_CONTEXT_MAJOR_VERSION = 17,
	/* I'm unsure about the below ones: */
	SDL_GL_CONTEXT_MINOR_VERSION = 19,
};
	

static SDL_GLES_Version desired_version = SDL_GLES_VERSION_NONE;
static SDL_bool gles_inited = SDL_FALSE;
static SDL_GLES_Context *context = NULL;

static int gles_init()
{
	if (!gles_inited) {
		int res = SDL_GLES_Init(desired_version);
		if (res != 0) return res;
		TRACE("SDL_GLES initialized (res=%d)\n", res);
		gles_inited = SDL_TRUE;

		/* Palm SDL has a 16 bit depth buffer by default,
		 * but I'm going to set it to 8 by default,
		 * because it's such a nice number. */
		res = SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, 8);
		if (res != 0) return res;
	}
	return 0;
}

static void gles_quit()
{
	if (gles_inited) {
		if (context) {
			SDL_GLES_DeleteContext(context);
			context = NULL;
		}
		SDL_GLES_Quit();
		gles_inited = SDL_FALSE;
	}
}

static inline int mini(int i1, int i2)
{
	if (i1 < i2) return i1;
	else return i2;
}

static inline int absi(int i)
{
	if (i < 0) return -i;
	else return i;
}

static void make_box(struct box *b, const struct point *p1, const struct point *p2)
{
	b->x = mini(p1->x, p2->x);
	b->y = mini(p1->y, p2->y);
	b->w = absi(p1->x - p2->x);
	b->h = absi(p1->y - p2->y);
}

static void p_scale(struct point *p) 
{
	const int ox = p->x, oy = p->y;
	switch (s_rotate) {
		case PDL_ORIENTATION_270:
			p->x = r_size.w - oy * s_scale.x;
			p->y = ox * s_scale.y;
		default:
			break;
	}
}

static void p_unscale(struct point *p) 
{
	const int ox = p->x, oy = p->y;
	switch (s_rotate) {
		case PDL_ORIENTATION_270:
			p->y = ox / s_scale.x;
			p->x = v_size.w - oy / s_scale.y;
		default:
			break;
	}
}

static int filter_active_event(const SDL_ActiveEvent *event) {
	SDL_Event fakeevent;
	SDL_ActiveEvent *fake = &fakeevent.active;
	fake->type = SDL_ACTIVEEVENT;
	fake->gain = event->gain;
	fake->state = 0;
	if (event->state & SDL_APPINPUTFOCUS) {
		TRACE("Sending %s focus event\n", fake->gain ? "gain" : "lose");
		PDL_NotifyFocus(fake->gain);
		fake->state = SDL_APPACTIVE;
	}
	if (fake->state) {
		return SDL_PushEvent(&fakeevent);
	}
	return 0;
}

static int filter_mouse_motion(const SDL_MouseMotionEvent *event) {
	SDL_Event fakeevent;
	SDL_MouseMotionEvent *fake = &fakeevent.motion;
	*fake = *event;
	struct point p = { event->x, event->y };
	struct point prel = { event->xrel, event->yrel };
	p_unscale(&p);
	p_unscale(&prel);
	fake->x = p.x;
	fake->y = p.y;
	fake->xrel = prel.x;
	fake->yrel = prel.y;
	return SDL_PushEvent(&fakeevent);
}

static int filter_mouse_button(const SDL_MouseButtonEvent *event) {
	SDL_Event fakeevent;
	SDL_MouseButtonEvent *fake = &fakeevent.button;
	*fake = *event;
	struct point p = { event->x, event->y };
	p_unscale(&p);
	fake->x = p.x;
	fake->y = p.y;
	return SDL_PushEvent(&fakeevent);
}

static int filter_event(const SDL_Event *event)
{
	switch (event->type) {
	case SDL_VIDEORESIZE:
		TRACE("Deleted a video resize event\n");
		return 0;
	case SDL_ACTIVEEVENT:
		return filter_active_event(&event->active);
	case SDL_MOUSEMOTION:
		return filter_mouse_motion(&event->motion);
	case SDL_MOUSEBUTTONDOWN:
	case SDL_MOUSEBUTTONUP:
		return filter_mouse_button(&event->button);
	}
	return 1;
}

void SDLPRE_RefreshScale()
{
	switch (v_orient) {
		case PDL_ORIENTATION_270:
			s_rotate = PDL_ORIENTATION_270;
			s_scale.x = (float)r_size.w / v_size.h;
			s_scale.y = (float)r_size.h / v_size.w;
			TRACE("Going for a 270 deg rotation\n");
		break;
		default:
			TRACE("This orientation is not handled yet\n");
		break;
	}
}

int SDL_Init(Uint32 flags)
{
	OVERRIDES(SDL_Init, (Uint32 flags), int);
	TRACE("called sdl_init with flags = 0x%x\n", flags);

	flags |= SDL_INIT_TIMER; /* We always want timers. */
	flags &= ~SDL_INIT_JOYSTICK; /* We override it. */

	int res = SUPER(flags);

	if (res == 0) {
		/* Success, hook some stuff. */
		SDL_ShowCursor(SDL_DISABLE); /* Pre disables cursor by default. */
		SDL_SetEventFilter(&filter_event);
	}

	return res;
}

void SDL_Quit()
{
	OVERRIDES(SDL_Quit, (), void);
	TRACE("called sdl_quit\n");
	gles_quit();
	SUPER();
}

int SDL_GL_SetAttribute(SDL_GLattr attr, int value)
{
	gles_init();
	switch (attr) {
		case SDL_GL_CONTEXT_MAJOR_VERSION:
			TRACE("Requested GL major version %d\n", value);
			switch (value) {
				case 1:
					desired_version = SDL_GLES_VERSION_1_1;
					return 0;
				case 2:
					desired_version = SDL_GLES_VERSION_2_0;
					return 0;
				default:
					SDL_SetError("Invalid GLES major version: %d\n", value);
					return -1;
			}
			break;
		case SDL_GL_CONTEXT_MINOR_VERSION:
			TRACE("Requested GL minor version %d\n", value);
			return 0;
		case SDL_GL_DEPTH_SIZE:
			TRACE("Requested depth buffer size %d\n", value);
			SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, value);
			return 0;
		default:
			WARN("Application asks for unknown GL attribute %u, value %d\n", attr, value);
			return 0;
	}
}

SDL_Surface * SDL_SetVideoMode(int width, int height, int bpp, Uint32 flags)
{
	OVERRIDES(SDL_SetVideoMode, (int width, int height, int bpp, Uint32 flags), SDL_Surface*);
	TRACE("Called SetVideoMode(%d, %d, %d, 0x%x)\n", width, height, bpp, flags);
	if (flags & SDL_OPENGL) {
		TRACE("Requested OpenGL video mode\n");
		flags &= ~SDL_OPENGL; /* Our platform SDL doesn't handle it. */

		int res = gles_init();
		if (res != 0 ) return NULL;

		if (!context) {
			context = SDL_GLES_CreateContext();
			if (!context) return NULL;
		}

		v_size.w = width;
		v_size.h = height;
		SDLPRE_RefreshScale();

		/* Override with fullscreen */
		screen = SUPER(r_size.w, r_size.h, bpp, flags | SDL_FULLSCREEN);
		if (!screen) return NULL;

		res = SDL_GLES_MakeCurrent(context);
		if (res != 0) return NULL;

		return screen;
	} else {
		if (gles_inited) {
			gles_quit();
		}

		screen = NULL;

		return SUPER(width, height, bpp, flags);
	}
}

void SDL_GL_SwapBuffers(void)
{
	SDL_GLES_SwapBuffers();
}

void SDL_SetEventFilter(SDL_EventFilter filter)
{
	OVERRIDES(SDL_SetEventFilter, (SDL_EventFilter filter), void);
	if (filter != &filter_event) {
		WARN("This application uses SetEventFilter, which is not implemented.\n");
	}
	SUPER(filter);
}

Uint8 SDL_GetMouseState(int *x, int *y)
{
	/* Stupid applications polling the mouse */
	OVERRIDES(SDL_GetMouseState, (int *x, int *y), Uint8);
	int sx, sy;
	Uint8 state = SUPER(&sx, &sy);
	struct point p = { sx, sy };
	p_unscale(&p);
	*x = p.x;
	*y = p.y;
	return state;
}

int SDL_UpperBlit(SDL_Surface *src, SDL_Rect *srcrect,
			 SDL_Surface *dst, SDL_Rect *dstrect)
{
	OVERRIDES(SDL_UpperBlit, (SDL_Surface *src, SDL_Rect *srcrect, SDL_Surface *dst, SDL_Rect *dstrect), int);
	if (dst == screen || src == screen) {
		WARN("Someone is trying to blit from/to screen; this breaks the fourth wall\n");
	}
	return SUPER(src, srcrect, dst, dstrect);
}

void glLoadIdentity()
{
	OVERRIDES(glLoadIdentity, (void), void);
	SUPER();
	GLint mode;
	glGetIntegerv(GL_MATRIX_MODE, &mode);
	if (mode == GL_PROJECTION) {
		switch (s_rotate) {
		case PDL_ORIENTATION_270:
			glRotatef(270.0f, 0.0, 0.0, -1.0f);
			break;
		default:
			/* Not handled yet, really. */
			break;
		}
	}
}

void glLoadMatrixf(const GLfloat * m)
{
	glLoadIdentity();
	glMultMatrixf(m);
}

void glViewport(GLint x, GLint y, GLsizei width, GLsizei height)
{
	OVERRIDES(glViewport, (GLint x, GLint y, GLsizei width, GLsizei height), void);
	struct point p1 = { x, y };
	struct point p2 = { x + width, y + height };
	struct box b;
	p_scale(&p1);
	p_scale(&p2);
	make_box(&b, &p1, &p2);
	SUPER(b.x, b.y, b.w, b.h);
}

void glScissor(GLint x, GLint y, GLsizei width, GLsizei height)
{
	OVERRIDES(glScissor, (GLint x, GLint y, GLsizei width, GLsizei height), void);
	struct point p1 = { x, y };
	struct point p2 = { x + width, y + height };
	struct box b;
	p_scale(&p1);
	p_scale(&p2);
	make_box(&b, &p1, &p2);
	SUPER(b.x, b.y, b.w, b.h);
}


