#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <glib.h>
#include <glib-object.h>
#include <libosso.h>
#include <hildon-mime.h>

#include "PDL.h"
#include "private.h"
#include "debug.h"

#define DEBUG_DOMAIN "PDL"
#define SCREEN_TIMEOUT_TIME (45*1000)

static GMainContext *main_context = NULL;
static GMainLoop *main_loop = NULL;
static osso_context_t *osso_context = NULL;
static char * exepath = NULL;
static char * exename = NULL;
static char * servicename = NULL;

/* Current status */
static SDL_bool dimming_prevention_enabled = FALSE;
static SDL_bool system_pause_ui_enabled = FALSE;



static SDL_TimerID dimming_timer_id = 0;

static char* getexepath()
{
	pid_t pid = getpid();
	char * linkname = g_strdup_printf("/proc/%i/exe", pid);
	char * link = g_file_read_link(linkname, NULL);
	g_free(linkname);
	return link;
}

PDL_Err PDL_Init(unsigned int flags)
{
	g_type_init();

	exepath = getexepath();
	if (!exepath) {
		return PDL_SYSTEMERROR_FILE;
	}

	exename = g_path_get_basename(exepath);
	g_set_prgname(exename);

	main_context = g_main_context_default();
	main_loop = g_main_loop_new(NULL, FALSE);

	const gchar *srv = g_getenv("PREENV_APPID");
	if (srv && strlen(srv) > 0) {
		servicename = g_strdup(srv);
	} else {
		servicename = g_strdup_printf("com.javispedro.preenv.%s", exename);
	}

	TRACE("Init: Service name is %s\n", servicename);

	osso_context = osso_initialize(servicename, "0.1", TRUE, NULL);
	if (!osso_context) {
		return PDL_ECONNECTION;
	}

	PDL_PumpEvents();

	return PDL_NOERROR;
}

void PDL_Quit()
{
	if (dimming_timer_id) {
		SDL_RemoveTimer(dimming_timer_id);
		dimming_timer_id = 0;
	}
	dimming_prevention_enabled = FALSE;
	system_pause_ui_enabled = FALSE;

	free(exepath);
	exepath = NULL;
	exename = NULL;
	free(servicename);
	servicename = NULL;
	osso_deinitialize(osso_context);
	osso_context = NULL;
	g_main_loop_unref(main_loop);
	main_loop = NULL;
	g_main_context_unref(main_context);
	main_context = NULL;
}

/* Stupid, stupid... */
#define CHECK_IMPLICIT_PDL_INIT() \
	if (!osso_context) { \
		PDL_Err err = PDL_Init(0); \
		if (err) return err; \
	}

PDL_Err PDL_BannerMessagesEnable(PDL_bool enable)
{
	// TODO: hildon_gtk_window_set_do_not_disturb from SDL
	/* On a second thought, maybe I should not forward this as most games just
       call it at startup and never remove the flag. */
	CHECK_IMPLICIT_PDL_INIT();
	TRACE("TODO: Set do_not_disturb flag to %s\n",
		enable ? "true" : "false");
	return PDL_NOERROR;
}

PDL_Err PDL_CustomPauseUiEnable(PDL_bool enable)
{
	if (enable) {
		TRACE("Application handles pause events\n");
		system_pause_ui_enabled = FALSE;
	} else {
		TRACE("I should provide pause UI, but I won't!\n");
		system_pause_ui_enabled = TRUE;
	}

	return PDL_NOERROR;
}

PDL_Err PDL_GetLanguage(char *buffer, int bufferLen)
{
	if (!buffer || bufferLen <= 0) {
		return PDL_INVALIDINPUT;
	}
	const char *lang = getenv("LANG");
	if (!lang || strlen(lang) == 0) {
		lang = "en_US";
	}

	TRACE("Application asked for lang, giving %s\n", lang);

	if (bufferLen < (strlen(lang) + 1)) {
		return PDL_STRINGTOOSMALL;
	}
	strcpy(buffer, lang);
	return PDL_NOERROR;
}

PDL_Err PDL_LaunchBrowser(const char* url)
{
	CHECK_IMPLICIT_PDL_INIT();
	TRACE("Launch browser, url=\"%s\"\n", url);
	GError *error = NULL;
	if (hildon_uri_open(url, NULL, &error)) {
		return PDL_NOERROR;
	} else {
		WARN("Failed to open browser: %s\n", error->message);
		g_error_free(error);
		return PDL_ECONNECTION;
	}
}

PDL_Err PDL_NotifyMusicPlaying(PDL_bool MusicPlaying)
{
	CHECK_IMPLICIT_PDL_INIT();
	TRACE("Music is%splaying\n", MusicPlaying ? " " : " not ");
	/* TODO: Integrate this with libplayback. */
	return PDL_NOERROR;
}

static Uint32 screen_timeout_cb(Uint32 interval, void *param)
{
	osso_return_t res = osso_display_blanking_pause(osso_context);
	if (res == OSSO_OK) {
		return interval;
	} else {
		dimming_timer_id = 0;
		return 0;
	}
}

PDL_Err PDL_ScreenTimeoutEnable(PDL_bool enable)
{
	CHECK_IMPLICIT_PDL_INIT();
	if (dimming_timer_id) {
		SDL_RemoveTimer(dimming_timer_id);
		dimming_timer_id = 0;
	}
	if (!enable) {
		dimming_prevention_enabled = SDL_TRUE;
		/* Do an initial blanking pause. */
		osso_return_t res = osso_display_blanking_pause(osso_context);
		if (res != OSSO_OK) {
			return PDL_ECONNECTION;
		}
		/* Schedule next one */
		dimming_timer_id = SDL_AddTimer(SCREEN_TIMEOUT_TIME,
			screen_timeout_cb, NULL);
	} else {
		dimming_prevention_enabled = SDL_FALSE;
	}

	TRACE("Screen dimming prevention %s\n",
		dimming_prevention_enabled ? "ON" : "OFF");

	return PDL_NOERROR;
}

PDL_Err PDL_SetOrientation(PDL_Orientation orientation)
{
	TRACE("Switch orientation to %hu\n", orientation);
	v_orient = orientation;
	SDLPRE_RefreshScale();
	return PDL_NOERROR;
}


void PDL_PumpEvents()
{
	g_main_context_iteration(main_context, FALSE);
}

void PDL_NotifyFocus(SDL_bool focus)
{
	if (dimming_prevention_enabled) {
		if (focus) {
			/* Gaining focus; readd timer. */
			if (!dimming_timer_id) {
				dimming_timer_id = SDL_AddTimer(SCREEN_TIMEOUT_TIME,
					screen_timeout_cb, NULL);
			}
		} else {
			/* Losing focus, kill timer. */
			if (dimming_timer_id) {
				SDL_RemoveTimer(dimming_timer_id);
				dimming_timer_id = 0;
			}
		}
		TRACE("Readjusting dimming prevention timer\n");
	}
}

