#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include <SDL.h>

#include "override.h"
#include "debug.h"

#define DEBUG_DOMAIN "ACCEL"
#define ACCEL_FILE "/sys/class/i2c-adapter/i2c-3/3-001d/coord"

static struct _SDL_Joystick
{
	FILE* fd;
	Sint16 x, y, z;
	int bad;
} accel = { 0 };

static inline Sint16 scale(long v)
{
	return (v * 32767) / 1020;
}

static inline void maybe_update(Uint8 axis, Sint16 * value, Sint16 new)
{
	Sint16 scaled = scale(new);
	if (*value != scaled) {
		// TODO: PushEvent axis
		*value = scaled;
	}
}

static void accel_update()
{
	if (!accel.fd) return;
	int r, x, y, z;

	fseek(accel.fd, 0, SEEK_SET);
	r = fscanf(accel.fd, "%i %i %i", &x, &y, &z);
	if (r != 3) return;

	maybe_update(0, &accel.x, y);
	maybe_update(1, &accel.y, -x);
	maybe_update(2, &accel.z, z);
}

int SDL_NumJoysticks(void)
{
	return 1;
}

SDL_Joystick *SDL_JoystickOpen(int index)
{
	if (index == 0) {
		TRACE("Opening accelerometer\n");
		accel.fd = fopen(ACCEL_FILE, "r");
		if (!accel.fd) {
			SDL_SetError("Failure to open accelerometer");
			return NULL;
		}
		setvbuf(accel.fd, NULL, _IONBF, 0);
		return &accel;
	} else {
		SDL_SetError("Invalid joystick index %d", index);
		return NULL;
	}
}

void SDL_JoystickClose(SDL_Joystick *joystick)
{
	TRACE("Closing accelerometer\n");
	fclose(accel.fd);
	accel.fd = NULL;
}

int SDL_JoystickEventState(int state)
{
	if (state == SDL_QUERY) {
		TRACE("Application queries current joystick event state\n");
	} else if (state == SDL_ENABLE) {
		TRACE("Application wants us to poll accelerometer\n");
	} else if (state == SDL_IGNORE) {
		TRACE("Application will poll accelerometer by itself\n");
	} else {
		TRACE("Application wants to kill the companion cube\n");
	}
	return SDL_DISABLE;
}

void SDL_JoystickUpdate()
{
	accel_update();
}

int SDL_JoystickOpened(int index)
{
	if (index == 0) {
		return accel.fd ? 1 : 0;
	} else {
		return 0;
	}
}

Sint16 SDL_JoystickGetAxis(SDL_Joystick *joystick, int axis)
{
	switch (axis) {
		case 0:
			return accel.x;
		case 1:
			return accel.y;
		case 2:
			return accel.z;
		default:
			return 0;
	}
}

