#include <stdlib.h>
#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <json-glib/json-glib.h>

#ifndef JSON_NODE_HOLDS
#define JSON_NODE_HOLDS(node,t) (json_node_get_node_type ((node)) == (t))
#endif

#ifndef JSON_NODE_HOLDS_OBJECT
#define JSON_NODE_HOLDS_OBJECT(node) (JSON_NODE_HOLDS ((node), JSON_NODE_OBJECT))
#endif

#ifndef JSON_NODE_HOLDS_VALUE
#define JSON_NODE_HOLDS_VALUE(node) (JSON_NODE_HOLDS ((node), JSON_NODE_VALUE))
#endif

#define DBUS_SERVICE_FILE_PATH "/usr/share/dbus-1/services/"
#define HILDON_DESKTOP_FILE_PATH "/usr/share/applications/hildon/"
#define HILDON_ICON_PATH "/usr/share/icons/hicolor/scalable/hildon/"

static gchar ** input_files = NULL;
static gchar * preenv_path = NULL;
static gboolean dry_run = FALSE;

static GOptionEntry entries[] = {
	{ "preenv", 'p', 0, G_OPTION_ARG_FILENAME, &preenv_path, "Path to Preenv", "PATH" },
	{ "dry-run", 'n', 0, G_OPTION_ARG_NONE, &dry_run, "Show desktop files instead of writing them", NULL },
	{ G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &input_files, "Path to appinfo.json file", "FILE" },
	{ NULL }
};

static const gchar * get_appinfo(JsonObject *obj, const gchar *key)
{
	JsonNode *node = json_object_get_member(obj, key);
	if (node && JSON_NODE_HOLDS_VALUE(node)) {
		return json_node_get_string(node);
	} else {
		g_printerr("Not a string attribute: %s\n", key);
		return NULL;
	}
}

static void write_file(const gchar *file, const gchar *contents)
{
	if (dry_run) {
		g_print("Would create \"%s\" with:\n%s\n", file, contents);
	} else {
		GError *error = NULL;
		if (g_file_set_contents(file, contents, -1, &error)) {
			g_print("Wrote \"%s\"\n", file);
		} else {
			g_printerr("Cannot write file \"%s\": %s\n", file, error->message);
			exit(4);
		}
	}
}

static void copy_file(const gchar *file, const gchar *dest)
{
	if (dry_run) {
		g_print("Would copy \"%s\" to \"%s\"\n", file, dest);
	} else {
		GFile *src = g_file_new_for_path(file);
		GFile *dst = g_file_new_for_path(dest);
		GError *error = NULL;
		if (g_file_copy(src, dst, G_FILE_COPY_OVERWRITE,
			NULL, NULL, NULL, &error)) {
			g_print("Wrote \"%s\"\n", dest);
		} else {
			g_printerr("Cannot copy file \"%s\" to \"%s\": %s\n",
				file, dest, error->message);
			exit(4);
		}
		g_object_unref(dst);
		g_object_unref(src);
	}
}

static void global_sanity_check()
{
	if (!preenv_path) {
		preenv_path = g_getenv("PREENV_ROOT");
	}
	if (!preenv_path) {
		g_printerr("Cannot find PREENV_ROOT. You must set one with '-p'\n");
		exit(1);
	}
	if (!g_file_test(preenv_path, G_FILE_TEST_IS_DIR)) {
		g_printerr("Cannot access Preenv directory: %s\n", preenv_path);
		exit(3);
	}
}

static void sanity_check(JsonObject *app, const gchar * input_path)
{
	const gchar *id = get_appinfo(app, "id");
	if (!id) {
		g_printerr("'%s' attribute is mandatory\n", "id");
		exit(2);
	}
	const gchar *bin = get_appinfo(app, "main");
	if (!bin) {
		g_printerr("'%s' attribute is mandatory\n", "main");
		exit(2);
	}
	const gchar *type = get_appinfo(app, "type");
	if (!type || (g_ascii_strcasecmp(type, "game") != 0 &&
		g_ascii_strcasecmp(type, "pdk")) != 0) {
		g_printerr("'%s' attribute is mandatory (and should be '%s')\n",
			"type", "pdk");
		exit(2);
	}
	gchar *binpath = g_build_filename(input_path, bin, NULL);
	if (!g_file_test(binpath, G_FILE_TEST_IS_EXECUTABLE)) {
		g_printerr("'%s' is not executable\n", binpath);
		exit(3);
	}
	if (!get_appinfo(app, "version")) {
		g_printerr("'%s' attribute is mandatory\n", "version");
		exit(2);
	}
	if (!get_appinfo(app, "title")) {
		g_printerr("'%s' attribute is mandatory\n", "title");
		exit(2);
	}
	const gchar *icon = get_appinfo(app, "icon");
	if (icon) {
		if (!g_str_has_suffix(icon, ".png")) {
			g_printerr("Currently, icon must be a png file\n");
			exit(2);
		}
	}
	g_free(binpath);
}

static void copy_icon(const gchar * id, const gchar * srciconpath)
{
	gchar *dsticonname = g_strdup_printf("%s.png", id);
	gchar *dsticonpath = g_build_filename(HILDON_ICON_PATH, dsticonname, NULL);

	copy_file(srciconpath, dsticonpath);

	g_free(dsticonpath);
	g_free(dsticonname);
}

static void build_service_file(JsonObject *app, const gchar * input_path)
{
	const gchar *id = get_appinfo(app, "id");
	gchar *path = g_strdup_printf(DBUS_SERVICE_FILE_PATH "%s.service", id);
	gchar *wrapperpath = g_build_filename(preenv_path, "wrapper.sh", NULL);
	gchar *binpath = g_build_filename(input_path, get_appinfo(app, "main"), NULL);
	gchar *contents = g_strdup_printf(
"[D-BUS Service]\n"
"Name=%s\n"
"Exec=\"%s\" \"%s\" \"%s\"\n"
, id, wrapperpath, id, binpath);
	write_file(path, contents);
	g_free(path);
	g_free(wrapperpath);
	g_free(binpath);
	g_free(contents);
}

static void build_desktop_file(JsonObject *app, const gchar * input_path)
{
	const gchar *id = get_appinfo(app, "id");
	gchar *path = g_strdup_printf(HILDON_DESKTOP_FILE_PATH "%s.desktop", id);
	gchar *wrapperpath = g_build_filename(preenv_path, "wrapper.sh", NULL);
	gchar *binpath = g_build_filename(input_path, get_appinfo(app, "main"), NULL);
	GString *contents = g_string_sized_new(500);
	g_string_printf(contents,
"[Desktop Entry]\n"
"Encoding=UTF-8\n"
"Type=Application\n"
"Name=%s\n"
"Version=%s\n"
"Exec=\"%s\" \"%s\" \"%s\"\n"
"X-Osso-Type=application/x-executable\n"
"X-Osso-Service=%s\n"
, get_appinfo(app, "title"), get_appinfo(app, "version"),
	wrapperpath, id, binpath, id);
	if (get_appinfo(app, "icon")) {
		gchar *srciconpath = g_build_filename(input_path, get_appinfo(app, "icon"), NULL);
		copy_icon(id, srciconpath);
		g_string_append_printf(contents, "Icon=%s\n", id);
		g_free(srciconpath);
	}
	write_file(path, contents->str);
	g_free(binpath);
	g_free(wrapperpath);
	g_free(path);
	g_string_free(contents, TRUE);
}

static void process_file(const gchar * input_file)
{
	GError *error = NULL;
	JsonParser *parser;
	JsonNode *root;
	JsonObject *obj;
	gchar * input_path = g_path_get_dirname(input_file);
	char * full_input_path = realpath(input_path, NULL);

	parser = json_parser_new();
	json_parser_load_from_file(parser, input_file, &error);
	if (error) {
		g_printerr("Unable to parse \"%s\": %s\n", input_file, error->message);
		exit(2);
	}

	root = json_parser_get_root(parser);
	if (!JSON_NODE_HOLDS_OBJECT(root)) {
		g_printerr("JSON file is not an object\n");
		exit(2);
	}
	obj = json_node_get_object(root);

	sanity_check(obj, full_input_path);
	build_service_file(obj, full_input_path);
	build_desktop_file(obj, full_input_path);

	free(full_input_path);
	g_free(input_path);
	g_object_unref(parser);
}

int main(int argc, char ** argv)
{
	GError *error = NULL;
	GOptionContext *context;

	g_type_init();

	context = g_option_context_new("- generate Hildon desktop files from WebOS appinfo");
	g_option_context_add_main_entries(context, entries, NULL);

	if (!g_option_context_parse(context, &argc, &argv, &error)) {
		g_printerr("Invalid options passed: %s\n", error->message);
		return 1;
	}

	global_sanity_check();

	if (input_files) {
		for (gint i = 0; input_files[i]; i++) {
			process_file(input_files[i]);
		}
	} else {
		g_printerr("Warning: no files processed\n");
	}

	return 0;
}
