#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <unistd.h>

#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <libosso.h>
#include <hildon-mime.h>

#include "PDL.h"
#include "private.h"
#include "debug.h"

#define DEBUG_DOMAIN "PDL"
#define SCREEN_TIMEOUT_TIME (45*1000)

#define BT_SERVICE				"org.bluez"
#define BT_DEF_ADAPTER_IF		"org.bluez.Manager"
#define BT_DEF_ADAPTER_PATH		"/"
#define BT_DEF_ADAPTER_METHOD	"DefaultAdapter"
#define BT_ADAPTER_IF			"org.bluez.Adapter"
#define BT_ADAPTER_GETPROPS		"GetProperties"

static GMainContext *main_context = NULL;
static GMainLoop *main_loop = NULL;
static osso_context_t *osso_context = NULL;
static char * exepath = NULL;
static char * servicename = NULL;

/* Current status */
static SDL_bool dimming_prevention_enabled = FALSE;
static SDL_bool system_pause_ui_enabled = FALSE;

static gchar * last_error = NULL;
static SDL_TimerID dimming_timer_id = 0;

static char* get_exe_path()
{
	pid_t pid = getpid();
	char * linkname = g_strdup_printf("/proc/%i/exe", pid);
	char * link = g_file_read_link(linkname, NULL);
	g_free(linkname);
	return link;
}

static gchar* get_device_name()
{
	GError *error = NULL;
	gchar *name = NULL;
	DBusGConnection *bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (!bus) {
		WARN("Failed to get system bus: %s", error->message);
		return NULL;
	}
	DBusGProxy* mgr_proxy = dbus_g_proxy_new_for_name(bus,
		BT_SERVICE, BT_DEF_ADAPTER_PATH, BT_DEF_ADAPTER_IF);
	DBusGProxy* adp_proxy = NULL;
	if (!dbus_g_proxy_call(mgr_proxy, BT_DEF_ADAPTER_METHOD, &error,
		G_TYPE_INVALID, DBUS_TYPE_G_PROXY, &adp_proxy, G_TYPE_INVALID)) {
		WARN("Failed to get Bluez manager: %s", error->message);
		goto err_clean_mgr;
	}

	GHashTable *props = NULL;
	dbus_g_proxy_set_interface(adp_proxy, BT_ADAPTER_IF);
	if (!dbus_g_proxy_call(adp_proxy, BT_ADAPTER_GETPROPS, &error,
		G_TYPE_INVALID,
		dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE),
		&props, G_TYPE_INVALID)) {
		WARN("Failed to get Bluez properties: %s", error->message);
		goto err_clean_adp;
	}

	GValue *value = g_hash_table_lookup(props, "Name");
	if (!value) {
		WARN("Failed to get Bluez propertie 'Name'");
		goto err_clean_ht;
	}

	name = g_value_dup_string(value);

err_clean_ht:
	g_hash_table_destroy(props);
err_clean_adp:
	g_object_unref(adp_proxy);
err_clean_mgr:
	g_object_unref(mgr_proxy);
	return name;
}

static void set_error(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	last_error = g_strdup_vprintf(format, args);
	va_end(args);
}

static PDL_Err copy_to_user(const char *data, char *buffer, int bufferLen)
{
	if (!buffer || bufferLen <= 0) {
		set_error("buffer is NULL or bufferLen is 0");
		return PDL_INVALIDINPUT;
	}
	if (bufferLen < (strlen(data) + 1)) {
		set_error("bufferLen not enough to hold %d characters", strlen(data));
		return PDL_STRINGTOOSMALL;
	}
	strcpy(buffer, data);
	return PDL_NOERROR;
}

PDL_Err PDL_Init(unsigned int flags)
{
	g_type_init();

	exepath = get_exe_path();
	if (!exepath) {
		/* Some applications also need /proc, so we might never reach here */
		WARN("Is /proc mounted?");
		set_error("Failed to get executable path");
		return PDL_SYSTEMERROR_FILE;
	}

	gchar * exename = g_path_get_basename(exepath);
	g_set_prgname(exename);

	main_context = g_main_context_default();
	main_loop = g_main_loop_new(NULL, FALSE);

	const gchar *srv = g_getenv("PREENV_APPID");
	if (srv && strlen(srv) > 0) {
		servicename = g_strdup(srv);
	} else {
		/* Use the executable name as D-BUS service name, properly sanitized. */
		g_strcanon(exename,
			G_CSET_A_2_Z G_CSET_a_2_z G_CSET_DIGITS "_", '_');
		servicename = g_strdup_printf("com.javispedro.preenv.%s", exename);
	}
	g_free(exename);

	TRACE("Init: Service name is %s", servicename);

	osso_context = osso_initialize(servicename, "0.1", TRUE, NULL);
	if (!osso_context) {
		WARN("Failed to initialize libosso");
		set_error("Failed to connect to services");
		return PDL_ECONNECTION;
	}

	PDL_PumpEvents();

	return PDL_NOERROR;
}

void PDL_Quit()
{
	if (dimming_timer_id) {
		SDL_RemoveTimer(dimming_timer_id);
		dimming_timer_id = 0;
	}
	dimming_prevention_enabled = FALSE;
	system_pause_ui_enabled = FALSE;

	free(exepath);
	exepath = NULL;
	free(servicename);
	servicename = NULL;
	osso_deinitialize(osso_context);
	osso_context = NULL;
	g_main_loop_unref(main_loop);
	main_loop = NULL;
	g_main_context_unref(main_context);
	main_context = NULL;
	g_free(last_error);
	last_error = NULL;
}

/* Stupid, stupid... */
#define CHECK_IMPLICIT_PDL_INIT() \
	if (!osso_context) { \
		PDL_Err err = PDL_Init(0); \
		if (err) return err; \
	}

PDL_Err PDL_BannerMessagesEnable(PDL_bool enable)
{
	CHECK_IMPLICIT_PDL_INIT();
	TRACE("Set do_not_disturb flag to %s",
		enable ? "false" : "true");
	/* I am not sure if this is a good idea after all */
	X11_SetDoNotDisturb(enable ? SDL_FALSE : SDL_TRUE);
	return PDL_NOERROR;
}

PDL_Err PDL_CustomPauseUiEnable(PDL_bool enable)
{
	if (enable) {
		TRACE("Application handles pause events");
		system_pause_ui_enabled = FALSE;
	} else {
		TRACE("I should provide pause UI, but I won't!");
		system_pause_ui_enabled = TRUE;
	}

	return PDL_NOERROR;
}

PDL_Err PDL_GetCallingPath(char *buffer, int bufferLen)
{
	TRACE("Application asked for calling path, giving './'");
	return copy_to_user("./", buffer, bufferLen);
}

PDL_Err PDL_GetDeviceName(char *buffer, int bufferLen)
{
	PDL_Err err;
	gchar *name = get_device_name();
	if (!name) {
		WARN("Unable to get device name");
		set_error("Unable to get device name");
		return PDL_EOTHER;
	}

	TRACE("Application asked for device name, giving '%s'", name);

	err = copy_to_user(name, buffer, bufferLen);
	g_free(name);
	return err;
}

const char *PDL_GetError(void)
{
	if (last_error) {
		return last_error;
	} else {
		return "No error";
	}
}

const char *PDL_GetHardware(void)
{
	return "pre";
}

PDL_Err PDL_GetLanguage(char *buffer, int bufferLen)
{
	const char *lang = getenv("LANG");
	if (!lang || strlen(lang) == 0) {
		lang = "en_US";
	}

	TRACE("Application asked for lang, giving %s", lang);

	return copy_to_user(lang, buffer, bufferLen);
}

PDL_Err PDL_GetNetInfo(const char *interfaceName, PDL_NetInfo * interfaceInfo)
{
	if (!interfaceName || !interfaceInfo) {
		set_error("interfaceName or interfaceInfo is NULL");
		return PDL_INVALIDINPUT;
	}

	/* Seems that most applications will say interfaceName=eth0 here */
	if (g_ascii_strcasecmp(interfaceName, "eth0") == 0) {
		int fd;
		struct ifreq ifr;
		TRACE("Getting info from wlan0");
		fd = socket(AF_INET, SOCK_DGRAM, 0);
		ifr.ifr_addr.sa_family = AF_INET;
		strncpy(ifr.ifr_name, "wlan0", IFNAMSIZ-1);

		if (ioctl(fd, SIOCGIFADDR, &ifr) < 0) {
			WARN("Failed to get interface address");
			set_error("Failed to get interface address");
			return PDL_SYSTEMERROR_NET;
		}
		interfaceInfo->ipaddress = ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr;
		TRACE("Found IP address 0x%x = %s", interfaceInfo->ipaddress,
			inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));

		if (ioctl(fd, SIOCGIFNETMASK, &ifr) < 0) {
			WARN("Failed to get interface netmask");
			set_error("Failed to get interface address");
			return PDL_SYSTEMERROR_NET;
		}
		interfaceInfo->netmask = ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr;
		TRACE("Found Netmask 0x%x = %s", interfaceInfo->netmask,
			inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));

		if (ioctl(fd, SIOCGIFBRDADDR, &ifr) < 0) {
			WARN("Failed to get interface netmask");
			set_error("Failed to get interface address");
			return PDL_SYSTEMERROR_NET;
		}
		interfaceInfo->broadcast = ((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr;
		TRACE("Found Broadcast address 0x%x = %s", interfaceInfo->broadcast,
			inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));

		return PDL_NOERROR;
	} else {
		WARN("I do not know how to translate interface '%s'", interfaceName);
		set_error("Unknown interface '%s'", interfaceName);
		return PDL_INVALIDINPUT;
	}
}

PDL_Err PDLNet_Get_Info(const char *interfaceName, PDL_NetInfo * interfaceInfo)
{
	return PDL_GetNetInfo(interfaceName, interfaceInfo);
}

PDL_Err PDL_GetOSVersion(PDL_OSVersion *version)
{
	TRACE("Application asks for OS version");
	version->majorVersion = 1;
	version->minorVersion = 4;
	version->revision = 5;
	version->versionStr = "Preenv 0.1.1";
	return PDL_NOERROR;
}

PDL_Err PDL_GetScreenMetrics(PDL_ScreenMetrics* metrics)
{
	TRACE("Application asks for screen metrics; giving true ones");
	metrics->horizontalPixels = 480;
	metrics->verticalPixels = 800;
	metrics->horizontalDPI = 213;
	metrics->verticalDPI = 183;
	metrics->aspectRatio = 1.164;
	return PDL_NOERROR;
}

PDL_Err PDL_GetUniqueID(char *buffer, int bufferLen)
{
	TRACE("Application asked for device ID, giving %s", "0000");
	return copy_to_user("0000", buffer, bufferLen);
}

PDL_Err PDL_LaunchBrowser(const char* url)
{
	CHECK_IMPLICIT_PDL_INIT();
	TRACE("Launch browser, url=\"%s\"", url);
	GError *error = NULL;
	if (hildon_uri_open(url, NULL, &error)) {
		return PDL_NOERROR;
	} else {
		WARN("Failed to open browser: %s", error->message);
		set_error("Failed to open browser: %s", error->message);
		g_error_free(error);
		return PDL_ECONNECTION;
	}
}

PDL_Err PDL_NotifyMusicPlaying(PDL_bool MusicPlaying)
{
	CHECK_IMPLICIT_PDL_INIT();
	TRACE("Music is%splaying", MusicPlaying ? " " : " not ");
	/* TODO: Integrate this with libplayback. */
	return PDL_NOERROR;
}

static Uint32 screen_timeout_cb(Uint32 interval, void *param)
{
	osso_return_t res = osso_display_blanking_pause(osso_context);
	if (res == OSSO_OK) {
		return interval;
	} else {
		dimming_timer_id = 0;
		return 0;
	}
}

PDL_Err PDL_ScreenTimeoutEnable(PDL_bool enable)
{
	CHECK_IMPLICIT_PDL_INIT();
	if (dimming_timer_id) {
		SDL_RemoveTimer(dimming_timer_id);
		dimming_timer_id = 0;
	}
	if (!enable) {
		dimming_prevention_enabled = SDL_TRUE;
		/* Do an initial blanking pause. */
		osso_return_t res = osso_display_blanking_pause(osso_context);
		if (res != OSSO_OK) {
			set_error("Cannot connect to services");
			return PDL_ECONNECTION;
		}
		/* Schedule next one */
		dimming_timer_id = SDL_AddTimer(SCREEN_TIMEOUT_TIME,
			screen_timeout_cb, NULL);
	} else {
		dimming_prevention_enabled = SDL_FALSE;
	}

	TRACE("Screen dimming prevention %s",
		dimming_prevention_enabled ? "ON" : "OFF");

	return PDL_NOERROR;
}

PDL_Err PDL_SetFirewallPortStatus(int port, PDL_bool open)
{
	/* Maemo has the best firewall of them all: none. */
	TRACE("%s network port %d", open ? "Open" : "Close", port);
	return PDL_NOERROR;
}

PDL_Err PDL_SetOrientation(PDL_Orientation orientation)
{
	TRACE("Switch orientation to %hu", orientation);
	v_orient = orientation;
	SDLPRE_RefreshScale();
	return PDL_NOERROR;
}

void PDL_PumpEvents()
{
	g_main_context_iteration(main_context, FALSE);
}

void PDL_NotifyFocus(SDL_bool focus)
{
	if (dimming_prevention_enabled) {
		if (focus) {
			/* Gaining focus; readd timer. */
			if (!dimming_timer_id) {
				dimming_timer_id = SDL_AddTimer(SCREEN_TIMEOUT_TIME,
					screen_timeout_cb, NULL);
			}
		} else {
			/* Losing focus, kill timer. */
			if (dimming_timer_id) {
				SDL_RemoveTimer(dimming_timer_id);
				dimming_timer_id = 0;
			}
		}
		TRACE("Readjusting dimming prevention timer");
	}
}

int PDL_isAppLicensedForDevice(const char *appid)
{
	WARN("Application is copy-protected. Sorry! (%s:%d)", __FILE__, __LINE__);
	return 0;
}

