#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include <SDL.h>

#include "override.h"
#include "debug.h"

#define DEBUG_DOMAIN "ACCEL"
#define ACCEL_FILE "/sys/class/i2c-adapter/i2c-3/3-001d/coord"
#define ACCEL_BUF_SIZE 256

#define EVENT_INTERVAL 100

static struct _SDL_Joystick
{
	int fd;
	Sint16 x, y, z;
	char opened;
} accel = { 0 };

static SDL_bool poll = SDL_TRUE;
static SDL_TimerID timer = 0;

static inline Sint16 scale(long v)
{
	return (v * 32767) / 1020;
}

static inline void maybe_update(Uint8 axis, Sint16 * value, Sint16 new)
{
	Sint16 scaled = scale(new);
	if (*value != scaled) {
		if (poll) {
			SDL_Event fakeevent;
			SDL_JoyAxisEvent *fake = &fakeevent.jaxis;
			fake->type = SDL_JOYAXISMOTION;
			fake->which = 0;
			fake->axis = axis;
			fake->value = scaled;
			SDL_PushEvent(&fakeevent);
		}
		*value = scaled;
	}
}

static inline int parse_values(const char *buf, int *x, int *y, int *z)
{
	return sscanf(buf, "%i %i %i", x, y, z) == 3;
}

static int accel_update()
{
	static char buffer[ACCEL_BUF_SIZE+1];
	ssize_t r;
	int x, y, z;

	if (!accel.opened) {
		TRACE("Timer called while accelerometer closed");
		return 0;
	}

	if (lseek(accel.fd, 0, SEEK_SET) != 0) {
		WARN("fseek failed");
		return 0;
	}
	if ((r = read(accel.fd, &buffer, ACCEL_BUF_SIZE)) <= 0) {
		WARN("failed to read accel file");
		return 0;
	}
	buffer[r] = '\0';
	if (!parse_values(buffer, &x, &y, &z)) {
		WARN("failed to parse values");
		return 0;
	}

	/* Note the rotation */
	maybe_update(0, &accel.x, y);
	maybe_update(1, &accel.y, -x);
	maybe_update(2, &accel.z, z);

	return 1;
}

static Uint32 accel_poll(Uint32 interval, void* param)
{
	if (!accel_update()) return 0;
	return interval;
}

static void update_polling()
{
	/* SDL normally uses event thread to poll joystick; I will set a timer to
     * get a similar behaviour. */
	if (accel.opened && poll) {
		/* Joystick is opened, and we want polling. */
		timer = SDL_AddTimer(EVENT_INTERVAL, accel_poll, NULL);
	} else if (timer) {
		/* Joystick is closed or polling is disabled, kill timer. */
		SDL_RemoveTimer(timer);
		timer = 0;
	}
}

int SDL_NumJoysticks(void)
{
	return 1;
}

const char *SDL_JoystickName(int index)
{
	if (index == 0) {
		return "Accelerometer";
	} else {
		SDL_SetError("No such joystick");
		return NULL;
	}
}

SDL_Joystick *SDL_JoystickOpen(int index)
{
	if (index == 0) {
		TRACE("Opening accelerometer");
		accel.fd = open(ACCEL_FILE, O_RDONLY|O_NONBLOCK);
		if (accel.fd < 0) {
			SDL_SetError("Failure to open accelerometer");
			return NULL;
		}
		accel.opened = 1;
		update_polling();
		return &accel;
	} else {
		SDL_SetError("Invalid joystick index %d", index);
		return NULL;
	}
}

void SDL_JoystickClose(SDL_Joystick *joystick)
{
	TRACE("Closing accelerometer");
	accel.opened = 0;
	update_polling();
	close(accel.fd);
	accel.fd = -1;
}

int SDL_JoystickEventState(int state)
{
	int cur_state = poll ? SDL_ENABLE : SDL_IGNORE;
	if (state == SDL_QUERY) {
		TRACE("Application queries current joystick event state");
	} else if (state == SDL_ENABLE) {
		TRACE("Application wants us to poll accelerometer");
		poll = SDL_TRUE;
	} else if (state == SDL_IGNORE) {
		TRACE("Application will poll accelerometer by itself");
		poll = SDL_FALSE;
	}
	update_polling();
	return cur_state;
}

void SDL_JoystickUpdate()
{
	accel_update();
}

int SDL_JoystickOpened(int index)
{
	if (index == 0) {
		return accel.opened;
	} else {
		return 0;
	}
}

Sint16 SDL_JoystickGetAxis(SDL_Joystick *joystick, int axis)
{
	switch (axis) {
		case 0:
			return accel.x;
		case 1:
			return accel.y;
		case 2:
			return accel.z;
		default:
			SDL_SetError("Joystick only has 3 axes");
			return 0;
	}
}

