// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QFile>
#include <QMap>
#include <QDebug>
#include <QMessageBox>
#include <QFileDialog>
#include <QToolBar>
#include <QMenuBar>
#include <QCloseEvent>
#include <QSettings>
#include <timeshop.hpp>
#include "plansplant/widgets.hpp"
#include "plansplant/export.hpp"
#ifdef Q_WS_MAEMO_5
#include <QStackedWidget>
#include <QX11Info>
#include <X11/Xcursor/Xcursor.h>
#include <QAbstractKineticScroller>
#include <QScrollBar>
#include <QStyleFactory>
#endif // Q_WS_MAEMO_5

namespace PlansPlant
{
  TasksTreeModel::Carrier* TasksTreeModel::carrier_from_index( const QModelIndex& Index )
  { return static_cast<Carrier*>( Index.internalPointer() ); } // carrier_from_index( const QModelIndex& )
  Task* TasksTreeModel::task_from_index( const QModelIndex& Index )
  {
    Task* Result = 0;
    if( Index.isValid() )
      if( Carrier* Item = carrier_from_index( Index ) )
	Result = &Item->task();
    return Result;
  } // task_from_index( const QModelIndex& )
  TasksTreeModel::Carrier::Carrier( Task& Object0, Pointer Parent0, Relation Rel0 ) : Object( Object0 ), Rel( Rel0 ), Parent( 0 ), Populated( false )
  {
    parent( Parent0 );
    if( !has_subitems() )
      Populated = true; // Because nothing to populate.
  } // Carrier( Task&, Pointer, Relation )
  TasksTreeModel::Carrier::~Carrier()
  {
    parent( 0 ); // Remove from parent
    while( !SubItems.empty() )
      delete SubItems.back();
  } // ~Carrier()
  void TasksTreeModel::Carrier::parent( Pointer NewParent )
  {
    if( NewParent != Parent )
    {
      if( Parent )
	Parent->SubItems.removeAll( this );
      Parent = NewParent;
      if( Parent && !Parent->SubItems.contains( this ) )
      {
	if( Rel == Subtask && Parent->SubItems.size() >= Parent->task().subtasks().size() )
	  Parent->SubItems.insert( Parent->task().subtasks().size()-1, this );
	else
	  Parent->SubItems.push_back( this );
      }
    }
  } // parent( Pointer )
  bool TasksTreeModel::Carrier::has_subitems() const { return !( Object.subtasks().empty() && Object.needs().empty() ); }
  const TasksTreeModel::Carrier::List& TasksTreeModel::Carrier::populate()
  {
    if( !Populated )
    {
      Populated = true;
      foreach( Task* SubTask, Object.subtasks() )
	if( SubTask )
	  new Carrier( *SubTask, this );
      foreach( Task* Block, Object.needs() )
	if( Block )
	  new Carrier( *Block, this, Blocker );
    }
    return SubItems;
  } // populate()
  TasksTreeModel::Carrier::Pointer TasksTreeModel::Carrier::subitem( int Index ) const
  {
    Pointer Result = 0;
    if( Index >= 0 && Index < SubItems.size() )
      Result = SubItems[ Index ];
    return Result;
  } // subitem( int ) const
  bool TasksTreeModel::Carrier::move_subitem( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < SubItems.size() && To >= 0 && To < SubItems.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap items" << Index << "and" << Index+Inc;
	  SubItems.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_subitem( int, int )

  TasksTreeModel::TasksTreeModel( const QString& FileName0, QObject* Parent ) : QAbstractItemModel( Parent ), FileName( FileName0 ), Modified( false ), Good( true )
  {
    if( !FileName.isEmpty() )
    {
      QFile File( FileName );
      if( File.open( QIODevice::ReadOnly ) )
      {
	qDebug() << "Loading tasks";
	Task::Map TasksMap;
	QXmlStreamReader Stream( &File );
	if( Timeshop::Persistent::Loader::next_subelement( Stream ) && Stream.name() == "plansplant" )
	{
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	  {
	    if( Stream.isStartElement() )
	    {
	      if( Stream.name() == "task" )
	      {
		Task* NewTask = new Task;
		Roots.push_back( NewTask );
		NewTask->load( Stream );
		NewTask->add_to_map( TasksMap );
		qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
	      }
	      else if( Stream.name() == "dependencies" )
	      {	
		qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
		Task::ID ForID = 0;
		if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "task_id", ForID ) )
		{
		  if( Task* For = TasksMap[ ForID ] )
		    while( Timeshop::Persistent::Loader::next_subelement( Stream ) && Stream.name() == "blocker" )
		    {
		      qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
		      Task::ID UponID = Stream.readElementText().toInt();
		      if( Task* Upon = TasksMap[ UponID ] )
			For->add_dependency( *Upon );
		      else qDebug() << "Task with ID" << UponID << "(UponID) not found.";
		    }
		  else qDebug() << "Task with ID" << ForID << "(ForID) not found.";
		}
		else qDebug() << "Found dependencies tag without task id";
	      }
	      else Timeshop::Persistent::Loader::skip( Stream );
	    }
	    else Timeshop::Persistent::Loader::skip( Stream );
	  }
	  qDebug() << "after while" << Stream.tokenType() << Stream.name() << Stream.tokenString() << Stream.errorString();
	  Good = ( File.error() == QFile::NoError ) && !Stream.hasError();
	}
	else
	{
	  qDebug() << "No plansplant tag in the file.";
	  Good = false;
	}
      }
      else
	Good = false;
#if 0
      else
      {
	// Prepare test model
	Task* NewTask = new Task( "1" );
	Roots.push_back( NewTask );
	Task* SubTask = new Task( "1.1", NewTask );
	new Task( "1.1.1", SubTask );
	new Task( "1.1.2", SubTask );
	SubTask = new Task( "1.2", NewTask );
	NewTask = new Task( "2" );
	Roots.push_back( NewTask );
	SubTask->add_dependency( *NewTask );
      }
#endif
      if( !Good ) qDebug() << "Can't open file for reading.";
    }
  } // TasksTreeModel( QObject* )
  TasksTreeModel::~TasksTreeModel()
  {
    while( !Carriers.empty() )
    {
      delete Carriers.back();
      Carriers.pop_back();
    }
    while( !Roots.empty() )
    {
      delete Roots.back();
      Roots.pop_back();
    }
  } // ~TasksTreeModel()
  bool TasksTreeModel::save( const QString& NewFileName )
  {
    bool Result = false;
    if( !NewFileName.isEmpty() ) FileName = NewFileName;
    if( !FileName.isEmpty() )
    {
      qDebug() << "Save tasks";
      QFile File( FileName );
      if( File.open( QIODevice::WriteOnly ) )
      {
	QXmlStreamWriter Stream( &File );
	Stream.setAutoFormatting( true );
	Stream.setAutoFormattingIndent( 2 );
	Stream.writeStartDocument();
	Stream.writeStartElement( "plansplant" );
	foreach( Task* CurrTask,  Roots )
	  if( CurrTask )
	    CurrTask->write( Stream );
	foreach( Task* CurrTask,  Roots )
	  if( CurrTask )
	    CurrTask->write_dependencies( Stream );
	Stream.writeEndElement();
	Stream.writeEndDocument();
      }
      Result = ( File.error() == QFile::NoError );
    }
    if( Result ) modified( false );
    return Result;
  } // save( const QString )
  int TasksTreeModel::columnCount( const QModelIndex& /*Parent*/ ) const { return TotalCols; }
  int TasksTreeModel::rowCount( const QModelIndex& Parent ) const
  {
    int Result = 0;
    if( Parent.isValid() )
    {
      if( Carrier* CurrItem = carrier_from_index( Parent ) ) //! \todo else report error
	Result = CurrItem->subitems().size();
    }
    else
      Result = Carriers.size();
    return Result;
  } // rowCount( const QModelIndex& ) const
  bool TasksTreeModel::hasChildren( const QModelIndex& Parent ) const
  {
    bool Result = false;
    if( Parent.isValid() )
    {
      if( Task* T = task_from_index( Parent ) )
	Result = T->subtasks().size() + T->needs().size() > 0;
    }
    else
      Result = Roots.size() > 0;
    return Result;
  } // hasChildren( const QModelIndex& ) const
  bool TasksTreeModel::canFetchMore( const QModelIndex& Parent ) const
  {
    bool Result = false;
    if( Parent.isValid() )
    {
      if( Carrier* Car = carrier_from_index( Parent ) )
	Result = !Car->populated();
    }
    else
      Result = Carriers.size() < Roots.size();
    return Result;
  } // canFetchMore( const QModelIndex& ) const
  void TasksTreeModel::fetchMore( const QModelIndex& Parent )
  {
    if( Carrier* Car = carrier_from_index( Parent ) )
    {
      beginInsertRows( Parent, 0, Car->task().subtasks().size() + Car->task().needs().size() - 1 );
      Car->populate();
      endInsertRows();
    }
    else
    {
      beginInsertRows( Parent, 0, Roots.size() );
      foreach( Task* Root, Roots )
	if( Root )
	  Carriers.push_back( new Carrier( *Root ) );
      endInsertRows();
    }
  } // fetchMore( const QModelIndex& )
  QModelIndex TasksTreeModel::index( int Row, int Column, const QModelIndex& Parent ) const
  {
    QModelIndex Result;
    if( Parent.isValid() )
    {
      if( Carrier* ParItem = carrier_from_index( Parent ) ) //! \todo else report error
      {
	ParItem->populate();
	if( Carrier* Item = ParItem->subitem( Row ) )
	  Result = createIndex( Row, Column, Item );
      }
    }
    else
    {
      if( Row >= 0 && Row < Carriers.size() )
	Result = createIndex( Row, Column, Carriers[ Row ] );
    }
    return Result;
  } // index( int Row, int Column, const QModelIndex& Parent ) const
  QModelIndex TasksTreeModel::index( Carrier& ForItem ) const
  {
    QModelIndex Result;
    if( Carrier* Parent = ForItem.parent() )
      Result = createIndex( Parent->subitems().indexOf( &ForItem ), 0, &ForItem );
    else
      Result = createIndex( Carriers.indexOf( &ForItem ), 0, &ForItem );
    return Result;
  } // index( Carrier& ) 
 QModelIndex TasksTreeModel::parent( const QModelIndex& Index ) const
  {
    QModelIndex Result;
    if( Index.isValid() )
      if( Carrier* Item = carrier_from_index( Index ) ) //! \todo else report error
	if( Carrier* Parent = Item->parent() )
	  Result = index( *Parent );
    return Result;
  } // parent( const QModelIndex& ) const
  QVariant TasksTreeModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( Index.isValid() )
      if( Carrier* CurrItem = carrier_from_index( Index ) ) //! \todo else report error
      {
	Task& CurrTask = CurrItem->task();
	if( Role == Qt::DisplayRole )
	{
	  switch( Index.column() )
	  {
	  case NameCol: Result = CurrTask.name(); break;
	  case CompletedCol: Result = QString::number( CurrTask.completed() * 100 ) + '%'; break;
	  case PlanStartCol:
	    if( CurrTask.plan_start().isValid() )
	      Result = CurrTask.plan_start();
	    break;
	  case PlanFinishCol:
	    if( CurrTask.plan_finish().isValid() )
	      Result = CurrTask.plan_finish();
	    break;
	  case EstimationCol:
	    if( CurrTask.estimation() > 0 )
	    {
	      QString EstStr = QString::number( CurrTask.estimation() ) + ' ';
	      switch( CurrTask.estimation_units() )
	      {
	      case Task::Seconds: EstStr += tr( "s" ); break;
	      case Task::Minutes: EstStr += tr( "m" ); break;
	      case Task::Hours: EstStr += tr( "h" ); break;
	      case Task::WorkDays: EstStr += tr( "wd" ); break;
	      case Task::Days: EstStr += tr( "d" ); break;
	      case Task::WorkWeeks: EstStr += tr( "ww" ); break;
	      case Task::Weeks: EstStr += tr( "w" ); break;
	      case Task::Months: EstStr += tr( "M" ); break;
	      case Task::Quarters: EstStr += tr( "Q" ); break;
	      case Task::Years: EstStr += tr( "Y" ); break;
	      default: break;
	      }
	      Result = EstStr;
	    }
	    break;
	  default: break;
	  }
	}
	else if( Role == Qt::TextAlignmentRole )
	{ if( Index.column() == CompletedCol ) Result = int( Qt::AlignRight | Qt::AlignVCenter ); }
	else if( Role == Qt::DecorationRole )
	{ if( Index.column() == NameCol ) Result = QIcon( is_blocker( Index ) ? ":/images/dependency.svg" : ":/images/task.svg" ); } //! \todo Select from the list of preloaded icons
	else if( Role == Qt::BackgroundRole )
	{ //! \todo This don't change the background color on Maemo, but it seems that it speeds up scrolling.
	  if( !MainWindow::desktop_view() )
	  {
	    if( CurrItem->relation() == Carrier::Blocker )
	    {
	      if( CurrTask.completed() < 1 )
		Result = QBrush( QColor( 160, 64, 0 ) );
	      else
		Result = QBrush( QColor( 0, 128, 0 ) );
	    }
	    else
	      Result = QBrush( QColor( 0, 0, 0 ) );
	  }
	}
	else if( Role == Qt::ForegroundRole ) //! \todo On Maemo this cause very high load when scrolling.
	{
	  if( MainWindow::desktop_view() )
	    if( CurrItem->relation() == Carrier::Blocker )
	    {
	      if( CurrTask.completed() < 1 )
		Result = QBrush( QColor( 160, 120, 0 ) );
	      else
		Result = QBrush( QColor( 0, 200, 0 ) );
	    }
	}
	else if( Role == Qt::CheckStateRole )
	{
	  if( Index.column() == CompletedMarkCol )
	  {
	    if( CurrTask.completed() >= 1 )
	      Result = Qt::Checked;
	    else
	    { if( !CurrTask.blocked() ) Result = Qt::Unchecked; }
	  }
	}
#ifdef Q_WS_MAEMO_5
	else if( Role == Qt::SizeHintRole )
	{ if( !MainWindow::desktop_view() ) Result = QSize( 300, 42 ); }
#endif
      }
    return Result;
  } // data( const QModelIndex&, int ) const
  QVariant TasksTreeModel::headerData( int Section, Qt::Orientation Orient, int Role ) const
  {
    QVariant Result;
    if( Orient == Qt::Horizontal && Role == Qt::DisplayRole )
    {
      switch( Section )
      {
      case NameCol: Result = tr( "Name" ); break;
      case CompletedCol: Result = tr( "Completed" ); break;
      case PlanStartCol: Result = tr( "Start" ); break;
      case PlanFinishCol: Result = tr( "Finish" ); break;
      case EstimationCol: Result = tr( "Estimation" ); break;
      default: break;
      }
    }
    return( Result );
  } // headerData( int, Qt::Orientation, int )
  void TasksTreeModel::add_root_task( Task& NewRoot )
  {
    if( !Roots.contains( &NewRoot ) )
    {
      int NextRow = rowCount( QModelIndex() );
      beginInsertRows( QModelIndex(), NextRow, NextRow );
      Roots.push_back( &NewRoot );
      modified();
      Carriers.push_back( new Carrier( NewRoot ) );
      endInsertRows();
    }
  } // add_root_task( Task& )
  void TasksTreeModel::remove_from_root_tasks( Task& OldRoot )
  { //! \todo Maybe check that OldRoot has supertask.
    for( int Index = Roots.indexOf( &OldRoot ); Index >= 0; Index = Roots.indexOf( &OldRoot, Index ) )
    {
      beginRemoveRows( QModelIndex(), Index, Index );
      Carriers.removeAt( Index );
      Roots.removeAt( Index );
      modified();
      endRemoveRows();
    }
  } // remove_from_root_tasks( Task& )
  class TreeIterator
  {
  public:
    TreeIterator( TasksTreeModel& Model0 ) : Model( Model0 ), Current( Model0.index( 0, 0 ) ) {}
    TreeIterator( TasksTreeModel& Model0, const QModelIndex& Start ) : Model( Model0 ), Current( Start ) {}
    TreeIterator& operator++();
    TreeIterator& operator--();
    const QModelIndex& index() const { return Current; }
    const QModelIndex& operator*() const { return Current; }
    operator bool() const { return Current.isValid(); }
  protected:
    TasksTreeModel& Model;
    QModelIndex Current;
  }; // TreeIterator
  TreeIterator& TreeIterator::operator++()
  {
    QModelIndex Next = Current.child( 0, 0 ); // Search the model in-depth.
    if( !Next.isValid() )
    {
      Next = Current.sibling( Current.row()+1, 0 );
      while( !Next.isValid() && Current.isValid() ) // Try to find next sibling on any level.
      {
	Current = Current.parent();
	if( Current.isValid() )
	  Next = Current.sibling( Current.row()+1, 0 );
      }
    }
    Current = Next;
    return *this;
  } // operator++()
  TreeIterator& TreeIterator::operator--()
  {
    QModelIndex Prev;
    if( Current.row() > 0 ) Prev = Current.sibling( Current.row()-1, 0 ); // Try to step backward.
    if( !Prev.isValid() ) Prev = Current.parent(); // Try to step up.
    Current = Prev;
    return *this;
  } // operator--()
  bool TasksTreeModel::move_task( const QModelIndex& Index, int From, int To )
  {
    bool Result = false;
    qDebug() << "Move task from" << From << "to" << To;
    int Rows = rowCount( Index );
    if( From >= 0 && From < Rows && To >= 0 && To < Rows && From != To )
    {
      int FixedTo = (To > From) ? To+1 : To;
      if( beginMoveRows( Index, From, From, Index, FixedTo ) )
      {
	if( Carrier* Car = carrier_from_index( Index ) )
	{
	  Task& T = Car->task();
	  int SubTasksNum = T.subtasks().size();
	  
	  if( From >= 0 && From < SubTasksNum && To >= 0 && To < SubTasksNum )
	    Result = T.move_subtask( From, To );
	  else
	    Result = T.move_dependency( From-SubTasksNum, To-SubTasksNum );
	  modified();
	  Car->move_subitem( From, To );
	  endMoveRows();
	  qDebug() << "Rows moved";
	  for( TreeIterator It( *this ); It.index().isValid(); ++It )
	    if( Carrier* TestCar = carrier_from_index( It.index() ) )
	      if( TestCar != Car && &TestCar->task() == &T && beginMoveRows( It.index(), From, From, It.index(), FixedTo ) )
	      {
		TestCar->move_subitem( From, To );
		endMoveRows();
	      }
	}
	else if( !Index.isValid() ) // It's a root item
	{
	  Task* TmpTask = Roots[ From ];
	  Carrier* TmpCar = Carriers[ From ];
	  int Step = To > From ? 1 : -1;
	  for( int I = From; I != To; I += Step )
	  {
	    Roots[ I ] = Roots[ I+Step ];
	    Carriers[ I ] = Carriers[ I+Step ];
	  }
	  Roots[ To ] = TmpTask;
	  Carriers[ To ] = TmpCar;
	  endMoveRows();
	  Result = true;
	  modified();
	}
      }
      else
	qDebug() << "Can\'t move" << From << '-' << From << "to" << To;
    }
    return Result;
  } // move_task( const QModelIndex&, int, int )
  void TasksTreeModel::add_task( Task& NewTask )
  {
    modified();
    Task* SuperTask = NewTask.supertask();
    if( !SuperTask )
      add_root_task( NewTask );
    else
      for( TreeIterator It( *this ); It.index().isValid(); ++It )
	if( Carrier* Car = carrier_from_index( It.index() ) )
	  if( &Car->task() == SuperTask )
	  {
	    int Row = SuperTask->subtasks().size()-1; // We've updated it before.
	    beginInsertRows( It.index(), Row, Row );
	    new Carrier( NewTask, Car );
	    endInsertRows();
	  }
  } // add_task( Task& )
  void TasksTreeModel::delete_task( Task& OldTask )
  {
    Task* SuperTask = OldTask.supertask();
    if( !SuperTask )
      remove_from_root_tasks( OldTask );
    for( TreeIterator It( *this ); It; ++It )
      if( Carrier* Car = carrier_from_index( *It ) )
	if( &Car->task() == &OldTask )
	{
	  int Row = It.index().row();
	  beginRemoveRows( parent( *It ), Row, Row );
	  --It;
	  delete Car;
	  endRemoveRows();
	}
    delete &OldTask;
    modified();
  } // delete_task( Task& )
  bool TasksTreeModel::change_parent( Task* ForTask, Task* NewParent )
  {
    bool Result = false;
    if( ForTask )
    {
      Task* OldParent = ForTask->supertask();
      if( OldParent != NewParent )
      {
	if( NewParent && NewParent->check_loop( *ForTask ) )
	  QMessageBox::warning( 0, tr( "PlansPlant" ),
				tr( "Can't set \"" ) + NewParent->name() + tr( "\"\nas a supertask for \"" ) + ForTask->name() + tr( "\":\nit will create a loop." ) );
	else
	{
	  ForTask->supertask( NewParent );
	  modified();
	  if( !OldParent ) remove_from_root_tasks( *ForTask );
	  else if( !NewParent ) add_root_task( *ForTask );
	  for( TreeIterator It( *this ); It.index().isValid(); ++It )
	    if( Carrier* Car = carrier_from_index( It.index() ) )
	    {
	      if( &Car->task() == NewParent && NewParent )
	      {
		int Row = NewParent->subtasks().size()-1; // We've updated it before.
		beginInsertRows( It.index(), Row, Row );
		new Carrier( *ForTask, Car );
		endInsertRows();
	      }
	      else if( &Car->task() == ForTask && Car->relation() == Carrier::Subtask && Car->parent() && &Car->parent()->task() == OldParent )
	      {
		beginRemoveRows( parent( It.index() ), It.index().row(), It.index().row() );
		--It;
		delete Car;
		endRemoveRows();
	      }
	    }
	  Result = true;
	}
      }
    }
    return Result;
  } // change_parent( Task*, Task* )
  bool TasksTreeModel::is_subtask( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( Carrier* Item = carrier_from_index( Index ) )
      Result = ( Item->relation() == Carrier::Subtask );
    return Result;
  } // is_subtask( const QModelIndex& ) const
  void TasksTreeModel::add_blocker( Task& To, Task& Block )
  {
    if( !To.needs().contains( &Block ) )
    {
      if( To.check_loop( Block ) )
	QMessageBox::warning( 0, tr( "PlansPlant" ), tr( "Can't add \"" ) + Block.name() + tr( "\"\nas a dependency for \"" ) + To.name() + tr( "\":\nit will create a loop." ) );
      else
      {
	To.add_dependency( Block );
	modified();
	for( TreeIterator It( *this ); It.index().isValid(); ++It )
	  if( Carrier* Car = carrier_from_index( It.index() ) )
	  {
	    if( &Car->task() == &To )
	    {
	      qDebug() << "Add blocker" << Block.name() << "to" << To.name();
	      int Row = Car->subitems().size();
	      beginInsertRows( It.index(), Row, Row );
	      new Carrier( Block, Car, Carrier::Blocker );
	      endInsertRows();
	    }
	  }
      }
    }
  } // add_blocker( Task&, Task& )
  bool TasksTreeModel::is_blocker( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( Carrier* Item = carrier_from_index( Index ) )
      Result = ( Item->relation() == Carrier::Blocker );
    return Result;
  } // is_blocker( const QModelIndex& ) const
  void TasksTreeModel::remove_blocker( const QModelIndex& Index )
  {
    if( Carrier* Item = carrier_from_index( Index ) )
      if( Carrier* Parent = Item->parent() )
	if( Item->relation() == Carrier::Blocker )
	  remove_blocker( Parent->task(), Item->task() );
  } // remove_blocker( const QModelIndex& )
  void TasksTreeModel::remove_blocker( Task& From, Task& Block )
  {
    if( From.needs().contains( &Block ) )
    {
      for( TreeIterator It( *this ); It.index().isValid(); ++It )
	if( Carrier* Car = carrier_from_index( It.index() ) )
	  if( &Car->task() == &Block && Car->relation() == Carrier::Blocker && Car->parent() && &Car->parent()->task() == &From )
	  {
	    int Row = It.index().row();
	    beginRemoveRows( parent( It.index() ), Row, Row );
	    --It;
	    delete Car;
	    endRemoveRows();
	  }
      From.remove_dependency( Block );
      modified();
    }
    else
      qDebug() << "Blocker not found in the task.";
  } // remove_blocker( Task&, Task& );
  bool TasksTreeModel::change_blockers( Task& Object, const Task::List& NewBlockers, Task* /*SuperTask*/ )
  {
    bool Result = false;
    const int Subtasks = Object.subtasks().size();
    const int OldRows = Subtasks + Object.needs().size();
    const int NewRows = Subtasks + NewBlockers.size();
    const int LessRows = Object.needs().size() < NewBlockers.size() ? Object.needs().size() : NewBlockers.size();
    int SameRows = 0;
    for( int I = 0; I < LessRows && Object.needs( I ) == NewBlockers[ I ]; I++ )
      SameRows++;
    int RowsStart = Subtasks+SameRows;
    if( RowsStart < NewRows )
    {
      //! \todo Maybe check for loops
      Result = true;
      while( Object.needs().size() > SameRows )
	Object.remove_dependency( *Object.needs().back() );
      for( int I = SameRows; I < NewBlockers.size(); I++ )
	Object.add_dependency( *NewBlockers[ I ] );
      //! \todo Don't delete existing branches but move them.
      for( TreeIterator It( *this ); It.index().isValid(); ++It )
	if( Carrier* Car = carrier_from_index( It.index() ) )
	  if( &Car->task() == &Object && Car->populated() )
	  {
	    if( OldRows > RowsStart )
	    {
	      qDebug() << "Delete" << OldRows-RowsStart << "rows.";
	      beginRemoveRows( It.index(), RowsStart, OldRows-1 );
	      while( Car->subitems().size() > RowsStart )
		delete Car->subitems().back();
	      endRemoveRows();
	    }
	    if( NewRows > RowsStart )
	    {
	      qDebug() << "Add" << NewRows-RowsStart << "rows.";
	      beginInsertRows( It.index(), RowsStart, NewRows-1 );
	      for( int I = SameRows; I < Object.needs().size(); I++ )
		new Carrier( *Object.needs( I ), Car, Carrier::Blocker );
	      endInsertRows();
	    }
	  }
      modified();
    } // Has something to change
    return Result;
  } // change_blockers( Task&, const Task::List&, Task* )

  TaskSelectDialog::TaskSelectDialog( TasksTreeModel& Model0, QWidget* Parent, Task* CurrentTask0, const QString& Title )
    : QDialog( Parent ), CurrentTask( CurrentTask0 ), TasksTree( 0 ), Buttons( 0 )
  {
    setWindowTitle( Title );
    TasksTree = new TasksTreeWidget( this, &Model0 );
    TasksTree->setMinimumHeight( 300 );
    QVBoxLayout* Layout = new QVBoxLayout( this );
    Layout->addWidget( TasksTree );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() ) Layout->setContentsMargins( 0, 0, 0, 0 );
    foreach( QAction* Act, TasksTree->actions() )
      Act->setShortcuts( QList<QKeySequence>() );
#endif // Q_WS_MAEMO_5
    Buttons = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( CurrentTask );
    connect( Buttons, SIGNAL( accepted() ), SLOT( accept() ) );
    connect( Buttons, SIGNAL( rejected() ), SLOT( reject() ) );
    Layout->addWidget( Buttons );
    setLayout( Layout );
    connect( TasksTree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ), SLOT( selection_changed( const QItemSelection& ) ) );
    TasksTree->select_task( CurrentTask );
  } // TaskSelectDialog( TasksTreeModel&, QWidget*, Task* )
  void TaskSelectDialog::accept()
  {
    CurrentTask = TasksTree->selected_task();
    if( CurrentTask ) QDialog::accept();
    else QMessageBox::warning( this, tr( "PlansPlant" ), tr( "Select a task before." ) );
  } // accept()
  void TaskSelectDialog::selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front(); // We've got indexes for every column. Take only the first one.
    selection_changed( NewIndex );
  } // selection_changed( const QItemSelection& )
  void TaskSelectDialog::selection_changed( const QModelIndex& NewIndex )
  {
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( TasksTree->tasks() && TasksTreeModel::task_from_index( NewIndex ) );
  } // selection_changed( const QModelIndex& )

  ParentSelectDialog::ParentSelectDialog( TasksTreeModel& Model0, QWidget* Parent, const Task::List& Blockers0, Task* ForTask0, Task* SuperTask0 )
    : TaskSelectDialog( Model0, Parent, SuperTask0, tr( "Select new supertask for " ) + ( ForTask0 ? ( '\"' + ForTask0->name() + '\"' ) : tr( "new task" ) ) ),
      ForTask( ForTask0 ), Blockers( Blockers0 )
  {
    selection_changed( TasksTree->selected_index() );
  } // ParentSelectDialog( TasksTreeModel&, Task&, QWidget*, Task* )
  void ParentSelectDialog::selection_changed( const QModelIndex& NewIndex )
  {
    bool Enable = false;
    if( Task* NewParent = TasksTreeModel::task_from_index( NewIndex ) )
      Enable = !NewParent->check_loop( Blockers, ForTask );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( Enable );
  } // selection_changed( const QModelIndex& )
  BlockerSelectDialog::BlockerSelectDialog( TasksTreeModel& Model0, QWidget* Parent, Task* ForTask0, Task* Blocker0 )
    : TaskSelectDialog( Model0, Parent, Blocker0, tr( "Select new dependency for " ) + ( ForTask0 ? ( '\"' + ForTask0->name() + '\"' ) : tr( "new task" ) ) + '\"' ),
      ForTask( ForTask0 )
  {
    selection_changed( TasksTree->selected_index() );
  } // BlockerSelectDialog( TasksTreeModel&, Task&, QWidget*, Task* )
  void BlockerSelectDialog::selection_changed( const QModelIndex& NewIndex )
  {
    bool Enable = false;
    if( Task* NewBlocker = TasksTreeModel::task_from_index( NewIndex ) )
      Enable = !ForTask || !ForTask->check_loop( *NewBlocker );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( Enable );
  } // selection_changed( const QModelIndex& )

  class BlockersListModel : public QAbstractItemModel
  {
  public:
    BlockersListModel( QObject* Parent = 0, const Task::List& Blockers0 = Task::List() ) : QAbstractItemModel( Parent ), Blockers( Blockers0 ) {}
    int columnCount( const QModelIndex& Parent = QModelIndex() ) const;
    int rowCount( const QModelIndex& Parent = QModelIndex() ) const;
    QModelIndex index( int Row, int Column, const QModelIndex& Parent = QModelIndex() ) const;
    QModelIndex parent( const QModelIndex& Index ) const;
    QVariant data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const;
    QVariant headerData( int Section, Qt::Orientation Orient, int Role = Qt::DisplayRole ) const;
    const Task::List& blockers() const { return Blockers; }
    void add_blocker( Task& NewBlocker );
    void remove_blocker( Task& OldBlocker );
    void move_up( const QModelIndex& Index );
    void move_down( const QModelIndex& Index );
  protected:
    Task::List Blockers;
  }; // BlockersListModel
  int BlockersListModel::columnCount( const QModelIndex& /*Parent*/ ) const { return TasksTreeModel::TotalCols; }
  int BlockersListModel::rowCount( const QModelIndex& Parent ) const { return Parent.isValid() ? 0 : Blockers.size(); }
  QModelIndex BlockersListModel::index( int Row, int Column, const QModelIndex& Parent ) const
  {
    QModelIndex Result;
    if( !Parent.isValid() && Row >= 0 && Row < Blockers.size() ) 
      Result = createIndex( Row, Column, Blockers[ Row ] );
    return Result;
  } // index( int Row, int Column, const QModelIndex& Parent ) const
  QModelIndex BlockersListModel::parent( const QModelIndex& /*Index*/ ) const { return QModelIndex(); }
  QVariant BlockersListModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    const int Row = Index.row(); 
    if( Index.isValid() && Row >= 0 && Row < Blockers.size() ) 
    {
      Task* Blocker = Blockers[ Row ];
      if( Role == Qt::DisplayRole )
	switch( Index.column() )
	{
	case TasksTreeModel::NameCol: Result = Blocker->name(); break;
	case TasksTreeModel::CompletedCol: Result = QString::number( Blocker->completed() * 100 ) + '%'; break;
	case TasksTreeModel::PlanStartCol:
	  if( Blocker->plan_start().isValid() )
	    Result = Blocker->plan_start();
	  break;
	case TasksTreeModel::PlanFinishCol:
	  if( Blocker->plan_finish().isValid() )
	    Result = Blocker->plan_finish();
	  break;
	case TasksTreeModel::EstimationCol:
	  if( Blocker->estimation() > 0 )
	    Result = QString::number( Blocker->estimation() ) + ' ' + Task::units_short_name( Blocker->estimation_units() );
	  break;
	default: break;
	}
      else if( Role == Qt::DecorationRole )
      {
	if( Index.column() == TasksTreeModel::NameCol )
	  Result = QIcon( ":/images/dependency.svg" );
      }
      else if( Role == Qt::CheckStateRole )
      {
	if( Index.column() == TasksTreeModel::CompletedMarkCol )
	{
	  if( Blocker->completed() >= 1 )
	    Result = Qt::Checked;
	  else
	  { if( !Blocker->blocked() ) Result = Qt::Unchecked; }
	}
      }
#ifdef Q_WS_MAEMO_5
      else if( Role == Qt::SizeHintRole )
      { if( !MainWindow::desktop_view() ) Result = QSize( 300, 42 ); }
#endif
    }
    return Result;
  } // data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const
  QVariant BlockersListModel::headerData( int Section, Qt::Orientation Orient, int Role ) const
  {
    QVariant Result;
    if( Role == Qt::DisplayRole && Orient == Qt::Horizontal )
      switch( Section )
      {
      case TasksTreeModel::NameCol: Result = TasksTreeModel::tr( "Name" ); break;
      case TasksTreeModel::CompletedMarkCol: break;
      case TasksTreeModel::CompletedCol: Result = TasksTreeModel::tr( "Completed" ); break;
      case TasksTreeModel::PlanStartCol: Result = TasksTreeModel::tr( "Start" ); break;
      case TasksTreeModel::PlanFinishCol: Result = TasksTreeModel::tr( "Finish" ); break;
      case TasksTreeModel::EstimationCol: Result = TasksTreeModel::tr( "Estimation" ); break;
      default: Result = QString::number( Section ); break;
      }
    return Result;
  } // headerData( int, Qt::Orientation, int ) const
  void BlockersListModel::add_blocker( Task& NewBlocker )
  {
    if( !Blockers.contains( &NewBlocker ) )
    {
      beginInsertRows( QModelIndex(), Blockers.size(), Blockers.size() );
      Blockers.push_back( &NewBlocker );
      endInsertRows();
    }
  } // add_blocker( Task& )
  void BlockersListModel::remove_blocker( Task& OldBlocker )
  {
    int Index = Blockers.indexOf( &OldBlocker );
    if( Index >= 0 )
    {
      beginRemoveRows( QModelIndex(), Index, Index );
      Blockers.removeAt( Index );
      endRemoveRows();
    }
  } // remove_blocker( Task& )
  void BlockersListModel::move_up( const QModelIndex& Index )
  {
    if( Index.isValid() )
    {
      int NewRow = Index.row()-1;
      if( NewRow >= 0 )
	if( beginMoveRows( parent( Index ), Index.row(), Index.row(), parent( Index ), NewRow ) )
	{
	  Blockers.swap( Index.row(), NewRow );
	  endMoveRows();
	}
    }
  } // move_up( const QModelIndex& )
  void BlockersListModel::move_down( const QModelIndex& Index )
  {
    if( Index.isValid() )
    {
      int NewRow = Index.row()+1;
      if( NewRow < Blockers.size() )
	if( beginMoveRows( parent( Index ), Index.row(), Index.row(), parent( Index ), NewRow+1 ) )
	{
	  Blockers.swap( Index.row(), NewRow );
	  endMoveRows();
	}
    }
  } // move_down( const QModelIndex& )
#ifdef Q_WS_MAEMO_5
  void disable_kinetic_scroller( QAbstractScrollArea* Widget )
  {
    if( Widget )
      if( QAbstractKineticScroller *Scroller = Widget->property( "kineticScroller" ).value<QAbstractKineticScroller*>() )
      {
	Scroller->setEnabled( false );
	if( !MainWindow::desktop_view() ) //! \todo Find a way to determine that we're using Hildon style.
	  if( QStyle* GoodStyle = QStyleFactory::create( "GTK+" ) )
	  {
	    Widget->horizontalScrollBar()->setStyle( GoodStyle );
	    Widget->horizontalScrollBar()->setContextMenuPolicy( Qt::PreventContextMenu );
	    Widget->verticalScrollBar()->setStyle( GoodStyle );
	    Widget->verticalScrollBar()->setContextMenuPolicy( Qt::PreventContextMenu );
	  }
      }
  } // disable_kinetic_scroller( QAbstractScrollArea* )
#endif // Q_WS_MAEMO_5
  BlockersEditor::BlockersEditor( TasksTreeModel& Model0, Task* Object0, QWidget* Parent )
    : QWidget( Parent ), Model( Model0 ), Blockers( 0 ), Object( Object0 ), SuperTask( 0 )
  {
    UI.setupUi( this );
    UI.BlockerAddButton->setIcon( QIcon::fromTheme( "add", QIcon( ":/images/add.svg" ) ) );
    UI.BlockerRemoveButton->setIcon( QIcon::fromTheme( "remove", QIcon( ":/images/remove.svg" ) ) );
    UI.BlockerMoveUpButton->setIcon( QIcon::fromTheme( "go-up", QIcon( ":/images/go-up.svg" ) ) );
    UI.BlockerMoveDownButton->setIcon( QIcon::fromTheme( "go-down", QIcon( ":/images/go-down.svg" ) ) );
    UI.BlockersTreeSwitchButton->setIcon( QIcon( ":/images/show-tree.svg" ) );
    UI.BlockersPoolTree->setVisible( UI.BlockersTreeSwitchButton->isChecked() );
    foreach( QAction* Act, UI.BlockersPoolTree->actions() )
#ifndef Q_WS_MAEMO_5
      Act->setShortcutContext( Qt::WidgetShortcut );
#else // Q_WS_MAEMO_5
      Act->setShortcuts( QList<QKeySequence>() );
    if( MainWindow::desktop_view() ) disable_kinetic_scroller( UI.BlockersList );
    else UI.BlockersSplitter->setMinimumHeight( 220 );
#endif // Q_WS_MAEMO_5
    if( Object )
    {
      SuperTask = Object->supertask();
      Blockers = new BlockersListModel( this, Object->needs() );
    }
    else
      Blockers = new BlockersListModel( this );
    UI.BlockersList->setModel( Blockers );
    UI.BlockersList->setColumnWidth( TasksTreeModel::NameCol, 400 );
    if( MainWindow::desktop_view() )
    {
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedMarkCol, 24 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedCol, 43 );
    }
    else
    {
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedMarkCol, 55 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedCol, 70 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::PlanStartCol, 128 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::PlanFinishCol, 128 );
    }
    UI.BlockersPoolTree->tasks( &Model );
    connect( UI.BlockerAddButton, SIGNAL( clicked() ), SLOT( add_blocker() ) );
    connect( UI.BlockerRemoveButton, SIGNAL( clicked() ), SLOT( remove_blocker() ) );
    connect( UI.BlockerMoveUpButton, SIGNAL( clicked() ), SLOT( move_blocker_up() ) );
    connect( UI.BlockerMoveDownButton, SIGNAL( clicked() ), SLOT( move_blocker_down() ) );
    connect( UI.BlockersPoolTree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
	     SLOT( blockers_pool_selection_changed( const QItemSelection& ) ) );
    connect( UI.BlockersList->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
	     SLOT( blockers_list_selection_changed( const QItemSelection& ) ) );
    connect( UI.BlockersTreeSwitchButton, SIGNAL( toggled( bool ) ), SLOT( blockers_pool_switched( bool ) ) );
  } // BlockersEditor( TasksTreeModel&, Task*, QWidget* )
  const Task::List& BlockersEditor::blockers() const { return Blockers->blockers(); }
  QModelIndex BlockersEditor::selected_blocker() const
  {
    QModelIndex Result;
    if( UI.BlockersList )
      if( QItemSelectionModel* SelectionModel = UI.BlockersList->selectionModel() )
      {
	QModelIndexList Indexes = SelectionModel->selectedIndexes();
	if( !Indexes.isEmpty() )
	  Result = Indexes.front();
      }
    return Result;
  } // selected_blocker() const
  void BlockersEditor::supertask( Task* NewSuperTask )
  {
    if( SuperTask != NewSuperTask )
    {
      SuperTask = NewSuperTask;
      blockers_pool_switched( UI.BlockersTreeSwitchButton->isChecked() );
    }
  } // supertask( Task* )
  void BlockersEditor::add_blocker()
  {
    if( Blockers )
      if( Task* Blocker = UI.BlockersPoolTree->selected_task() )
	if( can_add_blocker( *Blocker ) )
	{
	  Blockers->add_blocker( *Blocker );
	  blockers_list_selection_changed( selected_blocker() );
	  blockers_pool_switched( UI.BlockersTreeSwitchButton->isChecked() ); // To update buttons
	}
  } // add_blocker()
  void BlockersEditor::remove_blocker()
  {
    if( Blockers )
      if( Task* Blocker = TasksTreeModel::task_from_index( selected_blocker() ) ) //!< \todo Don't take it from the other model.
      {
	Blockers->remove_blocker( *Blocker );
	blockers_list_selection_changed( selected_blocker() );
	blockers_pool_switched( UI.BlockersTreeSwitchButton->isChecked() ); // To update buttons
      }
  } // remove_blocker()
  void BlockersEditor::move_blocker_up()
  {
    if( Blockers )
    {
      Blockers->move_up( selected_blocker() );
      blockers_list_selection_changed( selected_blocker() );
    }
  } // move_blocker_up()
  void BlockersEditor::move_blocker_down()
  {
    if( Blockers )
    {
      Blockers->move_down( selected_blocker() );
      blockers_list_selection_changed( selected_blocker() );
    }
  } // move_blocker_down()
  bool BlockersEditor::can_add_blocker( Task& NewBlocker )
  {
    bool Result = false;
    if( &NewBlocker != Object && !( Blockers && Blockers->blockers().contains( &NewBlocker ) )
	&& !( SuperTask && SuperTask->check_loop( NewBlocker ) ) )
    {
      Result = true;
      if( Object )
	for( Task::List::const_iterator It = Object->dependents().begin(); Result && It != Object->dependents().end(); It++ )
	  Result = !(*It)->check_loop( NewBlocker );
    }
    return Result;
  } // can_add_blocker( Task& )
  void BlockersEditor::blockers_pool_switched( bool On )
  {
    if( On )
      blockers_pool_selection_changed( UI.BlockersPoolTree->selected_index() );
    else
      UI.BlockerAddButton->setEnabled( false );
  } // blockers_pool_switched( bool )
  void BlockersEditor::blockers_pool_selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front();
    blockers_pool_selection_changed( NewIndex );
  } // blockers_pool_selection_changed( const QItemSelection& )
  void BlockersEditor::blockers_pool_selection_changed( const QModelIndex& NewIndex )
  {
    bool CanAdd = false;
    if( Task* NewBlocker = TasksTreeModel::task_from_index( NewIndex ) )
      CanAdd = can_add_blocker( *NewBlocker );
    UI.BlockerAddButton->setEnabled( CanAdd );
  } // blockers_pool_selection_changed( const QModelIndex& )
  void BlockersEditor::blockers_list_selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front();
    blockers_list_selection_changed( NewIndex );
  } // blockers_list_selection_changed( const QItemSelection& )
  void BlockersEditor::blockers_list_selection_changed( const QModelIndex& NewIndex )
  {
    if( NewIndex != selected_blocker() )
      qDebug() << "Wrong list index:" << NewIndex << selected_blocker();
    if( NewIndex.isValid() && NewIndex.internalPointer() )
    {
      UI.BlockerRemoveButton->setEnabled( true );
      UI.BlockerMoveUpButton->setEnabled( NewIndex.row() > 0 );
      UI.BlockerMoveDownButton->setEnabled( Blockers && NewIndex.row() < Blockers->blockers().size()-1 );
    }
    else
    {
      UI.BlockerRemoveButton->setEnabled( false );
      UI.BlockerMoveUpButton->setEnabled( false );
      UI.BlockerMoveDownButton->setEnabled( false );
    }
  } // blockers_list_selection_changed( const QModelIndex& )

  BlockersEditorDialog::EditorWidget::EditorWidget( TasksTreeModel& Model0, Task* Object0, QDialog* Parent )
    : BlockersEditor( Model0, Object0, Parent )
  {
    QDialogButtonBox* Buttons = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    UI.ButtonsLayout->addWidget( Buttons );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() )
    {
      UI.ButtonsLayout->setSpacing( 0 );
      layout()->setSpacing( 0 );
    }
#endif // Q_WS_MAEMO_5
    Parent->connect( Buttons, SIGNAL( accepted() ), SLOT( accept() ) );
    Parent->connect( Buttons, SIGNAL( rejected() ), SLOT( reject() ) );
    UI.BlockersLabel->setText( Object ? BlockersEditorDialog::tr( "Task" ) + ": " + Object->name() : BlockersEditorDialog::tr( "Unknown task" ) );
    UI.BlockersTreeSwitchButton->setChecked( true );
  } // EditorWidget( TasksTreeModel&, Task*, QWidget* )
  bool BlockersEditorDialog::EditorWidget::accept()
  {
    bool Result = true;
    if( Blockers && Object )
      Model.change_blockers( *Object, Blockers->blockers(), SuperTask );
    else
      Result = false;
    return Result;
  } // accept()
  BlockersEditorDialog::BlockersEditorDialog( TasksTreeModel& Model0, Task* Object0, QWidget* Parent )
  : QDialog( Parent ), Editor( Model0, Object0, this )
  {
    setWindowTitle( tr( "Dependencies" ) );
    QVBoxLayout* Layout = new QVBoxLayout( this );
    Layout->addWidget( &Editor );
    setLayout( Layout );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() )
    {
      Qt::WindowFlags Flags = windowFlags() & ~Qt::Dialog | Qt::Window;
      setWindowFlags( Flags );
      setAttribute( Qt::WA_Maemo5StackedWindow );
      Layout->setContentsMargins( 0, 0, 0, 0 );
    }
#endif // Q_WS_MAEMO_5
  } // BlockersEditorDialog( TasksTreeModel&, Task*, QWidget* )
  QSize BlockersEditorDialog::sizeHint() const { return QSize( 800, 480 ); }
  void BlockersEditorDialog::accept()
  {
    if( Editor.accept() )
      QDialog::accept();
  } // accept()

  TaskDialog::TaskDialog( TasksTreeModel& Model0, Task* Object0, QWidget* Parent ) : QDialog( Parent ), Model( Model0 ), Object( Object0 ), SuperTask( 0 ), Blockers( 0 )
  {
    UI.setupUi( this );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() )
    {
      Qt::WindowFlags Flags = windowFlags() & ~Qt::Dialog | Qt::Window;
      setWindowFlags( Flags );
      setAttribute( Qt::WA_Maemo5StackedWindow );
    }
#endif // Q_WS_MAEMO_5
    UI.EstimatedUnits->addItem( tr( "seconds" ), Task::Seconds );
    UI.EstimatedUnits->addItem( tr( "minutes" ), Task::Minutes );
    UI.EstimatedUnits->addItem( tr( "hours" ), Task::Hours );
    UI.EstimatedUnits->addItem( tr( "workdays" ), Task::WorkDays );
    UI.EstimatedUnits->addItem( tr( "days" ), Task::Days );
    UI.EstimatedUnits->addItem( tr( "workweeks" ), Task::WorkWeeks );
    UI.EstimatedUnits->addItem( tr( "weeks" ), Task::Weeks );
    UI.EstimatedUnits->addItem( tr( "months" ), Task::Months );
    UI.EstimatedUnits->addItem( tr( "quarters" ), Task::Quarters );
    UI.EstimatedUnits->addItem( tr( "years" ), Task::Years );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() )
    {
      UI.Description->setMinimumHeight( 128 );
      UI.Comment->setMinimumHeight( 128 );
      QMargins Margins = UI.PlanGroup->layout()->contentsMargins();
      Margins.setTop( 20 );
      UI.PlanGroup->layout()->setContentsMargins( Margins );
      layout()->setContentsMargins( 0, 0, 0, 0 );
    }
#endif // Q_WS_MAEMO_5
    if( Object )
    {
      setWindowTitle( tr( "Task: " ) + Object->name() );
      UI.Name->setText( Object->name() );
      UI.Description->setPlainText( Object->description() );
      UI.Comment->setPlainText( Object->comment() );
      bool HasPlan = Object->estimation() != 0;
      if( Object->plan_start().isValid() )
      {
	HasPlan = true;
	UI.HasStart->setChecked( true );
	UI.PlanStart->setDateTime( Object->plan_start() );
      }
      if( Object->plan_finish().isValid() )
      {
	HasPlan = true;
	UI.HasFinish->setChecked( true );
	UI.PlanFinish->setDateTime( Object->plan_finish() );
      }
      UI.PlanGroup->setChecked( HasPlan );
      UI.Complete->setValue( Object->completed() * 100 );
      UI.Estimated->setValue( Object->estimation() );
      UI.EstimatedUnits->setCurrentIndex( UI.EstimatedUnits->findData( Object->estimation_units() ) );
      if( Object->supertask() ) supertask( Object->supertask() );
    }
    else // No Object
      UI.PlanGroup->setChecked( false );
#ifndef PLANSPLANT_NO_BLOCKERS_SPLITTER
    if( MainWindow::desktop_view() )
    {
#if 0 // def Q_WS_MAEMO_5 // If we disable kinetic scroller all controls inside the scrollarea will not get input.
      disable_kinetic_scroller( UI.scrollArea );
#endif // Q_WS_MAEMO_5
      Blockers = new BlockersEditor( Model, Object, UI.scrollArea );
      if( QBoxLayout* Layout = qobject_cast<QBoxLayout*>( UI.scrollAreaWidgetContents->layout() ) )
	Layout->insertWidget( 3, Blockers );
    }
#endif // PLANSPLANT_NO_BLOCKERS_SPLITTER
    UI.Name->setFocus();
    connect( UI.SelectParent, SIGNAL( clicked( bool ) ), SLOT( select_supertask() ) );
    connect( UI.PlanStartNow, SIGNAL( clicked( bool ) ), SLOT( plan_start_now() ) );
    connect( UI.PlanFinishNow, SIGNAL( clicked( bool ) ), SLOT( plan_finish_now() ) );
    connect( UI.HasParent, SIGNAL( toggled( bool ) ), SLOT( supertask_toggled( bool ) ) );
  } // TaskDialog
  void TaskDialog::accept()
  {
    int CurrUnits = UI.EstimatedUnits->currentIndex();
    if( CurrUnits >= 0 ) CurrUnits = UI.EstimatedUnits->itemData( CurrUnits ).toInt();
    else
    {
      qDebug() << "Wrong estimation time units:" << CurrUnits;
      CurrUnits = 1;
    }
    bool TimesOk = true;
    Time Est = 0;
    QDateTime PlanStart;
    QDateTime PlanFinish;
    if( UI.PlanGroup->isChecked() )
    {
      Est = UI.Estimated->value();
      if( UI.HasStart->isChecked() )
	PlanStart = UI.PlanStart->dateTime();
      if( UI.HasFinish->isChecked() )
      {
	PlanFinish = UI.PlanFinish->dateTime();
	if( UI.HasStart->isChecked() )
	{
	  if( Est > 0 )
	    TimesOk = PlanFinish >= Task::add_time( PlanStart, Est, Task::TimeUnits( CurrUnits ) );
	  else
	    TimesOk = PlanFinish >= PlanStart;
	  if( !TimesOk ) //! \todo Add auto-adjust options.
	    TimesOk = QMessageBox::question( this, tr( "Plans Plant" ), tr( "Finish time is too small.\nContinue?" ),
					     QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes;
	}
      }
    } // Has plan
    if( TimesOk )
    {
      if( !Object )
      {
	Object = new Task( UI.Name->text(), UI.HasParent->isChecked() ? SuperTask : 0 );
	Object->description( UI.Description->toPlainText() );
	Object->comment( UI.Comment->toPlainText() );
	Object->plan_start( PlanStart );
	Object->plan_finish( PlanFinish );
	Object->completed( UI.Complete->value() / 100.0 );
	Object->estimation( Est );
	Object->estimation_units( Task::TimeUnits( CurrUnits ) );
	if( Blockers )
	  foreach( Task* Blocker, Blockers->blockers() )
	    Object->add_dependency( *Blocker );
	Model.add_task( *Object );
      }
      else
      {
	bool Modified = false;
	if( Object->name() != UI.Name->text() )			   { Object->name( UI.Name->text() ); Modified = true; }
	if( Object->description() != UI.Description->toPlainText() ) { Object->description( UI.Description->toPlainText() ); Modified = true; }
	if( Object->comment() != UI.Comment->toPlainText() )	   { Object->comment( UI.Comment->toPlainText() ); Modified = true; }
	if( Object->plan_start() != PlanStart )	   		   { Object->plan_start( PlanStart ); Modified = true; }
	if( Object->plan_finish() != PlanFinish )	   		   { Object->plan_finish( PlanFinish ); Modified = true; }
	double Comp = UI.Complete->value() / 100.0;
	if( Object->completed() != Comp )				   { Object->completed( Comp ); Modified = true; }
	if( Object->estimation() != Est )				   { Object->estimation( Est ); Modified = true; }
	if( Object->estimation_units() != CurrUnits )
	{ Object->estimation_units( Task::TimeUnits( CurrUnits ) ); Modified = true; }
	if( !UI.HasParent->isChecked() ) SuperTask = 0;
	if( Blockers )
	  Model.change_blockers( *Object, Blockers->blockers(), SuperTask );
	if( Object->supertask() != SuperTask )
	  Model.change_parent( Object, SuperTask );
	if( Modified ) Model.modified();
      }
      QDialog::accept();
    }
  } // accept()
  void TaskDialog::supertask( Task* NewSuperTask )
  {
    if( NewSuperTask != SuperTask )
    {
      SuperTask = NewSuperTask;
      if( SuperTask )
      {
	UI.SelectParent->setText( SuperTask->name() );
	UI.HasParent->setChecked( true );
      }
      else
      {
	UI.SelectParent->setText( tr( "none" ) );
	UI.HasParent->setChecked( false );
      }
      supertask_changed();
    }
  } // supertask( Task* )
  void TaskDialog::select_supertask()
  {
    ParentSelectDialog Dlg( Model, this, Blockers ? Blockers->blockers() : ( Object ? Object->needs() : Task::List() ),
			    Object, SuperTask );
    if( Dlg.exec() )
      supertask( Dlg.current_task() );
  } // select_supertask()
  void TaskDialog::supertask_toggled( bool On )
  {
    if( On )
    {
      if( SuperTask )
      {
	if( SuperTask->check_loop( Blockers ? Blockers->blockers() : ( Object ? Object->needs() : Task::List() ), Object ) )
	{
	  QMessageBox::warning( this, tr( "Plans Plant" ),
				tr( "This will create a loop in the tree.\nSelect another task as parent." ) );
	  UI.SelectParent->setText( tr( "none" ) );
	  SuperTask = 0;
	}
	else
	  supertask_changed();
      }
      if( !SuperTask )
      {
	select_supertask();
	if( !SuperTask )
	  UI.HasParent->setChecked( false );
      }
    }
    else
      supertask_changed();
  } // supertask_toggled( bool )
  void TaskDialog::supertask_changed()
  {
    if( Blockers )
      Blockers->supertask( UI.HasParent->isChecked() ? SuperTask : 0 );
  } // supertask_changed()
  void TaskDialog::plan_start_now() { UI.PlanStart->setDateTime( QDateTime::currentDateTime() ); }
  void TaskDialog::plan_finish_now() { UI.PlanFinish->setDateTime( QDateTime::currentDateTime() ); }

  TasksTreeWidget::TasksTreeWidget( QWidget* Parent, TasksTreeModel* Model0 ) : QTreeView( Parent )
  {
    setUniformRowHeights( true );
#ifdef Q_WS_MAEMO_5
    if( MainWindow::desktop_view() )
    {
      disable_kinetic_scroller( this );
      setIconSize( QSize( 14, 14 ) );
    }
#else // Q_WS_MAEMO_5
    setContextMenuPolicy( Qt::ActionsContextMenu );
#endif // Q_WS_MAEMO_5
    TaskAddAction = new QAction( QIcon( ":/images/task-new.svg" ), tr( "&Add task..." ), this );
    TaskAddAction->setShortcut( Qt::CTRL | Qt::Key_T );
    connect( TaskAddAction, SIGNAL( triggered() ), SLOT( add_task() ) );
    addAction( TaskAddAction );
    TaskEditAction = new QAction( QIcon( ":/images/task-edit.svg" ), tr( "Open task..." ), this );
    TaskEditAction->setShortcut( Qt::CTRL | Qt::Key_E );
    connect( TaskEditAction, SIGNAL( triggered() ), SLOT( open_task() ) );
    addAction( TaskEditAction );
    TaskDeleteAction = new QAction( tr( "&Delete task" ), this );
    connect( TaskDeleteAction, SIGNAL( triggered() ), SLOT( delete_task() ) );
    addAction( TaskDeleteAction );
    QAction* Act = new QAction( this );
    Act->setSeparator( true );
    addAction( Act );
    DependencyAddAction = new QAction( QIcon( ":/images/dependency-new.svg" ), tr( "Add de&pendency..." ), this );
    DependencyAddAction->setShortcut( Qt::CTRL | Qt::Key_D );
    connect( DependencyAddAction, SIGNAL( triggered() ), SLOT( add_blocker() ) );
    addAction( DependencyAddAction );
    DependencyRemoveAction = new QAction( tr( "Remo&ve dependency" ), this );
    connect( DependencyRemoveAction, SIGNAL( triggered() ), SLOT( remove_blocker() ) );
    addAction( DependencyRemoveAction );
    Act = new QAction( this );
    Act->setSeparator( true );
    addAction( Act );
    MoveUpAction = new QAction( QIcon::fromTheme( "go-up", QIcon( ":/images/go-up.svg" ) ), tr( "Move &up" ), this );
    MoveUpAction->setShortcut( Qt::CTRL | Qt::Key_Up );
    connect( MoveUpAction, SIGNAL( triggered() ), SLOT( move_up() ) );
    addAction( MoveUpAction );
    MoveDownAction = new QAction( QIcon::fromTheme( "go-down", QIcon( ":/images/go-down.svg" ) ), tr( "Move do&wn" ), this );
    MoveDownAction->setShortcut( Qt::CTRL | Qt::Key_Down );
    connect( MoveDownAction, SIGNAL( triggered() ), SLOT( move_down() ) );
    addAction( MoveDownAction );
    connect( this, SIGNAL( clicked( const QModelIndex& ) ), SLOT( item_clicked( const QModelIndex& ) ) );
    if( Model0 ) tasks( Model0 );
    else selection_changed( QModelIndex() );
  } // TasksTreeWidget( TasksTreeModel&, QWidget* )
  QSize TasksTreeWidget::sizeHint() const { return QSize( 800, 0 ); }
  void TasksTreeWidget::tasks( TasksTreeModel* Tasks )
  {
    setModel( Tasks );
    setColumnWidth( TasksTreeModel::NameCol, 400 );
    if( MainWindow::desktop_view() )
    {
      setColumnWidth( TasksTreeModel::CompletedMarkCol, 24 );
      setColumnWidth( TasksTreeModel::CompletedCol, 43 );
      setAlternatingRowColors( true );
    }
    else
    {
      setColumnWidth( TasksTreeModel::CompletedMarkCol, 55 );
      setColumnWidth( TasksTreeModel::CompletedCol, 70 );
      setColumnWidth( TasksTreeModel::PlanStartCol, 128 );
      setColumnWidth( TasksTreeModel::PlanFinishCol, 128 );
    }
    connect( selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ), SLOT( selection_changed( const QItemSelection& ) ) );
    selection_changed( QModelIndex() );
    if( MainWindow::desktop_view() )
    {
      if( QHeaderView* Header = header() )
	Header->moveSection( 1, 0 );
      else qDebug() << "No header in the tree";
    }
  } // tasks( TasksTreeModel* )
  QModelIndex TasksTreeWidget::selected_index() const
  {
    QModelIndex Result;
    QModelIndexList Selected = selectedIndexes();
    if( !Selected.isEmpty() )
      Result = Selected.front();
    return Result;
  } // selected_index() const
  bool TasksTreeWidget::select_task( const Task* NewCurrent )
  {
    bool Result = false;
    QModelIndex Index;
    if( NewCurrent )
    {
      if( TasksTreeModel* Tasks = tasks() )
	for( TreeIterator It( *Tasks ); It && !Index.isValid(); ++It )
	  if( Tasks->is_subtask( *It ) && TasksTreeModel::task_from_index( *It ) == NewCurrent )
	    Index = *It;
    }
    else
      Result = true;
    setCurrentIndex( Index );
    return Result;
  } // select_task( const Task* )
  void TasksTreeWidget::add_task()
  {
    if( TasksTreeModel* Tasks = tasks() )
    {
      TaskDialog Dlg( *Tasks, 0, this );
      if( Task* Sel = selected_task() )
	Dlg.supertask( Sel );
      Dlg.exec();
    }
  } // add_task()
  void TasksTreeWidget::open_task()
  {
    if( Task* Sel = selected_task() )
      if( TasksTreeModel* Tasks = tasks() )
      {
	TaskDialog Dlg( *Tasks, Sel, this );
	Dlg.exec();
      }
  } // open_task()
  void TasksTreeWidget::delete_task()
  {
    if( Task* Sel = selected_task() )
      if( QMessageBox::question( this, tr( "PlansPlant" ), tr( "Are you shure that you want to delete task\n\"" ) + Sel->name() + tr( "\"?\nThere's NO WAY to restore it." ),
				 QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes )
	if( TasksTreeModel* Tasks = tasks() )
	  Tasks->delete_task( *Sel ); //! \todo else Error message
  } // delete_task()
  void TasksTreeWidget::add_blocker()
  {
    if( Task* Object = selected_task() )
      if( TasksTreeModel* Tasks = tasks() )
      {
#if 0
	BlockerSelectDialog Dlg( *Tasks, this, Object, Object->needs().empty() ? 0 : Object->needs().back() );
	if( Dlg.exec() )
	{
	  if( Dlg.current_task() )
	  {
	    Tasks->add_blocker( *Object, *Dlg.current_task() );
	    qDebug() << "Add blocker" << Dlg.current_task()->name() << "to" << Object->name();
	  }
	  else
	    qDebug() << "No task selected. Don't add anything.";
	}
#else
	BlockersEditorDialog Dlg( *Tasks, Object, this );
	Dlg.exec();
#endif
      }
  } // add_blocker()
  void TasksTreeWidget::remove_blocker()
  {
    QModelIndex Current = selected_index();
    if( Current.isValid() )
    {
      if( TasksTreeModel* Tasks = tasks() )
      {
	if( Tasks->is_blocker( Current ) )
	  Tasks->remove_blocker( Current );
	else
	  qDebug() << "Can't remove blocker: it's not a blocker.";
      }
    }
    else
      qDebug() << "Can't remove blocker: nothing selected.";
  } // remove_blocker()
  void TasksTreeWidget::move_up()
  {
    QModelIndex Index = selected_index();
    if( Index.isValid() )
      if( TasksTreeModel* Tasks = tasks() )
	if( Tasks->move_task( Index.parent(), Index.row(), Index.row()-1 ) )
	  selection_changed( selected_index() ); // The index row was changed, but no signal emitted.
  } // move_up()
  void TasksTreeWidget::move_down()
  {
    QModelIndex Index = selected_index();
    if( Index.isValid() )
      if( TasksTreeModel* Tasks = tasks() )
	if( Tasks->move_task( Index.parent(), Index.row(), Index.row()+1 ) )
	  selection_changed( selected_index() ); // The index row was changed, but no signal emitted.
  } // move_down()
  void TasksTreeWidget::item_clicked( const QModelIndex& Index )
  {
    if( Index.column() == TasksTreeModel::CompletedMarkCol )
      if( Task* Cur = TasksTreeModel::task_from_index( Index ) )
	if( !Cur->blocked() && Cur->completed() < 1
	    && QMessageBox::question( this, tr( "Plans Plant" ),
				      tr( "Complete task\n\"" ) + Cur->name() + tr( "\"?" ),
				      QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes ) == QMessageBox::Yes )
	{
	  Cur->completed( 1 ); //! \todo Do this not directly but through the model.
	  if( TasksTreeModel* Tasks = tasks() )
	    Tasks->modified();
	}
  } // item_clicked( const QModelIndex& )
  void TasksTreeWidget::selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front();
    selection_changed( NewIndex );
  } // selection_changed( const QItemSelection& )
  void TasksTreeWidget::selection_changed( const QModelIndex& NewIndex )
  {
    bool TasksActions = false;
    bool CanMoveUp = false;
    bool CanMoveDown = false;
    if( TasksTreeModel* Tasks = tasks() )
    {
      TaskAddAction->setEnabled( true );
      if( Task* CurTask = Tasks->task_from_index( NewIndex ) )
      {
	TasksActions = true;
	QModelIndex Parent = Tasks->parent( NewIndex );
	if( Tasks->is_blocker( NewIndex ) )
	{
	  DependencyRemoveAction->setEnabled( true );
	  if( Task* Dependent = Tasks->task_from_index( Parent ) )
	  {
	    CanMoveUp = CurTask != Dependent->needs().front();
	    CanMoveDown = CurTask != Dependent->needs().back();
	  }
	}
	else
	{
	  DependencyRemoveAction->setEnabled( false );
	  if( Task* Supertask = Tasks->task_from_index( Parent ) )
	  {
	    CanMoveUp = Supertask->subtasks().front() != CurTask;
	    CanMoveDown = Supertask->subtasks().back() != CurTask;
	  }
	  else
	  {
	    CanMoveUp = Tasks->roots().front() != CurTask;
	    CanMoveDown = Tasks->roots().back() != CurTask;
	  }
	}
      }
    }
    else // No model - maybe issue an error message
      TaskAddAction->setEnabled( true );
    TaskEditAction->setEnabled( TasksActions );
    TaskDeleteAction->setEnabled( TasksActions );
    DependencyAddAction->setEnabled( TasksActions );
    if( !TasksActions ) DependencyRemoveAction->setEnabled( false ); // Don't always enable if we have tasks actions.
    MoveUpAction->setEnabled( CanMoveUp );
    MoveDownAction->setEnabled( CanMoveDown );
  } // selection_changed( const QModelIndex& )

  QString MainWindow::basename( const QString& FileName )
  {
    int DirIndex = FileName.lastIndexOf( '/' );
    if( DirIndex >= 0 ) DirIndex++;
    else DirIndex = 0;
    QString Result = FileName.mid( DirIndex );
    if( Result.right( 11 ) == ".plansplant" )
      Result = Result.left( Result.size()-11 );
    return Result;
  } // basename( const QString& )
  void MainWindow::recent_file( const QString& FileName )
  {
    QSettings().setValue( "Status/RecentFile", FileName );
  } // recent_file( const QString& )
  class MenuHolder
  {
  public:
    MenuHolder( MainWindow* Parent ) : Bar( new QMenuBar( Parent ) ), Menu( 0 ) {}
    void add_menu( const QString& Name )
    {
      Menu = new QMenu( Name );
      Bar->addMenu( Menu );
    } // add_menu( const QString& )
    void add_action( QAction* Act )
    {
      if( Menu ) Menu->addAction( Act );
      else Bar->addAction( Act );
    } // add_action( QAction* )
    void add_separator()
    {
      if( Menu ) Menu->addSeparator();
    } // add_separator()
    operator QMenuBar*() { return Bar; }
  protected:
    QMenuBar* Bar;
    QMenu* Menu;
  }; // MenuHolder
  MainWindow::MainWindow( QWidget* Parent, Qt::WindowFlags Flags ) : QMainWindow( Parent, Flags ), Model( 0 ), Tree( 0 )
  {
    setWindowTitle( tr( "Plans Plant" ) );
    setWindowIcon( QIcon( ":/images/plansplant.svg" ) );
    Tree = new TasksTreeWidget();
    setCentralWidget( Tree );
    QToolBar* Tools = new QToolBar( tr( "Standard" ), this );
#ifdef Q_WS_MAEMO_5
    if( desktop_view() )
      XcursorSetTheme( QX11Info::display(), "DMZ-White" );
    else
      setAttribute( Qt::WA_Maemo5StackedWindow );
    Tree->setContextMenuPolicy( Qt::ActionsContextMenu );
#endif // Q_WS_MAEMO_5
    MenuHolder Menu( this );
    if( desktop_view() ) Menu.add_menu( tr( "&File" ) );
    QAction* Act = new QAction( QIcon::fromTheme( "document-new", QIcon( ":/images/document-new.svg" ) ), tr( "New file" ), this );
    Act->setShortcut( QKeySequence::New );
    connect( Act, SIGNAL( triggered() ), SLOT( new_file() ) );
    Tools->addAction( Act );
    Menu.add_action( Act );
    Act = new QAction( QIcon::fromTheme( "document-open", QIcon( ":/images/document-open.svg" ) ), tr( "Open file..." ), this );
    Act->setShortcut( QKeySequence::Open );
    connect( Act, SIGNAL( triggered() ), SLOT( open_file() ) );
    Tools->addAction( Act );
    Menu.add_action( Act );
    if( desktop_view() ) Menu.add_separator();
    Act = new QAction( QIcon::fromTheme( "document-save", QIcon( ":/images/document-save.svg" ) ), tr( "Save file" ), this );
    Act->setShortcut( QKeySequence::Save );
    connect( Act, SIGNAL( triggered() ), SLOT( save_file() ) );
    Tools->addAction( Act );
    Menu.add_action( Act );
    Act = new QAction( tr( "Save file as..." ), this );
    Act->setShortcut( QKeySequence::SaveAs );
    connect( Act, SIGNAL( triggered() ), SLOT( save_file_as() ) );
    Menu.add_action( Act );
    Tools->addSeparator();
    if( desktop_view() ) Menu.add_separator();
    Act = new QAction( tr( "Export to HTML..." ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( export_to_html() ) );
    Menu.add_action( Act );
    if( desktop_view() )
    {
      Menu.add_separator();
      Act = new QAction( tr( "E&xit" ), this );
      Act->setShortcut( QKeySequence::Quit );
      Act->setMenuRole( QAction::QuitRole );
      connect( Act, SIGNAL( triggered() ), SLOT( close() ) );
      Menu.add_action( Act );
      Menu.add_menu( tr( "&Edit" ) );
    }
    Tools->addAction( Tree->task_add_action() );
    Menu.add_action( Tree->task_add_action() );
    Tools->addAction( Tree->task_edit_action() );
    Menu.add_action( Tree->task_edit_action() );
    if( desktop_view() ) Menu.add_separator();
    Tools->addAction( Tree->dependency_add_action() );
    Menu.add_action( Tree->dependency_add_action() );
    Menu.add_action( Tree->dependency_remove_action() );
    Tools->addSeparator();
    if( desktop_view() ) Menu.add_separator();
    Tools->addAction( Tree->move_up_action() );
    Menu.add_action( Tree->move_up_action() );
    Tools->addAction( Tree->move_down_action() );
    Menu.add_action( Tree->move_down_action() );

    if( desktop_view() )
      Menu.add_menu( tr( "&Help" ) );
    Act = new QAction( tr( "About..." ), this );
    Act->setMenuRole( QAction::AboutRole );
    connect( Act, SIGNAL( triggered() ), SLOT( about() ) );
    Menu.add_action( Act );
    Act = new QAction( tr( "About Qt..." ), this );
    Act->setMenuRole( QAction::AboutQtRole );
    connect( Act, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
    Menu.add_action( Act );
    setMenuBar( Menu );
#ifdef Q_WS_MAEMO_5
    if( desktop_view() )
    {
      if( menuWidget() ) menuWidget()->show();
      // Force toolbar placement
      Tools->setMovable( true );
      addToolBar( Qt::TopToolBarArea, Tools );
    }
    else
      addToolBar( Tools );
#else
    addToolBar( Tools );
#endif // Q_WS_MAEMO_5
  } // MainWindow( QWidget*, Qt::WindowFlags )
  MainWindow::~MainWindow()
  {
    close_file( true );
  } // ~MainWindow()
  QSize MainWindow::sizeHint() const { return QSize( 800, 480 ); }
  bool MainWindow::close_file( bool Force )
  {
    if( Model )
    {
      if( Model->is_modified() )
      {
	QMessageBox::StandardButtons Buttons = QMessageBox::Save | QMessageBox::Discard;
	if( !Force ) Buttons |= QMessageBox::Cancel;
	switch( QMessageBox::question( this, QString(), tr( "The file is modified. Do you want to save it?" ), Buttons ) )
	{
	case QMessageBox::Save:
	  if( !save_file() && !Force ) break;
	case QMessageBox::Discard:
	  Model->deleteLater();
	  Model = 0;
	default:
	  break;
	}
      }
      else
      {
	Model->deleteLater();
	Model = 0;
      }
    }
    return !Model;
  } // close_file( bool )
  bool MainWindow::open_file()
  {
    bool Result = false;
    QFileDialog::Options Opt = 0;
#ifdef PLANSPLANT_USE_QT_FILE_DIALOG
    Opt = QFileDialog::DontUseNativeDialog;
#endif
    QString Filter;
    if( desktop_view() )
      Filter = tr( "Plans Plant files" ) + " (*.plansplant);;" + tr( "All files" ) + " (*)";
    else
      Filter = "Plan (*.plansplant)";
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Open file:" ), Model->file_name(), Filter, 0, Opt );
    if( !NewName.isEmpty() )
      Result = open_file( NewName );
    return Result;
  } // open_file()
  bool MainWindow::open_file( const QString& NewFileName, bool AddToRecent )
  {
    bool Result = false;
    TasksTreeModel* NewModel = new TasksTreeModel( NewFileName );
    if( NewModel->good() && close_file() )
    {
      Model = NewModel;
      Tree->tasks( Model );
      setWindowTitle( tr( "Plans Plant: " ) + basename( NewFileName ) );
      if( AddToRecent ) recent_file( NewFileName );
      Result = true;
    }
    else
    {
      if( !NewModel->good() )
	QMessageBox::warning( this, tr( "Plans Plant" ), tr( "Can't open file\n\"" ) + NewFileName + ( "\"." ) );
      delete NewModel;
    }
    return Result;
  } // open_file( const QString&, bool )
  bool MainWindow::save_file()
  {
    bool Result = false;
    if( Model )
    {
      if( Model->file_name().isEmpty() )
	Result = save_file_as();
      else
	Result = Model->save();
    }
    return Result;
  } // save_file()
  bool MainWindow::save_file_as()
  {
    bool Result = false;
    if( Model )
    {
      QFileDialog::Options Opt = 0;
#ifdef PLANSPLANT_USE_QT_FILE_DIALOG
      Opt = QFileDialog::DontUseNativeDialog;
#endif
      QString Filter;
      QString Extension = "plansplant";
      if( desktop_view() ) 
	Filter = tr( "Plans Plant files" ) + " (*." + Extension + ");;" + tr( "All files" ) + " (*)";
      QString NewName = QFileDialog::getSaveFileName( this, tr( "Select file for saving:" ), Model->file_name(), Filter, 0, Opt );
      if( !NewName.isEmpty() )
      {
	if( !desktop_view() )
	{
	  if( !NewName.endsWith( '.'+Extension ) ) // Force .plansplant extension
	  {
	    if( !NewName.endsWith( '.' ) ) NewName += '.';
	    NewName += Extension;
	  }
	}
	Result = Model->save( NewName );
      }
      if( Result )
      {
	setWindowTitle( tr( "Plans Plant: " ) + basename( NewName ) );
	recent_file( NewName );
      }
    }
    return Result;
  } // save_file_as()
  bool MainWindow::export_to_html()
  {
    bool Result = false;
    if( Model )
    {
      QString ExportName = QSettings().value( "Status/HTMLpath" ).toString();
      QFileDialog::Options Opt = 0;
#ifdef PLANSPLANT_USE_QT_FILE_DIALOG
      Opt = QFileDialog::DontUseNativeDialog;
#endif
#ifdef Q_WS_MAEMO_5
      if( ExportName.isEmpty() )
	ExportName = "/home/user/MyDocs/";
#endif // Q_WS_MAEMO_5
      QString BaseName = basename( Model->file_name() );
      if( BaseName.isEmpty() ) BaseName = "plans";
      ExportName += BaseName + ".html";
      QString Filter;
      if( desktop_view() )
	Filter = tr( "HTML files" ) + " (*.html *.htm);;" + tr( "All files" ) + " (*)";
      ExportName = QFileDialog::getSaveFileName( this, tr( "Export to HTML. Select file:" ), ExportName, Filter, 0, Opt );
      if( !ExportName.isEmpty() )
      {
	HTMLExportFilter Exp( ExportName );
	Result = Exp.export_tasks( Model->roots() );
	if( Result )
	{ //! \todo move save/restore html path to export object
	  int DirIndex = ExportName.lastIndexOf( '/' );
	  if( DirIndex < 0 ) DirIndex = ExportName.lastIndexOf( '\\' );
	  QSettings().setValue( "Status/HTMLpath", DirIndex < 0 ? QString() : ExportName.left( DirIndex+1 ) );
	}
      }
    }
    return Result;
  } // export_to_html()
  bool MainWindow::new_file()
  {
    bool Result = false;
    if( close_file() )
    {
      Model = new TasksTreeModel;
      Tree->tasks( Model );
      Result = true;
      setWindowTitle( tr( "Plans Plant (new document)" ) );
    }
    return Result;
  } // new_file()
  void MainWindow::about()
  {
    QMessageBox::about( this, tr( "About Plans Plant..." ),
			tr( "Plans Plant. Version " ) + QApplication::applicationVersion() +
			trUtf8( "\n© Copyright 2010 Nick Slobodsky (Николай Слободской)\nWeb: http://plansplant.garage.maemo.org"
				"\n\nThis is a simple planning application: tasks tree with planned start/due times."
				"\nSome icons and icons' parts are from the GNOME environment, © 2007–2009 Jakub \'jimmac\' Steiner" ) );
  } // about()
  void MainWindow::closeEvent( QCloseEvent* Close )
  {
    if( close_file() )
      QMainWindow::closeEvent( Close );
    else
      Close->ignore();
  } // closeEvent( QCloseEvent* )
#ifdef Q_WS_MAEMO_5
  bool MainWindow::DesktopView = false;
#else
  bool MainWindow::DesktopView = true;
#endif // Q_WS_MAEMO_5
} // PlansPlant
