// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#ifndef PLANS_PLANT_WIDGETS_HPP
#define PLANS_PLANT_WIDGETS_HPP
#include <QAbstractItemModel>
#include <QTreeView>
#include <QDialog>
#include <QMainWindow>
#include "ui_task.h" //! \todo Move to cpp.
#include "tasks.hpp"

namespace PlansPlant
{
  class TasksTreeModel : public QAbstractItemModel
  { //! \todo Populate incrementally
    Q_OBJECT
    class Carrier
    {
    public:
      typedef Carrier* Pointer;
      typedef QList<Pointer> List;
      enum Relation { Subtask, Blocker, Dependent };
      Carrier( Task& Object0, Pointer Parent0 = 0, Relation Rel0 = Subtask );
      ~Carrier();
      Task& task() const { return Object; }
      Pointer parent() const { return Parent; }
      void parent( Pointer NewParent );
      Relation relation() const { return Rel; }
      void relation( Relation NewRel ) { Rel = NewRel; } // I don't think it will be used
      bool populated() const { return Populated; }
      bool has_subitems() const; // Return true if the item must have subitems, even if the list isn't populated yet
      const List& populate(); // Fill and return SutItems list
      const List& subitems() const { return SubItems; } // Return the list as is. Check if it's populated before.
      Pointer subitem( int Index ) const; // See previous comment
      bool move_subitem( int From, int To );
    protected:
      Task& Object;
      Relation Rel;
      Pointer Parent;
      bool Populated;
      List SubItems;
    }; // Carrier
    static Carrier* carrier_from_index( const QModelIndex& Index );
  public:
    enum ColumnNum { NameCol, CompletedMarkCol, CompletedCol, PlanStartCol, PlanFinishCol, EstimationCol, DescriptionCol, CommentCol, TotalCols = DescriptionCol };
    static Task* task_from_index( const QModelIndex& Index );
    TasksTreeModel( const QString& FileName = QString(), QObject* Parent = 0 );
    ~TasksTreeModel();
    bool good() const { return Good; }
    bool is_modified() const { return Modified; }
    void modified( bool NewModified = true ) { Modified = NewModified; }
    const QString& file_name() const { return FileName; }
    bool save( const QString& FileName = QString() );
    int columnCount( const QModelIndex& Parent = QModelIndex() ) const;
    int rowCount( const QModelIndex& Parent = QModelIndex() ) const;
    QModelIndex index( int Row, int Column, const QModelIndex& Parent = QModelIndex() ) const;
    QModelIndex index( Carrier& ForItem ) const;
    QModelIndex parent( const QModelIndex& Index ) const;
    QVariant data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const;
    QVariant headerData( int Section, Qt::Orientation Orient, int Role = Qt::DisplayRole ) const;
    const Task::List& roots() const { return Roots; }
    bool move_task( const QModelIndex& Index, int From, int To );
    void add_task( Task& NewTask );
    void delete_task( Task& OldTask );
    bool change_parent( Task* ForTask, Task* NewParent );
    bool is_subtask( const QModelIndex& Index ) const;
    void add_blocker( Task& To, Task& Block );
    bool is_blocker( const QModelIndex& Index ) const;
    void remove_blocker( const QModelIndex& Index );
    void remove_blocker( Task& From, Task& Block );
  protected:
    void add_root_task( Task& NewRoot );
    void remove_from_root_tasks( Task& OldRoot );
    Task::List Roots;
    Carrier::List Carriers;
    QString FileName;
    bool Modified;
    bool Good;
  }; // TasksTreeModel;

  class TasksTreeWidget : public QTreeView
  {
    Q_OBJECT
  public:
    TasksTreeWidget( QWidget* Parent = 0, TasksTreeModel* Model0 = 0 );
    QSize sizeHint() const;
    TasksTreeModel* tasks() { return static_cast<TasksTreeModel*>( model() ); }
    void tasks( TasksTreeModel* Tasks );
    Task* selected_task() { return TasksTreeModel::task_from_index( currentIndex() ); }
    bool select_task( const Task* NewCurrent = 0 );
  public slots:
    void add_task();
    void open_task();
    void delete_task();
    void add_blocker();
    void remove_blocker();
    void move_up();
    void move_down();
    void item_clicked( const QModelIndex& Index );
  }; // TasksTreeWidget

  class TaskDialog : public QDialog
  {
    Q_OBJECT
  public:
    TaskDialog( TasksTreeModel& Model0, Task* Object0 = 0, QWidget* Parent = 0 );
    void accept();
    void supertask( Task* NewSuperTask );
  public slots:
    void select_supertask();
    void add_blocker();
    void remove_blocker();
    void plan_start_now();
    void plan_finish_now();
  protected:
    TasksTreeModel& Model;
    Task* Object;
    Task* SuperTask;
    Ui::TaskDialog UI;
  }; // TaskDialog
  class TaskSelectDialog : public QDialog
  {
    Q_OBJECT
  public:
    TaskSelectDialog( TasksTreeModel& Model0, QWidget* Parent = 0, Task* CurrentTask0 = 0, const QString& Title = tr( "Select Task" ) );
    void accept();
    Task* current_task() const { return CurrentTask; }
  protected slots:
    virtual void current_changed( const QModelIndex& New );
  protected:
    Task* CurrentTask;
    TasksTreeWidget* TasksTree;
    QDialogButtonBox* Buttons;
  }; // TaskSelectDialog
  class ParentSelectDialog : public TaskSelectDialog
  {
    Q_OBJECT
  public:
    ParentSelectDialog( TasksTreeModel& Model0, QWidget* Parent = 0, Task* ForTask0 = 0, Task* SuperTask0 = 0 );
  protected slots:
    void current_changed( const QModelIndex& New );
  protected:
    Task* ForTask;
  }; // ParentSelectDialog
  class BlockerSelectDialog : public TaskSelectDialog
  {
    Q_OBJECT
  public:
    BlockerSelectDialog( TasksTreeModel& Model0, QWidget* Parent = 0, Task* ForTask0 = 0, Task* Blocker0 = 0 );
  protected slots:
    void current_changed( const QModelIndex& New );
  protected:
    Task* ForTask;
  }; // BlockerSelectDialog

  class MainWindow : public QMainWindow
  {
    Q_OBJECT
  public:
    MainWindow( QWidget* Parent = 0, Qt::WindowFlags Flags = 0 );
    ~MainWindow();
  public slots:
    bool close_file( bool Force = false );
    bool open_file( const QString& FileName );
    bool open_file();
    bool save_file();
    bool save_file_as();
    bool new_file();
    void about();
  protected:
    void closeEvent( QCloseEvent* Close );
    TasksTreeModel* Model;
    TasksTreeWidget* Tree;
  }; // MainWindow
} // PlansPlant
#endif // PLANS_PLANT_WIDGETS_HPP
