#include "Latitude.h"
#include <QDebug>
#include <QCoreApplication>
#include <qjson/parser.h>


Latitude::Latitude(QObject *parent) :
    QObject(parent)
{
    qDebug() << "Latitude";

    m_location = QtMobility::QGeoPositionInfoSource::createDefaultSource(this);

    setUpdateInterval(oauthSettings.value("update_interval", 5).toInt());
    if (m_location) {
        m_location->setPreferredPositioningMethods(QGeoPositionInfoSource::AllPositioningMethods);
        m_location->startUpdates();

        connect(m_location, SIGNAL(positionUpdated(QGeoPositionInfo)), this, SLOT(setCurrentLocation(QGeoPositionInfo)));
    }

    connect(&oauthManager, SIGNAL(temporaryTokenReceived(QString,QString)), this, SLOT(onTemporaryTokenReceived(QString, QString)));
    connect(&oauthManager, SIGNAL(authorizationReceived(QString,QString)), this, SLOT( onAuthorizationReceived(QString, QString)));
    connect(&oauthManager, SIGNAL(accessTokenReceived(QString,QString)), this, SLOT(onAccessTokenReceived(QString,QString)));
    connect(&oauthManager, SIGNAL(requestReady(QByteArray)), this, SLOT(onRequestReady(QByteArray)));
    connect(&oauthManager, SIGNAL(authorizedRequestDone()), this, SLOT(onAuthorizedRequestDone()));

    if( oauthSettings.value("oauth_token").toString().isEmpty() ||
        oauthSettings.value("oauth_token_secret").toString().isEmpty()) {
        requestToken();
    }

}

void Latitude::requestToken()
{
    qDebug() << "Request Token";

    KQOAuthParameters additionalParameters;
    additionalParameters.insert("scope", "https://www.googleapis.com/auth/latitude");
    additionalParameters.insert("xoauth_displayname", "PGZ Google Latitude");
    //additionalParameters.insert("btmpl", "mobile");

    oauthRequest.clearRequest();
    oauthRequest.initRequest(KQOAuthRequest::TemporaryCredentials, QUrl("https://www.google.com/accounts/OAuthGetRequestToken"));

    oauthRequest.setAdditionalParameters(additionalParameters);
    oauthRequest.setConsumerKey("www.piggz.co.uk");
    oauthRequest.setConsumerSecretKey("L8nsB2zqC9YWREPSHrHTAV4E");
    oauthManager.setHandleUserAuthorization(true);
    oauthManager.executeRequest(&oauthRequest);
}

void Latitude::onTemporaryTokenReceived(QString token, QString tokenSecret)
{
    qDebug() << "Temporary token received: " << token << tokenSecret;

    QUrl userAuthURL("https://www.google.com/latitude/apps/OAuthAuthorizeToken?btmpl=mobile&domain=www.piggz.co.uk&location=current&granularity=best");

    if( oauthManager.lastError() == KQOAuthManager::NoError) {
        qDebug() << "Asking for user's permission to access protected resources. Opening URL: " << userAuthURL;
        oauthManager.getUserAuthorization(userAuthURL);
    }

}

void Latitude::onAuthorizationReceived(QString token, QString verifier) {
    qDebug() << "User authorization received: " << token << verifier;

    oauthManager.getUserAccessTokens(QUrl("https://www.google.com/accounts/OAuthGetAccessToken"));
    if( oauthManager.lastError() != KQOAuthManager::NoError) {
        qDebug() << "Error getting access token.  Quitting";
        QCoreApplication::exit();
    }
}

void Latitude::onAccessTokenReceived(QString token, QString tokenSecret) {
    qDebug() << "Access token received: " << token << tokenSecret;

    oauthSettings.setValue("oauth_token", token);
    oauthSettings.setValue("oauth_token_secret", tokenSecret);

    qDebug() << "Access tokens now stored";
}

void Latitude::onAuthorizedRequestDone() {
    qDebug() << "Authorized request done";

}

void Latitude::onRequestReady(QByteArray response) {
    qDebug() << "Response from the service: " << response;

    QGeoPositionInfo pos = parseLocationJSON(response);

    qDebug() << pos;

    if (pos.coordinate().latitude() != 0 && pos.coordinate().longitude() != 0 && pos.coordinate().altitude() != 0) {
        emit(gotPosition(pos));
    }
}

void Latitude::getCurrentLocation() {
    if( oauthSettings.value("oauth_token").toString().isEmpty() ||
        oauthSettings.value("oauth_token_secret").toString().isEmpty()) {
        qDebug() << "No access tokens. Aborting.";
        return;
    }

    oauthRequest.clearRequest();
    oauthRequest.initRequest(KQOAuthRequest::AuthorizedRequest, QUrl("https://www.googleapis.com/latitude/v1/currentLocation"));
    oauthRequest.setConsumerKey("www.piggz.co.uk");
    oauthRequest.setConsumerSecretKey("L8nsB2zqC9YWREPSHrHTAV4E");
    oauthRequest.setToken(oauthSettings.value("oauth_token").toString());
    oauthRequest.setTokenSecret(oauthSettings.value("oauth_token_secret").toString());

    KQOAuthParameters params;
    params.insert("key", "AIzaSyDfV85XemcX3Pj36iK_HqzlqAlfT7kDRXY");
    oauthRequest.setAdditionalParameters(params);

    oauthRequest.setHttpMethod(KQOAuthRequest::GET);
    oauthManager.executeRequest(&oauthRequest);
}

void Latitude::setCurrentLocation()
{
    qDebug() << "Set Location";

    if (m_location) {
        setCurrentLocation(m_location->lastKnownPosition());
    } else {
        QtMobility::QGeoPositionInfo pos;
        QtMobility::QGeoCoordinate coord;
        coord.setLongitude(10);
        coord.setLatitude(10);
        coord.setAltitude(10);
        pos.setCoordinate(coord);
        pos.setAttribute(QtMobility::QGeoPositionInfo::HorizontalAccuracy, 1);
        setCurrentLocation(pos);
    }
}

void Latitude::setCurrentLocation(const QGeoPositionInfo &pos) {
    qDebug() << "Set Location";

    if( oauthSettings.value("oauth_token").toString().isEmpty() ||
        oauthSettings.value("oauth_token_secret").toString().isEmpty()) {
        qDebug() << "No access tokens. Aborting.";
        return;
    }
    oauthRequest.clearRequest();
    oauthRequest.initRequest(KQOAuthRequest::AuthorizedRequest, QUrl("https://www.googleapis.com/latitude/v1/currentLocation"));
    oauthRequest.setConsumerKey("www.piggz.co.uk");
    oauthRequest.setConsumerSecretKey("L8nsB2zqC9YWREPSHrHTAV4E");
    oauthRequest.setToken(oauthSettings.value("oauth_token").toString());
    oauthRequest.setTokenSecret(oauthSettings.value("oauth_token_secret").toString());

    //KQOAuthParameters params;
    //params.insert("key", "AIzaSyDfV85XemcX3Pj36iK_HqzlqAlfT7kDRXY");
    //oauthRequest.setAdditionalParameters(params);

    oauthRequest.setContentType("application/json");
    oauthRequest.setRawData(generateLocationJSON(pos));
    oauthRequest.setHttpMethod(KQOAuthRequest::POST);
    oauthManager.executeRequest(&oauthRequest);
}

QByteArray Latitude::generateLocationJSON(const QGeoPositionInfo &pos)
{
    QByteArray data;
    data = "{\n";
    data += "\"data\": {\n";
    data += "\"kind\":\"latitude#location\",\n";
    data += QString("\"latitude\":%1,\n").arg(pos.coordinate().latitude());
    data += QString("\"longitude\":%1,\n").arg(pos.coordinate().longitude());
    data += QString("\"accuracy\":%1,\n").arg(pos.attribute(QtMobility::QGeoPositionInfo::HorizontalAccuracy));
    data += QString("\"altitude\":%1\n").arg(pos.coordinate().altitude());
    data += "}\n";
    data += "}";

    qDebug() << data;
    return data;
}

QtMobility::QGeoPositionInfo Latitude::parseLocationJSON(const QByteArray &data)
{
    QJson::Parser parser;
    bool ok;
    QVariantMap result;
    QGeoPositionInfo pos;
    QGeoCoordinate coord;

    result = parser.parse(data, &ok).toMap();

    QVariantMap coorddata = result["data"].toMap();

    if (!ok){
        coord.setLongitude(0);
        coord.setLatitude(0);
        coord.setAltitude(0);
    } else {
        coord.setLongitude(coorddata["longitude"].toDouble());
        coord.setLatitude(coorddata["latitude"].toDouble());
        coord.setAltitude(coorddata["altitude"].toDouble());
    }


    pos.setCoordinate(coord);
    pos.setAttribute(QtMobility::QGeoPositionInfo::HorizontalAccuracy, 1);

    return pos;
}

void Latitude::setUpdateInterval(int minutes)
{
    oauthSettings.setValue("update_interval", minutes);
    if (m_location) {
        m_location->setUpdateInterval(minutes * 60 * 1000);
    }
}
