#include "trailgpxrecorder.h"
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QStackedWidget>
#include <QLabel>
#include <QPushButton>
#include <QTableView>
#include <QHeaderView>
#include <QtLocation/QGeoPositionInfoSource>
#include <QtLocation/QGeoPositionInfo>
#include <QDebug>
#include <QFile>
#include <QTextStream>
#include <QResizeEvent>
#include <QFileDialog>
#include "trailinterface.h"
#include <QGraphicsView>
#include <QGraphicsScene>
#include <QGraphicsLinearLayout>
#include <QtLocation/QGeoBoundingBox>
#include "MapWidget.h"

TrailGPXRecorder::TrailGPXRecorder(TrailInterface *interface, QWidget *parent) :
    QWidget(parent), m_model(0), m_interface(interface)
{
    //Layout all the widgets
    m_layout = new QVBoxLayout(this);

    m_curPos = new QLabel("<b>Current Position:</b> unavailable", this);
    m_layout->addWidget(m_curPos);
    
    m_curLongitude = new QLabel("<b>Longitude: </b>unavailable", this);
    m_curLongitude->setFocusPolicy(Qt::TabFocus);
    m_layout->addWidget(m_curLongitude);
    
    m_curLatitude = new QLabel("<b>Latitude: </b>unavailable", this);
    m_curLatitude->setFocusPolicy(Qt::TabFocus);
    m_layout->addWidget(m_curLatitude);

    m_curAccuracy = new QLabel("<b>Accuracy: </b>unavailable", this);
    m_curAccuracy->setFocusPolicy(Qt::TabFocus);
    m_layout->addWidget(m_curAccuracy);

    m_btnLayout = new QHBoxLayout();
    
    m_record = new QPushButton(QIcon(":/images/media-record.png"), QString(), this);
    m_stop = new QPushButton(QIcon(":/images/media-playback-stop.png"), QString(), this);
    m_save = new QPushButton(QIcon(":/images/document-save.png"),QString(), this);
    m_switch = new QPushButton(QIcon(":/images/view-refresh.png"),QString(), this);

    m_btnLayout->addWidget(m_record);
    m_btnLayout->addWidget(m_stop);
    m_btnLayout->addWidget(m_save);
    m_btnLayout->addWidget(m_switch);
    m_layout->addLayout(m_btnLayout);
    
    m_mapWidget = new MapWidget(interface, this);

    m_gpxList = new QTableView(this);
    m_gpxList->horizontalHeader()->setVisible(true);
    m_gpxList->horizontalHeader()->setResizeMode(QHeaderView::Stretch);

    m_mapStack = new QStackedWidget(this);
    m_mapStack->addWidget(m_mapWidget);
    m_mapStack->addWidget(m_gpxList);

    m_layout->addWidget(m_mapStack);
    
    connect(m_save, SIGNAL(clicked(bool)), this, SLOT(slotSave()));
    connect(m_record, SIGNAL(clicked(bool)), this, SLOT(slotRecord()));
    connect(m_stop, SIGNAL(clicked(bool)), this, SLOT(slotStop()));
    connect(m_switch, SIGNAL(clicked(bool)), this, SLOT(slotSwitchView()));

    //Create the location source
    m_location = m_interface->location();
    
    if (m_location) {
        m_model = new TrailTrackModel(this);
        
        m_gpxList->setModel(m_model);

        connect(m_location, SIGNAL(positionUpdated(const QGeoPositionInfo&)), m_model, SLOT(positionUpdated(const QGeoPositionInfo&)));
        connect(m_location, SIGNAL(positionUpdated(const QGeoPositionInfo&)), this, SLOT(positionUpdated(const QGeoPositionInfo&)));
    }

    m_currentCircle = new QGeoMapCircleObject();
    m_accuracyCircle = new QGeoMapCircleObject();

    m_poly = new QGeoMapPolylineObject();
    m_poly->setPen(QPen(Qt::red));

    QList<QGeoCoordinate> invalid;
    invalid << QGeoCoordinate() << QGeoCoordinate();
    m_poly->setPath(invalid);

    m_currentCircle->setCenter(QGeoCoordinate(54.612293,-3.227448));
    m_currentCircle->setRadius(5);
    m_currentCircle->setPen(QPen(Qt::red, 3));
    m_currentCircle->setBrush(QBrush(Qt::red));

    m_accuracyCircle->setCenter(QGeoCoordinate(54.612293,-3.227448));
    m_accuracyCircle->setRadius(30);
    m_accuracyCircle->setPen(QPen(Qt::red, 1));
    m_accuracyCircle->setBrush(QBrush(QColor(128, 0, 0, 64)));

    m_mapWidget->addMapObject(m_currentCircle);
    m_mapWidget->addMapObject(m_accuracyCircle);

    m_mapWidget->addMapObject(m_poly);
    m_mapWidget->setCenter(m_currentCircle->center());
}

TrailGPXRecorder::~TrailGPXRecorder()
{

}

void TrailGPXRecorder::slotRecord()
{
    if (m_model) {
        m_model->clear();
        m_model->setRecord(true);
    }
}

void TrailGPXRecorder::slotSave()
{
    if (!m_model) {
        return;
    }
    qDebug() << "Saving";

    QString filename = QFileDialog::getSaveFileName(this, tr("Save File"), QString(), QString("*.gpx"));

    if (!filename.isEmpty()) {
        if (!filename.endsWith(".gpx", Qt::CaseInsensitive)) {
            filename += ".gpx";
        }

        QFile data(filename);
        if (data.open(QFile::WriteOnly | QFile::Truncate)) {
            QTextStream t(&data);
            t << m_model->toText();
            data.close();
        }
    }
}

void TrailGPXRecorder::slotStop()
{
    if (m_model) {
        m_model->setRecord(false);
    }
}

void TrailGPXRecorder::slotSwitchView()
{
    if (m_mapStack->currentWidget() == m_mapWidget) {
        m_mapStack->setCurrentWidget(m_gpxList);
    } else {
        m_mapStack->setCurrentWidget(m_mapWidget);
    }
}

void TrailGPXRecorder::positionUpdated(const QGeoPositionInfo& pos)
{
    m_curLatitude->setText(QString("<b>Latitude: </b>%1").arg(pos.coordinate().latitude()));
    m_curLongitude->setText(QString("<b>Longitude: </b>%1").arg(pos.coordinate().longitude()));
    m_curAccuracy->setText(QString("<b>Accuracy: </b>%1").arg(pos.attribute(QGeoPositionInfo::HorizontalAccuracy)));

    if (width() > height()) {
        //Landscape
        m_curPos->setText(QString("<b>Current Position: </b>%1/%2/%3").arg(m_model->currentPosition().coordinate().latitude()).arg(m_model->currentPosition().coordinate().longitude()).arg(m_model->currentPosition().attribute(QGeoPositionInfo::HorizontalAccuracy)));
    }
    else {
        //Portrait
        m_curPos->setText(QString("<b>Current Position: </b>"));
    }

    m_currentCircle->setCenter(pos.coordinate());
    m_accuracyCircle->setCenter(pos.coordinate());
    m_accuracyCircle->setRadius(pos.attribute(QGeoPositionInfo::HorizontalAccuracy));

    m_path.clear();

    for (int i = 0; i < m_model->rowCount(); ++i) {
        m_path << QGeoCoordinate(m_model->data(m_model->index(i, 2)).toDouble(), m_model->data(m_model->index(i, 2)).toDouble());
    }

    qDebug()<< m_path;

    m_poly->setPath(m_path);
    m_mapWidget->setCenter(m_currentCircle->center());
}

void TrailGPXRecorder::resizeEvent(QResizeEvent *event)
{
    if (event->size().width() > event->size().height()) {
        //Landscape
        m_curLongitude->hide();
        m_curLatitude->hide();
        m_curAccuracy->hide();

        if (m_model) {
            m_curPos->setText(QString("<b>Current Position: </b>%1/%2/%3").arg(m_model->currentPosition().coordinate().latitude()).arg(m_model->currentPosition().coordinate().longitude()).arg(m_model->currentPosition().attribute(QGeoPositionInfo::HorizontalAccuracy)));
        }
    } else {
        //Portrait
        m_curLongitude->show();
        m_curLatitude->show();
        m_curAccuracy->show();
        m_curPos->setText("<b>Current Position:</b>");
    }
}

void TrailGPXRecorder::clearMapObjects()
{
    m_mapWidget->clearMapObjects();
}
