#include "fingerbuttondialog.h"
#include "ui_fingerbuttondialog.h"
#include <QDebug>

/*!
    \class FingerButtonDialog

    \brief Simple "menu" with pushbuttons, for finger usage.

    Standard Qt menu provides a rather unfriendly finger
    experience, even when using the "hildonized" Qt on
    Diablo. This dialog type provides a convenient workaround
    for the problem, by just creating a dialog with a number
    of QPushButtons in a grid.

    Typical use:

        FingerButtonDialog* dlg = new FingerButtonDialog();
        dlg->setWindowTitle(tr("What next?"));
        QStringList buttons;
        buttons << tr("Browser") << tr("Tear") << tr("Stash")

        // Dim "Browser" button
        QPushButton* browserBtn = qobject_cast<QPushButton*>(dlg->widgetAt(0));
        browserBtn->setEnabled(false);

        int ret = dlg->exec();
        int sel = dlg->selectedIndex();
        delete dlg;
        if (ret == QDialog::Rejected) {
            return;
        }
        switch (sel)
        {
          case 0: ...

          case 1: ...
        }


*/

FingerButtonDialog::FingerButtonDialog(QWidget *parent) :
    QDialog(parent),
    m_ui(new Ui::FingerButtonDialog)
{
    m_ui->setupUi(this);
    m_signalMapper = new QSignalMapper(this);

    connect(m_ui->cancelButton, SIGNAL(clicked()), this, SLOT(reject()));
    m_selection = -1;
}

FingerButtonDialog::~FingerButtonDialog()
{
    delete m_ui;    
}

void FingerButtonDialog::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        m_ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

/*!
    Set captions for the available buttons as \a texts.
    The order of the buttons is important - selectedIndex()
    returns the index of the pressed button within this
    list.

    Cancel button will always be available.

    \sa selectedIndex()
*/

void FingerButtonDialog::setButtons(const QStringList& texts)
{
     QGridLayout *gridLayout = m_ui->gridLayout;

     for (int i = 0; i < texts.size(); ++i) {
         QPushButton *button = new QPushButton(texts[i], this);
         m_widgets.append(button);
         connect(button, SIGNAL(clicked()), m_signalMapper, SLOT(map()));
         m_signalMapper->setMapping(button, i);
         gridLayout->addWidget(button, i / 2, i % 2);
     }

     connect(m_signalMapper, SIGNAL(mapped(int)),
             this, SLOT(clicked(int)));
}

void FingerButtonDialog::clicked(int buttonindex)
{
    //qDebug("%i",buttonindex);
    m_selection = buttonindex;
    accept();
}

/*!
    Return the index of the button that was pressed.
    Return -1 if no selection was made (user pressed Cancel).

    \sa selectedIndex()
*/


int FingerButtonDialog::selectedIndex() const
{
    return m_selection;
}

/*!
    Returns the widget at index \a i.

    This can be used to dim, or otherwise "postprocess" the
    buttons after calling setButtons().
*/


QWidget* FingerButtonDialog::widgetAt(int i)
{
    return m_widgets[i];
}
