/*
 * Copyright (c) 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */
 
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include <string.h>


#include "nm_sbrsh_setup.h"
#include "nm_settings.h"
#include "nm_environment_manager.h"
#include "nm_file_dialog.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

typedef struct _NmSbrshSetupPrivate NmSbrshSetupPrivate;

struct _NmSbrshSetupPrivate
{
    GtkWidget *checkbutton;

    GtkWidget *port_label;
    GtkWidget *nosandbox_label;
    GtkWidget *allowroot_label;
    GtkWidget *debug_label;
    GtkWidget *allowed_ips_label;

    GtkWidget *port_entry;
    GtkWidget *nosandbox_checkbox;
    GtkWidget *allowroot_checkbox;
    GtkWidget *debug_checkbox;
    GtkWidget *allowed_ips_button;
    
    gchar *env;
};

#define NM_SBRSH_SETUP_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_SBRSH_SETUP_TYPE, NmSbrshSetupPrivate))

static void nm_sbrsh_setup_class_init (NmSbrshSetupClass *);
static void nm_sbrsh_setup_init       (NmSbrshSetup *);
static void nm_sbrsh_setup_dispose    (GObject *);
static void nm_sbrsh_setup_finalize   (GObject *);

static void nm_sbrsh_setup_defaults (NmSbrshSetup *, gboolean);
static void nm_sbrsh_setup_create_widget (NmSbrshSetup *);
static void nm_sbrsh_setup_enable_cb (GtkToggleButton *, gpointer);
static void nm_sbrsh_setup_enable_options (NmSbrshSetup *, gboolean);
static void nm_sbrsh_setup_manage_allowed_ips(gpointer , gpointer);
    
G_DEFINE_TYPE (NmSbrshSetup, nm_sbrsh_setup, NM_CONN_WIDGET_TYPE)

static void
nm_sbrsh_setup_class_init (NmSbrshSetupClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    NmConnWidgetClass *conn_widget_class = NM_CONN_WIDGET_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmSbrshSetupPrivate));

    object_class->dispose = nm_sbrsh_setup_dispose;
    object_class->finalize = nm_sbrsh_setup_finalize;

    conn_widget_class->apply_changes = nm_sbrsh_setup_apply_changes;
}

static void
nm_sbrsh_setup_init (NmSbrshSetup *self)
{
    nm_sbrsh_setup_create_widget (self);
}

static void
nm_sbrsh_setup_dispose (GObject *object)
{
    G_OBJECT_CLASS (nm_sbrsh_setup_parent_class)->dispose (object);
}

static void
nm_sbrsh_setup_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_sbrsh_setup_parent_class)->finalize (object);
}

GtkWidget*
nm_sbrsh_setup_new (gchar *env)
{
    GtkWidget *self = g_object_new (NM_SBRSH_SETUP_TYPE, NULL);
    
    NmSbrshSetupPrivate *priv;

    priv = NM_SBRSH_SETUP_GET_PRIVATE (self);
    
    priv->env = g_strdup(env);
    
    g_debug("[%s] - priv->env: %s", __FUNCTION__, priv->env);
    
    return self;
}

void
nm_sbrsh_setup_enable (NmSbrshSetup *self, gboolean enable)
{
    NmSbrshSetupPrivate *priv;

    priv = NM_SBRSH_SETUP_GET_PRIVATE (self);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(priv->checkbutton), enable);
}

gboolean
nm_sbrsh_setup_is_enabled (NmSbrshSetup *self)
{
    NmSbrshSetupPrivate *priv;

    priv = NM_SBRSH_SETUP_GET_PRIVATE (self);
    return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(priv->checkbutton));
}

static void
nm_sbrsh_setup_create_widget (NmSbrshSetup *self)
{
    GtkWidget *table;
    NmSbrshSetupPrivate *priv;

    priv = NM_SBRSH_SETUP_GET_PRIVATE (self);

    table = gtk_table_new (6, 2, TRUE);

    priv->checkbutton = gtk_check_button_new_with_label (_("enable"));
    g_signal_connect (priv->checkbutton, "toggled", G_CALLBACK(nm_sbrsh_setup_enable_cb),
                      self);

    /* Allowed ips */
    priv->allowed_ips_label = gtk_label_new(_("Allowed IPs: "));
    gtk_misc_set_alignment(GTK_MISC(priv->allowed_ips_label), 1, 0.5);
    priv->allowed_ips_button = gtk_button_new_with_label ("...");
    g_signal_connect (G_OBJECT (priv->allowed_ips_button),
                       "clicked",
                       G_CALLBACK (nm_sbrsh_setup_manage_allowed_ips), self);


    /* Port number */
    priv->port_label = gtk_label_new(_("Port Number: "));
    gtk_misc_set_alignment(GTK_MISC(priv->port_label), 1, 0.5);
    priv->port_entry = gtk_entry_new();

    /* No sandbox */
    priv->nosandbox_label = gtk_label_new(_("No Sandbox: "));
    gtk_misc_set_alignment(GTK_MISC(priv->nosandbox_label), 1, 0.5);
    priv->nosandbox_checkbox = gtk_check_button_new ();

    /* Allow root */
    priv->allowroot_label = gtk_label_new(_("Allow Root: "));
    gtk_misc_set_alignment(GTK_MISC(priv->allowroot_label), 1, 0.5);
    priv->allowroot_checkbox = gtk_check_button_new ();

    /* Debug */
    priv->debug_label = gtk_label_new(_("Debug: "));
    gtk_misc_set_alignment(GTK_MISC(priv->debug_label), 1, 0.5);
    priv->debug_checkbox = gtk_check_button_new ();

    gtk_table_attach_defaults(GTK_TABLE (table), priv->checkbutton, 0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->allowed_ips_label, 0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->allowed_ips_button, 1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->port_label, 0, 1, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->port_entry, 1, 2, 2, 3);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->nosandbox_label, 0, 1, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->nosandbox_checkbox, 1, 2, 3, 4);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->allowroot_label, 0, 1, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE (table), priv->allowroot_checkbox, 1, 2, 4, 5);
    gtk_table_attach_defaults(GTK_TABLE( table), priv->debug_label, 0, 1, 5, 6);
    gtk_table_attach_defaults(GTK_TABLE( table), priv->debug_checkbox, 1, 2, 5, 6);
    
    GtkWidget *scroll;
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           table);    
    
    gtk_container_add(GTK_CONTAINER (self), scroll);

    nm_sbrsh_setup_enable_options(self, FALSE);
}

static void
nm_sbrsh_setup_defaults (NmSbrshSetup *self, gboolean pressed)
{
    NmSbrshSetupPrivate *priv;

    // FIXME: This is not a good solution. It should be done during class init
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);

    priv = NM_SBRSH_SETUP_GET_PRIVATE (self);

    if (pressed) {

        gtk_entry_set_text (GTK_ENTRY (priv->port_entry),
                             nm_settings_get_value (settings, priv->env,
                                                     NM_SBRSH_SETTINGS_GROUP,
                                                     NM_SBRSH_PORT));

                                                     
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->nosandbox_checkbox),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_SBRSH_SETTINGS_GROUP,
                                                     NM_SETTINGS_NOSANDBOX));

        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->allowroot_checkbox),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_SBRSH_SETTINGS_GROUP,
                                                     NM_SETTINGS_ALLOWROOT));
                                                     
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->debug_checkbox),
                             nm_settings_get_boolean (settings, priv->env,
                                                     NM_SBRSH_SETTINGS_GROUP,
                                                     NM_SETTINGS_DEBUG));
    } else {
        gtk_entry_set_text(GTK_ENTRY (priv->port_entry), "");
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON (priv->nosandbox_checkbox), FALSE );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON (priv->allowroot_checkbox), FALSE );
        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON (priv->debug_checkbox), FALSE );
    }
}

static void
nm_sbrsh_setup_enable_cb (GtkToggleButton *button, gpointer data)
{
    NmSbrshSetupPrivate *priv;
    gboolean pressed = gtk_toggle_button_get_active (button);

    priv = NM_SBRSH_SETUP_GET_PRIVATE (data);
    
    nm_sbrsh_setup_defaults (data, pressed);

    // FIXME: remove this if the combo box is needed for something else
    nm_sbrsh_setup_enable_options (data, pressed);
    nm_conn_widget_settings_changed (data);
}

static void
nm_sbrsh_setup_enable_options (NmSbrshSetup *self, gboolean enable)
{
    NmSbrshSetupPrivate *priv;
    priv = NM_SBRSH_SETUP_GET_PRIVATE (self);

    gtk_widget_set_sensitive (priv->allowed_ips_label, enable);
    gtk_widget_set_sensitive (priv->allowed_ips_button, enable);
    gtk_widget_set_sensitive (priv->port_label, enable);
    gtk_widget_set_sensitive (priv->port_entry, enable);
    gtk_widget_set_sensitive (priv->nosandbox_label, enable);
    gtk_widget_set_sensitive (priv->nosandbox_checkbox, enable);
    gtk_widget_set_sensitive (priv->allowroot_label, enable);
    gtk_widget_set_sensitive (priv->allowroot_checkbox, enable);
    gtk_widget_set_sensitive (priv->debug_label, enable);
    gtk_widget_set_sensitive (priv->debug_checkbox, enable);
}

gboolean
nm_sbrsh_setup_apply_changes (NmConnWidget *self)
{
    g_debug("[%s]", __FUNCTION__);
    NmSbrshSetup *sbrshsetup = NM_SBRSH_SETUP(self);
    
    NmSbrshSetupPrivate *priv;
    priv = NM_SBRSH_SETUP_GET_PRIVATE (sbrshsetup);
    
    nm_sbrsh_setup_save_changes(sbrshsetup);
    
    if ( nm_sbrsh_setup_is_enabled(sbrshsetup) ) {

        /* Update the exports */
        GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_SBRSH_SETTINGS_GROUP, NM_FILE_DIALOG_TEXT_DIALOG);
        gsize length = 0;
        gchar **list = nm_file_dialog_get_list(NM_FILE_DIALOG(dialog), &length);
        
        gchar *concat_list = "";
        int i;
        for (i = 0; i < length; i++) {
            concat_list = g_strconcat(concat_list, " ", list[i], NULL);
        }
        
        g_debug("[%s] - concat_list: %s", __FUNCTION__, concat_list);
        gchar command[50 + strlen(concat_list)];
        
        sprintf(command, "sudo /usr/sbin/nm-sbrsh-setup allowed_ips %s", concat_list);
        system(command);
        
        gtk_widget_destroy(dialog);
        g_free(concat_list);
        /* END updating the exports */
        
        const gchar *port = gtk_entry_get_text(GTK_ENTRY(priv->port_entry));
        int nosandbox = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->nosandbox_checkbox));
        int allowroot = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->allowroot_checkbox));
        int debug = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(priv->debug_checkbox));
        
        sprintf(command, "sudo /usr/sbin/nm-sbrsh-setup config %d %d %d %s", nosandbox, allowroot, debug, port);
        
        g_debug("[%s] - command: %s ", __FUNCTION__, command);
        
        /* Update the config file*/
        system(command);
        
        /* Restart the deamon */ 
        system("sudo /usr/sbin/nm-sbrsh-setup stop");
        system("sudo /usr/sbin/nm-sbrsh-setup start");

        /* Export with avahi */
        sprintf(command, "sudo /usr/sbin/nm-avahi start_sbrsh %s", port);
        system(command);
        
    }
    else {
        /* Set the allowed ips to localhost only for security reasons */
        system("sudo /usr/sbin/nm-sbrsh-setup allowed_ips 127.0.0.1");
        
        system("sudo /usr/sbin/nm-sbrsh-setup stop");
        system("sudo /usr/sbin/nm-avahi stop_sbrsh");
    }
    
    
    return TRUE;   
}

gboolean 
nm_sbrsh_setup_activate_environment (gchar *env)
{
    g_debug("[%s]", __FUNCTION__);
    gboolean ret = TRUE;
    
    NmSettings *settings = nm_settings_new ();
    nm_settings_start (settings);
    
    gboolean enabled = nm_settings_get_boolean (settings, env,
                                                     NM_SBRSH_SETTINGS_GROUP,
                                                     NM_SETTINGS_ENABLE);

    GtkWidget *sbrsh = nm_sbrsh_setup_new(env);
    nm_sbrsh_setup_enable(NM_SBRSH_SETUP(sbrsh), enabled);
    ret = nm_sbrsh_setup_apply_changes(NM_CONN_WIDGET(sbrsh));

    return ret;
}

void 
nm_sbrsh_setup_save_changes (NmSbrshSetup *self) 
{
    NmSbrshSetupPrivate *priv = NM_SBRSH_SETUP_GET_PRIVATE (self);

    NmSettings *settings = nm_settings_new();
  
    nm_settings_set_boolean (settings, priv->env, NM_SBRSH_SETTINGS_GROUP,
                        NM_SETTINGS_ENABLE,  nm_sbrsh_setup_is_enabled(self));  
 
    if (nm_sbrsh_setup_is_enabled(self)) {
        nm_settings_set_value (settings, priv->env, NM_SBRSH_SETTINGS_GROUP,
                            NM_SBRSH_PORT,
                            gtk_entry_get_text (GTK_ENTRY (priv->port_entry)));
                            
        nm_settings_set_boolean (settings, priv->env, NM_SBRSH_SETTINGS_GROUP,
                            NM_SETTINGS_NOSANDBOX,
                            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->nosandbox_checkbox)));

        nm_settings_set_boolean (settings, priv->env, NM_SBRSH_SETTINGS_GROUP,
                            NM_SETTINGS_ALLOWROOT,
                            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->allowroot_checkbox)));

        nm_settings_set_boolean (settings, priv->env, NM_SBRSH_SETTINGS_GROUP,
                            NM_SETTINGS_DEBUG,
                            gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->debug_checkbox)));        
    }

    nm_settings_save(settings);
}

gboolean
sbrsh_setup_check_enabled (NmSbrshSetup *self)
{
    NmSbrshSetupPrivate *priv = NM_SBRSH_SETUP_GET_PRIVATE (self);

    if ( system("sudo /usr/sbin/nm-sbrsh-setup isrunning") ) {

        nm_sbrsh_setup_enable (self, TRUE);
        nm_sbrsh_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return TRUE;
    } else {

        nm_sbrsh_setup_enable (self, FALSE);
        nm_sbrsh_setup_enable_cb (GTK_TOGGLE_BUTTON (priv->checkbutton), self);
        return FALSE;
    }
/*    return TRUE;*/
}

void
nm_sbrsh_setup_fill_widget(NmSbrshSetup *self)
{
    NmSbrshSetupPrivate *priv = NM_SBRSH_SETUP_GET_PRIVATE (self);
    
    if (strcmp(priv->env, nm_environment_manager_get_active())) {
/*        g_debug("[%s] - %s is not the active environment", __FUNCTION__, priv->env);*/
        NmSettings *settings = nm_settings_new ();
        nm_settings_start (settings);
/*        g_debug("[%s] - sbrsh enabled: %d", __FUNCTION__, */
/*                    nm_settings_get_boolean (settings, priv->env, NM_USB_SETTINGS_GROUP, NM_SETTINGS_ENABLE));*/
        
        nm_sbrsh_setup_enable(self, nm_settings_get_boolean (settings, priv->env, NM_SBRSH_SETTINGS_GROUP, NM_SETTINGS_ENABLE));
        
    }
    else {
/*        g_debug("[%s] - %s is the active environment", __FUNCTION__, priv->env);*/
        sbrsh_setup_check_enabled (self);
    }
}

static void
nm_sbrsh_setup_manage_allowed_ips(gpointer instance, gpointer self)
{
    
    NmSbrshSetupPrivate *priv = NM_SBRSH_SETUP_GET_PRIVATE (self);
    
    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);
    
    GtkWidget *dialog = nm_file_dialog_new(priv->env, NM_SBRSH_SETTINGS_GROUP, NM_FILE_DIALOG_TEXT_DIALOG);
   
    nm_file_dialog_show_dialog(NM_FILE_DIALOG(dialog));

    gtk_widget_destroy(dialog);

}

